"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Arbitrary (valid) array of intervals
 *
 * There are many invalid combinations of interval arrays, so we have
 * to be very specific about generating arrays that are valid. We do this
 * by taking a full, valid interval schedule and progressively stripping parts
 * away from it.
 *
 * Some of the changes may change its meaning, but we take care to never leave
 * a schedule with insufficient information so that the parser will error out.
 */
function generateArbitraryIntervals(mrng) {
    const ret = new Array();
    const absolute = mrng.nextBoolean();
    // Ascending or descending scaling
    const factor = (mrng.nextBoolean() ? 1 : -1) * (absolute ? 10 : 1);
    const bias = absolute ? 50 : 0;
    // Begin with a full schedule
    ret.push({ lower: 0, upper: 10, change: -2 * factor + bias });
    ret.push({ lower: 10, upper: 20, change: -1 * factor + bias });
    ret.push({ lower: 20, upper: 60, change: 0 + bias });
    ret.push({ lower: 60, upper: 80, change: 0 + bias });
    ret.push({ lower: 80, upper: 90, change: 1 * factor + bias });
    ret.push({ lower: 90, upper: Infinity, change: 2 * factor + bias });
    // Take away parts from this. First we see if we do something to the 0-change alarms.
    // The actions can be: remove it OR turn it into a regular change value.
    const noChanges = ret.filter(x => x.change === bias);
    if (!absolute) {
        if (mrng.nextBoolean()) {
            if (mrng.nextBoolean()) {
                ret.splice(ret.indexOf(noChanges[0]), 1);
            }
            else {
                noChanges[0] = { ...noChanges[0], change: -1 * factor + bias };
            }
        }
        if (mrng.nextBoolean()) {
            if (mrng.nextBoolean()) {
                ret.splice(ret.indexOf(noChanges[1]), 1);
            }
            else {
                noChanges[1] = { ...noChanges[1], change: 1 * factor + bias };
            }
        }
    }
    else {
        // In absolute mode both have to get the same treatment at the same time
        // otherwise we'll end up with a timeline with two gaps
        if (mrng.nextBoolean()) {
            ret.splice(ret.indexOf(noChanges[0]), 1);
            ret.splice(ret.indexOf(noChanges[1]), 1);
        }
        else {
            noChanges[0] = { ...noChanges[0], change: -1 * factor + bias };
            noChanges[1] = { ...noChanges[1], change: 1 * factor + bias };
        }
    }
    // We might also take away either the bottom or the upper half
    if (mrng.nextInt(0, 2) === 0) {
        const signToStrip = mrng.nextBoolean() ? -1 : 1;
        let ix = ret.findIndex(x => Math.sign(x.change - bias) === signToStrip);
        while (ix >= 0) {
            ret.splice(ix, 1);
            ix = ret.findIndex(x => Math.sign(x.change - bias) === signToStrip);
        }
    }
    // Then we're going to arbitrarily get rid of bounds in the most naive way possible
    const iterations = mrng.nextInt(0, 10);
    for (let iter = 0; iter < iterations; iter++) {
        const i = mrng.nextInt(0, ret.length - 1);
        if (mrng.nextBoolean()) {
            // scrap lower bound
            // okay if current interval has an upper bound AND the preceding interval has an upper bound
            if (ret[i].upper !== undefined && (i === 0 || ret[i - 1].upper !== undefined)) {
                ret[i] = { ...ret[i], lower: undefined };
            }
        }
        else {
            // scrap upper bound
            // okay if current interval has a lower bound AND the succeeding interval has a lower bound
            if (ret[i].lower !== undefined && (i === ret.length - 1 || ret[i + 1].lower !== undefined)) {
                ret[i] = { ...ret[i], upper: undefined };
            }
        }
    }
    return { absolute, intervals: ret };
}
exports.generateArbitraryIntervals = generateArbitraryIntervals;
//# sourceMappingURL=data:application/json;base64,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