"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AccessLogFormat = exports.AccessLogField = exports.LogGroupLogDestination = void 0;
/**
 * Use CloudWatch Logs as a custom access log destination for API Gateway.
 */
class LogGroupLogDestination {
    constructor(logGroup) {
        this.logGroup = logGroup;
    }
    /**
     * Binds this destination to the CloudWatch Logs.
     */
    bind(_stage) {
        return {
            destinationArn: this.logGroup.logGroupArn,
        };
    }
}
exports.LogGroupLogDestination = LogGroupLogDestination;
/**
 * $context variables that can be used to customize access log pattern.
 */
class AccessLogField {
    /**
     * The API owner's AWS account ID.
     */
    static contextAccountId() {
        return '$context.identity.accountId';
    }
    /**
     * The identifier API Gateway assigns to your API.
     */
    static contextApiId() {
        return '$context.apiId';
    }
    /**
     * A property of the claims returned from the Amazon Cognito user pool after the method caller is successfully authenticated.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-integrate-with-cognito.html
     *
     * @param property A property key of the claims.
     */
    static contextAuthorizerClaims(property) {
        return `$context.authorizer.claims.${property}`;
    }
    /**
     * The principal user identification associated with the token sent by the client and returned
     * from an API Gateway Lambda authorizer (formerly known as a custom authorizer).
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-use-lambda-authorizer.html
     */
    static contextAuthorizerPrincipalId() {
        return '$context.authorizer.principalId';
    }
    /**
     * The stringified value of the specified key-value pair of the `context` map returned from an API Gateway Lambda authorizer function.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-use-lambda-authorizer.html
     * @param property key of the context map.
     */
    static contextAuthorizer(property) {
        return `$context.authorizer.${property}`;
    }
    /**
     * The AWS endpoint's request ID.
     */
    static contextAwsEndpointRequestId() {
        return '$context.awsEndpointRequestId';
    }
    /**
     * The full domain name used to invoke the API. This should be the same as the incoming `Host` header.
     */
    static contextDomainName() {
        return '$context.domainName';
    }
    /**
     * The first label of the `$context.domainName`. This is often used as a caller/customer identifier.
     */
    static contextDomainPrefix() {
        return '$context.domainPrefix';
    }
    /**
     * A string containing an API Gateway error message.
     */
    static contextErrorMessage() {
        return '$context.error.message';
    }
    /**
     * The quoted value of $context.error.message, namely "$context.error.message".
     */
    static contextErrorMessageString() {
        return '$context.error.messageString';
    }
    /**
     * A type of GatewayResponse. This variable can only be used for simple variable substitution in a GatewayResponse body-mapping template,
     * which is not processed by the Velocity Template Language engine, and in access logging.
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-websocket-api-logging.html
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/customize-gateway-responses.html
     */
    static contextErrorResponseType() {
        return '$context.error.responseType';
    }
    /**
     * A string containing a detailed validation error message.
     */
    static contextErrorValidationErrorString() {
        return '$context.error.validationErrorString';
    }
    /**
     * The extended ID that API Gateway assigns to the API request, which contains more useful information for debugging/troubleshooting.
     */
    static contextExtendedRequestId() {
        return '$context.extendedRequestId';
    }
    /**
     * The HTTP method used. Valid values include: `DELETE`, `GET`, `HEAD`, `OPTIONS`, `PATCH`, `POST`, and `PUT`.
     */
    static contextHttpMethod() {
        return '$context.httpMethod';
    }
    /**
     * The AWS account ID associated with the request.
     */
    static contextIdentityAccountId() {
        return '$context.identity.accountId';
    }
    /**
     * For API methods that require an API key, this variable is the API key associated with the method request.
     * For methods that don't require an API key, this variable is
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-api-usage-plans.html
     */
    static contextIdentityApiKey() {
        return '$context.identity.apiKey';
    }
    /**
     * The API key ID associated with an API request that requires an API key.
     */
    static contextIdentityApiKeyId() {
        return '$context.identity.apiKeyId';
    }
    /**
     * The principal identifier of the caller making the request.
     */
    static contextIdentityCaller() {
        return '$context.identity.caller';
    }
    /**
     * The Amazon Cognito authentication provider used by the caller making the request.
     * Available only if the request was signed with Amazon Cognito credentials.
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-identity.html
     */
    static contextIdentityCognitoAuthenticationProvider() {
        return '$context.identity.cognitoAuthenticationProvider';
    }
    /**
     * The Amazon Cognito authentication type of the caller making the request.
     * Available only if the request was signed with Amazon Cognito credentials.
     */
    static contextIdentityCognitoAuthenticationType() {
        return '$context.identity.cognitoAuthenticationType';
    }
    /**
     * The Amazon Cognito identity ID of the caller making the request. Available only if the request was signed with Amazon Cognito credentials.
     */
    static contextIdentityCognitoIdentityId() {
        return '$context.identity.cognitoIdentityId';
    }
    /**
     * The Amazon Cognito identity pool ID of the caller making the request.
     * Available only if the request was signed with Amazon Cognito credentials.
     */
    static contextIdentityCognitoIdentityPoolId() {
        return '$context.identity.cognitoIdentityPoolId';
    }
    /**
     * The AWS organization ID.
     */
    static contextIdentityPrincipalOrgId() {
        return '$context.identity.principalOrgId';
    }
    /**
     * The source IP address of the TCP connection making the request to API Gateway.
     * Warning: You should not trust this value if there is any chance that the `X-Forwarded-For` header could be forged.
     */
    static contextIdentitySourceIp() {
        return '$context.identity.sourceIp';
    }
    /**
     * The principal identifier of the user making the request. Used in Lambda authorizers.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-lambda-authorizer-output.html
     */
    static contextIdentityUser() {
        return '$context.identity.user';
    }
    /**
     * The User-Agent header of the API caller.
     */
    static contextIdentityUserAgent() {
        return '$context.identity.userAgent';
    }
    /**
     * The Amazon Resource Name (ARN) of the effective user identified after authentication.
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/id_users.html
     */
    static contextIdentityUserArn() {
        return '$context.identity.userArn';
    }
    /**
     * The request path.
     * For example, for a non-proxy request URL of https://{rest-api-id.execute-api.{region}.amazonaws.com/{stage}/root/child,
     * this value is /{stage}/root/child.
     */
    static contextPath() {
        return '$context.path';
    }
    /**
     * The request protocol, for example, HTTP/1.1.
     */
    static contextProtocol() {
        return '$context.protocol';
    }
    /**
     * The ID that API Gateway assigns to the API request.
     */
    static contextRequestId() {
        return '$context.requestId';
    }
    /**
     * The request header override.
     * If this parameter is defined, it contains the headers to be used instead of the HTTP Headers that are defined in the Integration Request pane.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-override-request-response-parameters.html
     *
     * @param headerName
     */
    static contextRequestOverrideHeader(headerName) {
        return `$context.requestOverride.header.${headerName}`;
    }
    /**
     * The request path override. If this parameter is defined,
     * it contains the request path to be used instead of the URL Path Parameters that are defined in the Integration Request pane.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-override-request-response-parameters.html
     *
     * @param pathName
     */
    static contextRequestOverridePath(pathName) {
        return `$context.requestOverride.path.${pathName}`;
    }
    /**
     * The request query string override.
     * If this parameter is defined, it contains the request query strings to be used instead
     * of the URL Query String Parameters that are defined in the Integration Request pane.
     *
     * @param querystringName
     */
    static contextRequestOverrideQuerystring(querystringName) {
        return `$context.requestOverride.querystring.${querystringName}`;
    }
    /**
     * The response header override.
     * If this parameter is defined, it contains the header to be returned instead of the Response header
     * that is defined as the Default mapping in the Integration Response pane.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-override-request-response-parameters.html
     *
     * @param headerName
     */
    static contextResponseOverrideHeader(headerName) {
        return `$context.responseOverride.header.${headerName}`;
    }
    /**
     * The response status code override.
     * If this parameter is defined, it contains the status code to be returned instead of the Method response status
     * that is defined as the Default mapping in the Integration Response pane.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-override-request-response-parameters.html
     */
    static contextResponseOverrideStatus() {
        return '$context.responseOverride.status';
    }
    /**
     * The CLF-formatted request time (dd/MMM/yyyy:HH:mm:ss +-hhmm).
     */
    static contextRequestTime() {
        return '$context.requestTime';
    }
    /**
     * The Epoch-formatted request time.
     */
    static contextRequestTimeEpoch() {
        return '$context.requestTimeEpoch';
    }
    /**
     * The identifier that API Gateway assigns to your resource.
     */
    static contextResourceId() {
        return '$context.resourceId';
    }
    /**
     * The path to your resource.
     * For example, for the non-proxy request URI of `https://{rest-api-id.execute-api.{region}.amazonaws.com/{stage}/root/child`,
     * The $context.resourcePath value is `/root/child`.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-create-api-step-by-step.html
     */
    static contextResourcePath() {
        return '$context.resourcePath';
    }
    /**
     * The deployment stage of the API request (for example, `Beta` or `Prod`).
     */
    static contextStage() {
        return '$context.stage';
    }
    /**
     * The response received from AWS WAF: `WAF_ALLOW` or `WAF_BLOCK`. Will not be set if the stage is not associated with a web ACL.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-control-access-aws-waf.html
     */
    static contextWafResponseCode() {
        return '$context.wafResponseCode';
    }
    /**
     * The complete ARN of the web ACL that is used to decide whether to allow or block the request.
     * Will not be set if the stage is not associated with a web ACL.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-control-access-aws-waf.html
     */
    static contextWebaclArn() {
        return '$context.webaclArn';
    }
    /**
     * The trace ID for the X-Ray trace.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-enabling-xray.html
     */
    static contextXrayTraceId() {
        return '$context.xrayTraceId';
    }
    /**
     * The authorizer latency in ms.
     */
    static contextAuthorizerIntegrationLatency() {
        return '$context.authorizer.integrationLatency';
    }
    /**
     * The integration latency in ms.
     */
    static contextIntegrationLatency() {
        return '$context.integrationLatency';
    }
    /**
     * For Lambda proxy integration, this parameter represents the status code returned from AWS Lambda,
     * not from the backend Lambda function.
     */
    static contextIntegrationStatus() {
        return '$context.integrationStatus.';
    }
    /**
     * The response latency in ms.
     */
    static contextResponseLatency() {
        return '$context.responseLatency';
    }
    /**
     * The response payload length.
     */
    static contextResponseLength() {
        return '$context.responseLength';
    }
    /**
     * The method response status.
     */
    static contextStatus() {
        return '$context.status';
    }
}
exports.AccessLogField = AccessLogField;
/**
 * factory methods for access log format.
 */
class AccessLogFormat {
    constructor(format) {
        this.format = format;
    }
    /**
     * Custom log format.
     * You can create any log format string. You can easily get the $ context variable by using the methods of AccessLogField.
     * @param format
     * @example custom(JSON.stringify({
     *  requestId: AccessLogField.contextRequestId(),
     *  sourceIp: AccessLogField.contextIdentitySourceIp(),
     *  method: AccessLogFiled.contextHttpMethod(),
     *  userContext: {
     *    sub: AccessLogField.contextAuthorizerClaims('sub'),
     *    email: AccessLogField.contextAuthorizerClaims('email')
     *  }
     * }))
     */
    static custom(format) {
        return new AccessLogFormat(format);
    }
    /**
     * Generate Common Log Format.
     */
    static clf() {
        const requester = [AccessLogField.contextIdentitySourceIp(), AccessLogField.contextIdentityCaller(), AccessLogField.contextIdentityUser()].join(' ');
        const requestTime = AccessLogField.contextRequestTime();
        const request = [AccessLogField.contextHttpMethod(), AccessLogField.contextResourcePath(), AccessLogField.contextProtocol()].join(' ');
        const status = [AccessLogField.contextStatus(), AccessLogField.contextResponseLength(), AccessLogField.contextRequestId()].join(' ');
        return new AccessLogFormat(`${requester} [${requestTime}] "${request}" ${status}`);
    }
    /**
     * Access log will be produced in the JSON format with a set of fields most useful in the access log. All fields are turned on by default with the
     * option to turn off specific fields.
     */
    static jsonWithStandardFields(fields = {
        ip: true,
        user: true,
        caller: true,
        requestTime: true,
        httpMethod: true,
        resourcePath: true,
        status: true,
        protocol: true,
        responseLength: true,
    }) {
        return this.custom(JSON.stringify({
            requestId: AccessLogField.contextRequestId(),
            ip: fields.ip ? AccessLogField.contextIdentitySourceIp() : undefined,
            user: fields.user ? AccessLogField.contextIdentityUser() : undefined,
            caller: fields.caller ? AccessLogField.contextIdentityCaller() : undefined,
            requestTime: fields.requestTime ? AccessLogField.contextRequestTime() : undefined,
            httpMethod: fields.httpMethod ? AccessLogField.contextHttpMethod() : undefined,
            resourcePath: fields.resourcePath ? AccessLogField.contextResourcePath() : undefined,
            status: fields.status ? AccessLogField.contextStatus() : undefined,
            protocol: fields.protocol ? AccessLogField.contextProtocol() : undefined,
            responseLength: fields.responseLength ? AccessLogField.contextResponseLength() : undefined,
        }));
    }
    /**
     * Output a format string to be used with CloudFormation.
     */
    toString() {
        return this.format;
    }
}
exports.AccessLogFormat = AccessLogFormat;
//# sourceMappingURL=data:application/json;base64,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