"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaIntegration = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const aws_1 = require("./aws");
/**
 * Integrates an AWS Lambda function to an API Gateway method.
 *
 * @example
 *
 *    const handler = new lambda.Function(this, 'MyFunction', ...);
 *    api.addMethod('GET', new LambdaIntegration(handler));
 *
 */
class LambdaIntegration extends aws_1.AwsIntegration {
    constructor(handler, options = {}) {
        const proxy = options.proxy === undefined ? true : options.proxy;
        super({
            proxy,
            service: 'lambda',
            path: `2015-03-31/functions/${handler.functionArn}/invocations`,
            options,
        });
        this.handler = handler;
        this.enableTest = options.allowTestInvoke === undefined ? true : false;
    }
    bind(method) {
        super.bind(method);
        const principal = new iam.ServicePrincipal('apigateway.amazonaws.com');
        const desc = `${method.restApi.node.uniqueId}.${method.httpMethod}.${method.resource.path.replace(/\//g, '.')}`;
        this.handler.addPermission(`ApiPermission.${desc}`, {
            principal,
            scope: method,
            sourceArn: core_1.Lazy.stringValue({ produce: () => method.methodArn }),
        });
        // add permission to invoke from the console
        if (this.enableTest) {
            this.handler.addPermission(`ApiPermission.Test.${desc}`, {
                principal,
                scope: method,
                sourceArn: method.testMethodArn,
            });
        }
    }
}
exports.LambdaIntegration = LambdaIntegration;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibGFtYmRhLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsibGFtYmRhLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLHdDQUF3QztBQUV4Qyx3Q0FBcUM7QUFHckMsK0JBQXVDO0FBc0J2Qzs7Ozs7Ozs7R0FRRztBQUNILE1BQWEsaUJBQWtCLFNBQVEsb0JBQWM7SUFJbkQsWUFBWSxPQUF5QixFQUFFLFVBQW9DLEVBQUc7UUFDNUUsTUFBTSxLQUFLLEdBQUcsT0FBTyxDQUFDLEtBQUssS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQztRQUVqRSxLQUFLLENBQUM7WUFDSixLQUFLO1lBQ0wsT0FBTyxFQUFFLFFBQVE7WUFDakIsSUFBSSxFQUFFLHdCQUF3QixPQUFPLENBQUMsV0FBVyxjQUFjO1lBQy9ELE9BQU87U0FDUixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsT0FBTyxHQUFHLE9BQU8sQ0FBQztRQUN2QixJQUFJLENBQUMsVUFBVSxHQUFHLE9BQU8sQ0FBQyxlQUFlLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQztJQUN6RSxDQUFDO0lBRU0sSUFBSSxDQUFDLE1BQWM7UUFDeEIsS0FBSyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUNuQixNQUFNLFNBQVMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO1FBRXZFLE1BQU0sSUFBSSxHQUFHLEdBQUcsTUFBTSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsUUFBUSxJQUFJLE1BQU0sQ0FBQyxVQUFVLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLEtBQUssRUFBRSxHQUFHLENBQUMsRUFBRSxDQUFDO1FBRWhILElBQUksQ0FBQyxPQUFPLENBQUMsYUFBYSxDQUFDLGlCQUFpQixJQUFJLEVBQUUsRUFBRTtZQUNsRCxTQUFTO1lBQ1QsS0FBSyxFQUFFLE1BQU07WUFDYixTQUFTLEVBQUUsV0FBSSxDQUFDLFdBQVcsQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxNQUFNLENBQUMsU0FBUyxFQUFFLENBQUM7U0FDakUsQ0FBQyxDQUFDO1FBRUgsNENBQTRDO1FBQzVDLElBQUksSUFBSSxDQUFDLFVBQVUsRUFBRTtZQUNuQixJQUFJLENBQUMsT0FBTyxDQUFDLGFBQWEsQ0FBQyxzQkFBc0IsSUFBSSxFQUFFLEVBQUU7Z0JBQ3ZELFNBQVM7Z0JBQ1QsS0FBSyxFQUFFLE1BQU07Z0JBQ2IsU0FBUyxFQUFFLE1BQU0sQ0FBQyxhQUFhO2FBQ2hDLENBQUMsQ0FBQztTQUNKO0lBQ0gsQ0FBQztDQUNGO0FBdkNELDhDQXVDQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGlhbSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCAqIGFzIGxhbWJkYSBmcm9tICdAYXdzLWNkay9hd3MtbGFtYmRhJztcbmltcG9ydCB7IExhenkgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IEludGVncmF0aW9uT3B0aW9ucyB9IGZyb20gJy4uL2ludGVncmF0aW9uJztcbmltcG9ydCB7IE1ldGhvZCB9IGZyb20gJy4uL21ldGhvZCc7XG5pbXBvcnQgeyBBd3NJbnRlZ3JhdGlvbiB9IGZyb20gJy4vYXdzJztcblxuZXhwb3J0IGludGVyZmFjZSBMYW1iZGFJbnRlZ3JhdGlvbk9wdGlvbnMgZXh0ZW5kcyBJbnRlZ3JhdGlvbk9wdGlvbnMge1xuICAvKipcbiAgICogVXNlIHByb3h5IGludGVncmF0aW9uIG9yIG5vcm1hbCAocmVxdWVzdC9yZXNwb25zZSBtYXBwaW5nKSBpbnRlZ3JhdGlvbi5cbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgcHJveHk/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBBbGxvdyBpbnZva2luZyBtZXRob2QgZnJvbSBBV1MgQ29uc29sZSBVSSAoZm9yIHRlc3RpbmcgcHVycG9zZXMpLlxuICAgKlxuICAgKiBUaGlzIHdpbGwgYWRkIGFub3RoZXIgcGVybWlzc2lvbiB0byB0aGUgQVdTIExhbWJkYSByZXNvdXJjZSBwb2xpY3kgd2hpY2hcbiAgICogd2lsbCBhbGxvdyB0aGUgYHRlc3QtaW52b2tlLXN0YWdlYCBzdGFnZSB0byBpbnZva2UgdGhpcyBoYW5kbGVyLiBJZiB0aGlzXG4gICAqIGlzIHNldCB0byBgZmFsc2VgLCB0aGUgZnVuY3Rpb24gd2lsbCBvbmx5IGJlIHVzYWJsZSBmcm9tIHRoZSBkZXBsb3ltZW50XG4gICAqIGVuZHBvaW50LlxuICAgKlxuICAgKiBAZGVmYXVsdCB0cnVlXG4gICAqL1xuICByZWFkb25seSBhbGxvd1Rlc3RJbnZva2U/OiBib29sZWFuO1xufVxuXG4vKipcbiAqIEludGVncmF0ZXMgYW4gQVdTIExhbWJkYSBmdW5jdGlvbiB0byBhbiBBUEkgR2F0ZXdheSBtZXRob2QuXG4gKlxuICogQGV4YW1wbGVcbiAqXG4gKiAgICBjb25zdCBoYW5kbGVyID0gbmV3IGxhbWJkYS5GdW5jdGlvbih0aGlzLCAnTXlGdW5jdGlvbicsIC4uLik7XG4gKiAgICBhcGkuYWRkTWV0aG9kKCdHRVQnLCBuZXcgTGFtYmRhSW50ZWdyYXRpb24oaGFuZGxlcikpO1xuICpcbiAqL1xuZXhwb3J0IGNsYXNzIExhbWJkYUludGVncmF0aW9uIGV4dGVuZHMgQXdzSW50ZWdyYXRpb24ge1xuICBwcml2YXRlIHJlYWRvbmx5IGhhbmRsZXI6IGxhbWJkYS5JRnVuY3Rpb247XG4gIHByaXZhdGUgcmVhZG9ubHkgZW5hYmxlVGVzdDogYm9vbGVhbjtcblxuICBjb25zdHJ1Y3RvcihoYW5kbGVyOiBsYW1iZGEuSUZ1bmN0aW9uLCBvcHRpb25zOiBMYW1iZGFJbnRlZ3JhdGlvbk9wdGlvbnMgPSB7IH0pIHtcbiAgICBjb25zdCBwcm94eSA9IG9wdGlvbnMucHJveHkgPT09IHVuZGVmaW5lZCA/IHRydWUgOiBvcHRpb25zLnByb3h5O1xuXG4gICAgc3VwZXIoe1xuICAgICAgcHJveHksXG4gICAgICBzZXJ2aWNlOiAnbGFtYmRhJyxcbiAgICAgIHBhdGg6IGAyMDE1LTAzLTMxL2Z1bmN0aW9ucy8ke2hhbmRsZXIuZnVuY3Rpb25Bcm59L2ludm9jYXRpb25zYCxcbiAgICAgIG9wdGlvbnMsXG4gICAgfSk7XG5cbiAgICB0aGlzLmhhbmRsZXIgPSBoYW5kbGVyO1xuICAgIHRoaXMuZW5hYmxlVGVzdCA9IG9wdGlvbnMuYWxsb3dUZXN0SW52b2tlID09PSB1bmRlZmluZWQgPyB0cnVlIDogZmFsc2U7XG4gIH1cblxuICBwdWJsaWMgYmluZChtZXRob2Q6IE1ldGhvZCkge1xuICAgIHN1cGVyLmJpbmQobWV0aG9kKTtcbiAgICBjb25zdCBwcmluY2lwYWwgPSBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2FwaWdhdGV3YXkuYW1hem9uYXdzLmNvbScpO1xuXG4gICAgY29uc3QgZGVzYyA9IGAke21ldGhvZC5yZXN0QXBpLm5vZGUudW5pcXVlSWR9LiR7bWV0aG9kLmh0dHBNZXRob2R9LiR7bWV0aG9kLnJlc291cmNlLnBhdGgucmVwbGFjZSgvXFwvL2csICcuJyl9YDtcblxuICAgIHRoaXMuaGFuZGxlci5hZGRQZXJtaXNzaW9uKGBBcGlQZXJtaXNzaW9uLiR7ZGVzY31gLCB7XG4gICAgICBwcmluY2lwYWwsXG4gICAgICBzY29wZTogbWV0aG9kLFxuICAgICAgc291cmNlQXJuOiBMYXp5LnN0cmluZ1ZhbHVlKHsgcHJvZHVjZTogKCkgPT4gbWV0aG9kLm1ldGhvZEFybiB9KSxcbiAgICB9KTtcblxuICAgIC8vIGFkZCBwZXJtaXNzaW9uIHRvIGludm9rZSBmcm9tIHRoZSBjb25zb2xlXG4gICAgaWYgKHRoaXMuZW5hYmxlVGVzdCkge1xuICAgICAgdGhpcy5oYW5kbGVyLmFkZFBlcm1pc3Npb24oYEFwaVBlcm1pc3Npb24uVGVzdC4ke2Rlc2N9YCwge1xuICAgICAgICBwcmluY2lwYWwsXG4gICAgICAgIHNjb3BlOiBtZXRob2QsXG4gICAgICAgIHNvdXJjZUFybjogbWV0aG9kLnRlc3RNZXRob2RBcm4sXG4gICAgICB9KTtcbiAgICB9XG4gIH1cbn1cbiJdfQ==