// Copyright (c) Mito
import React, { useEffect } from 'react';
import { getRandomId } from '../../api';
import Dropdown from '../elements/Dropdown';
import DropdownItem from '../elements/DropdownItem';
import { ModalEnum } from '../modals/modals';
import { getDefaultGraphParams } from '../taskpanes/Graph/graphUtils';
/*
    Helper function for finding all of the graph tab names
    that are created from a given sheet index
*/
export const getGraphTabNamesAndIDsFromSheetIndex = (sheetIndex, graphDataDict) => {
    // Filter to only grapsh with the sheetIndex, and then get a list of the graph tab names
    const filteredGraphDataJSON = Object.fromEntries(Object.entries(graphDataDict).filter(([, graphData]) => {
        return graphData.graphParams.graphCreation.sheet_index === sheetIndex;
    }));
    return Object.entries(filteredGraphDataJSON).map(([graphID, graphData]) => {
        return { graphTabName: graphData.graphTabName, graphID: graphID };
    });
};
/*
    Displays a set of actions one can perform on a data sheet tab, including
    deleting, duplicating, or renaming, and creating a sheet.
*/
export default function SheetTabActions(props) {
    // Log opening the data sheet tab actions
    useEffect(() => {
        void props.mitoAPI.log('clicked_data_sheet_tab_actions', {
            sheet_index: props.sheetIndex
        });
    }, []);
    const onDelete = async () => {
        const dependantGraphTabNamesAndIDs = getGraphTabNamesAndIDsFromSheetIndex(props.sheetIndex, props.graphDataDict);
        if (dependantGraphTabNamesAndIDs.length > 0) {
            props.setUIState(prevUIState => {
                return Object.assign(Object.assign({}, prevUIState), { currOpenModal: {
                        type: ModalEnum.DeleteGraphs,
                        dependantGraphTabNamesAndIDs: dependantGraphTabNamesAndIDs,
                        sheetIndex: props.sheetIndex
                    } });
            });
        }
        else {
            // Select the previous sheet
            props.setUIState(prevUIState => {
                return Object.assign(Object.assign({}, prevUIState), { selectedTabType: 'data', selectedSheetIndex: prevUIState.selectedSheetIndex > 0 ? prevUIState.selectedSheetIndex - 1 : 0 });
            });
            // Close 
            props.closeOpenEditingPopups();
            await props.mitoAPI.editDataframeDelete(props.sheetIndex);
        }
    };
    const onDuplicate = async () => {
        // Close 
        props.closeOpenEditingPopups();
        await props.mitoAPI.editDataframeDuplicate(props.sheetIndex);
    };
    /* Rename helper, which requires changes to the sheet tab itself */
    const onRename = () => {
        props.setIsRename(true);
    };
    const graphData = async () => {
        const newGraphID = getRandomId(); // Create a new graph
        const graphParams = getDefaultGraphParams(props.sheetDataArray, props.sheetIndex);
        // In order to open the graph, we are watching for the graphDataDict to change in length. This allows us to only display the graph taskpane
        // when the sheet tab exists. However, we need to know the stepID of the graph creation so that the configuration of the graph so that editing
        // the graph doesn't cause a new step to get created. We could either create a new piece of state in the that stores the stepID or we can save the 
        // stepID in the graphDataDict. Neither are good options. 
        // NOTE: after trying to implement it as saving it in the state, I realized that this approach doesn't work due to race conditions 
        // in the return stepID from the editGraph call and the graphDATAJSON updating. 
        await props.mitoAPI.editGraph(newGraphID, graphParams.graphCreation.graph_type, graphParams.graphCreation.sheet_index, graphParams.graphPreprocessing.safety_filter_turned_on_by_user, graphParams.graphCreation.x_axis_column_ids, graphParams.graphCreation.y_axis_column_ids, `100%`, `100%`, undefined);
    };
    return (React.createElement(Dropdown, { closeDropdown: () => props.setDisplayActions(false), width: 'small' },
        React.createElement(DropdownItem, { title: 'Create graph', onClick: (e) => {
                // Stop propogation so that the onClick of the sheet tab div
                // doesn't compete updating the uiState to this sheet instead of
                // the new graphID that we're creating
                e === null || e === void 0 ? void 0 : e.stopPropagation();
                void graphData();
            } }),
        React.createElement(DropdownItem, { title: 'Delete', onClick: (e) => {
                // Stop propogation so that the onClick of the sheet tab div
                // doesn't compete updating the uiState to the graphID that is gettind deleted
                e === null || e === void 0 ? void 0 : e.stopPropagation();
                void onDelete();
            } }),
        React.createElement(DropdownItem, { title: 'Duplicate', onClick: onDuplicate }),
        React.createElement(DropdownItem, { title: 'Rename', onClick: onRename })));
}
//# sourceMappingURL=DataSheetTabActions.js.map