// Copyright (c) Mito
import React, { useEffect, useState } from 'react';
import DefaultTaskpane from '../DefaultTaskpane/DefaultTaskpane';
import FileBrowser from './FileBrowser';
import { getFileEnding, getInvalidFileError } from './FileBrowserElement';
import TextButton from '../../elements/TextButton';
import XLSXImport from './XLSXImport';
// CSS
import '../../../../css/taskpanes/Import/ImportTaskpane.css';
import DefaultTaskpaneHeader from '../DefaultTaskpane/DefaultTaskpaneHeader';
import DefaultTaskpaneBody from '../DefaultTaskpane/DefaultTaskpaneBody';
/*
    Helper function that returns if the import button is usable,
    and also the message to display on the button based on which
    element is selected.
*/
const getImportButtonStatus = (selectedElement, excelImportEnabled) => {
    if (selectedElement === undefined) {
        return {
            disabled: true,
            buttonText: 'Select a File to Import'
        };
    }
    if (selectedElement.isDirectory) {
        return {
            disabled: true,
            buttonText: 'That\'s a Directory. Select a File'
        };
    }
    const invalidFileError = getInvalidFileError(selectedElement, excelImportEnabled);
    if (invalidFileError !== undefined) {
        return {
            disabled: true,
            buttonText: 'Select a Supported File Type'
        };
    }
    return {
        disabled: false,
        buttonText: 'Import ' + selectedElement.name
    };
};
/*
    Imports the selected element by:
    1. Combining the path into one path string
    2. Passing this combined path into a simple import
*/
export async function doImport(mitoAPI, currPathParts, element) {
    // Construct the final path that must be imported
    const finalPath = [...currPathParts];
    finalPath.push(element.name);
    const joinedPath = await mitoAPI.getPathJoined(finalPath);
    if (joinedPath === undefined) {
        return;
    }
    // And then actually import it
    await mitoAPI.editSimpleImport([joinedPath]);
}
/*
    Provides a import modal that allows users to import data
    using a file browser
*/
function ImportTaskpane(props) {
    // The path data for the currently selected path
    const [pathContents, setPathContents] = useState(undefined);
    // The file/folder that is currently selected 
    const [selectedElement, setSelectedElement] = useState(undefined);
    // If the file being imported is an XLSX, we need additional configuration
    // and so we use an import wizard for help
    const [fileForImportWizard, setFileForImportWizard] = useState(undefined);
    // Loads the path data from the API and sets it for the file browser
    async function loadPathContents(currPathParts) {
        const _pathContents = await props.mitoAPI.getPathContents(currPathParts);
        setPathContents(_pathContents);
    }
    async function importElement(element) {
        const importButtonStatus = getImportButtonStatus(element, props.userProfile.excelImportEnabled);
        // Quit early if the selected thing is not importable, or if there
        // is nothing even selected
        if (importButtonStatus.disabled || element === undefined) {
            return;
        }
        if (!(element === null || element === void 0 ? void 0 : element.isDirectory) && (element === null || element === void 0 ? void 0 : element.name.toLowerCase().endsWith('.xlsx'))) {
            setFileForImportWizard(element.name);
            return;
        }
        // Do the actual import
        await doImport(props.mitoAPI, props.currPathParts, element);
        // And then clear the selected element
        setSelectedElement(undefined);
    }
    /*
        Any time the current path changes, we update
        the files that are displayed
    */
    useEffect(() => {
        // When the current path changes, we reload the path contents
        void loadPathContents(props.currPathParts);
        // We also unselect anything that might be selected
        setSelectedElement(undefined);
        // Log how long the path is
        void props.mitoAPI.log('curr_path_changed', { 'path_parts_length': props.currPathParts.length });
    }, [props.currPathParts]);
    /*
        Any time the selected element changes we log the file
        ending (or none, if it has none).
    */
    useEffect(() => {
        let selectedElementName = '';
        if (selectedElement === undefined) {
            selectedElementName = 'undefined';
        }
        else if (selectedElement.isDirectory) {
            selectedElementName = 'directory';
        }
        else {
            const fileEnding = getFileEnding(selectedElement.name);
            if (fileEnding !== undefined) {
                selectedElementName = fileEnding;
            }
            else {
                selectedElementName = 'No File Ending';
            }
        }
        void props.mitoAPI.log('selected_element_changed', { 'selected_element': selectedElementName });
    }, [selectedElement]);
    const importButtonStatus = getImportButtonStatus(selectedElement, props.userProfile.excelImportEnabled);
    return (React.createElement(DefaultTaskpane, null,
        React.createElement(DefaultTaskpaneHeader, { header: fileForImportWizard === undefined ? 'Import Files' : `Import ${fileForImportWizard}`, setUIState: props.setUIState, backCallback: fileForImportWizard === undefined ? undefined : () => {
                setFileForImportWizard(undefined);
            } }),
        React.createElement(DefaultTaskpaneBody, { noScroll: true },
            React.createElement("div", { className: 'import-taskpane flexbox-column flexbox-space-between' },
                fileForImportWizard === undefined &&
                    React.createElement(React.Fragment, null,
                        React.createElement(FileBrowser, { mitoAPI: props.mitoAPI, setCurrPathParts: props.setCurrPathParts, pathParts: pathContents === null || pathContents === void 0 ? void 0 : pathContents.path_parts, elements: (pathContents === null || pathContents === void 0 ? void 0 : pathContents.elements) || [], selectedElement: selectedElement, setSelectedElement: setSelectedElement, importElement: importElement, userProfile: props.userProfile }),
                        React.createElement("div", { className: 'import-taskpane-import-button-container' },
                            React.createElement(TextButton, { variant: 'dark', width: 'block', onClick: () => {
                                    void importElement(selectedElement);
                                }, disabled: importButtonStatus.disabled }, importButtonStatus.buttonText))),
                fileForImportWizard !== undefined &&
                    React.createElement(XLSXImport, { mitoAPI: props.mitoAPI, pathParts: [...props.currPathParts, fileForImportWizard] })))));
}
export default ImportTaskpane;
//# sourceMappingURL=ImportTaskpane.js.map