"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseCluster = exports.ServerlessLaravel = exports.ServerlessApi = void 0;
const cdk = require("@aws-cdk/core");
const apigateway = require("@aws-cdk/aws-apigatewayv2");
const lambda = require("@aws-cdk/aws-lambda");
const rds = require("@aws-cdk/aws-rds");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
/**
 * Use `ServerlessApi` to create the serverless API resource
 */
class ServerlessApi extends cdk.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l;
        super(scope, id);
        const DEFAULT_LAMBDA_ASSET_PATH = path.join(__dirname, '../composer/laravel58-bref');
        const DEFAULT_DB_MASTER_USER = 'admin';
        this.vpc = (_a = props.vpc) !== null && _a !== void 0 ? _a : new aws_ec2_1.Vpc(this, 'Vpc', { maxAzs: 3, natGateways: 1 });
        this.handler = (_b = props.handler) !== null && _b !== void 0 ? _b : new lambda.Function(this, 'handler', {
            runtime: lambda.Runtime.PROVIDED,
            handler: 'public/index.php',
            layers: [
                lambda.LayerVersion.fromLayerVersionArn(this, 'BrefPHPLayer', props.brefLayerVersion),
            ],
            code: lambda.Code.fromAsset((_c = props === null || props === void 0 ? void 0 : props.lambdaCodePath) !== null && _c !== void 0 ? _c : DEFAULT_LAMBDA_ASSET_PATH),
            environment: {
                APP_STORAGE: '/tmp',
                DB_WRITER: (_e = (_d = props.databaseConfig) === null || _d === void 0 ? void 0 : _d.writerEndpoint) !== null && _e !== void 0 ? _e : '',
                DB_READER: (_j = (_g = (_f = props.databaseConfig) === null || _f === void 0 ? void 0 : _f.readerEndpoint) !== null && _g !== void 0 ? _g : (_h = props.databaseConfig) === null || _h === void 0 ? void 0 : _h.writerEndpoint) !== null && _j !== void 0 ? _j : '',
                DB_USER: (_l = (_k = props.databaseConfig) === null || _k === void 0 ? void 0 : _k.masterUserName) !== null && _l !== void 0 ? _l : DEFAULT_DB_MASTER_USER,
            },
            timeout: cdk.Duration.seconds(120),
            vpc: this.vpc,
        });
        // allow lambda execution role to connect to RDS proxy
        if (props.rdsProxy) {
            this.handler.addToRolePolicy(new iam.PolicyStatement({
                actions: ['rds-db:connect'],
                resources: [props.rdsProxy.dbProxyArn],
            }));
        }
        const endpoint = new apigateway.HttpApi(this, 'apiservice', {
            defaultIntegration: new apigateway.LambdaProxyIntegration({
                handler: this.handler,
            }),
        });
        new cdk.CfnOutput(this, 'EndpointURL', { value: endpoint.url });
    }
}
exports.ServerlessApi = ServerlessApi;
/**
 * Use `ServerlessLaravel` to create the serverless Laravel resource
 */
class ServerlessLaravel extends ServerlessApi {
    constructor(scope, id, props) {
        super(scope, id, {
            lambdaCodePath: props.laravelPath,
            brefLayerVersion: props.brefLayerVersion,
            handler: props.handler,
            vpc: props.vpc,
            databaseConfig: props.databaseConfig,
            rdsProxy: props.rdsProxy,
        });
    }
}
exports.ServerlessLaravel = ServerlessLaravel;
class DatabaseCluster extends cdk.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id);
        this.masterUser = (_a = props.masterUserName) !== null && _a !== void 0 ? _a : 'admin';
        // generate and store password for masterUser in the secrets manager
        const masterUserSecret = new secretsmanager.Secret(this, 'DbMasterSecret', {
            secretName: `${cdk.Stack.of(this).stackName}-DbMasterSecret`,
            generateSecretString: {
                secretStringTemplate: JSON.stringify({
                    username: this.masterUser,
                }),
                passwordLength: 12,
                excludePunctuation: true,
                includeSpace: false,
                generateStringKey: 'password',
            },
        });
        this.masterPassword = masterUserSecret;
        const dbConnectionGroup = new aws_ec2_1.SecurityGroup(this, 'DB Secuirty Group', {
            vpc: props.vpc,
        });
        dbConnectionGroup.connections.allowInternally(aws_ec2_1.Port.tcp(3306));
        const dbCluster = new rds.DatabaseCluster(this, 'DBCluster', {
            engine: rds.DatabaseClusterEngine.auroraMysql({
                version: rds.AuroraMysqlEngineVersion.VER_2_08_1,
            }),
            instanceProps: {
                vpc: props.vpc,
                instanceType: (_b = props.instanceType) !== null && _b !== void 0 ? _b : new aws_ec2_1.InstanceType('t3.medium'),
                securityGroups: [dbConnectionGroup],
            },
            masterUser: {
                username: masterUserSecret.secretValueFromJson('username').toString(),
                password: masterUserSecret.secretValueFromJson('password'),
            },
            instances: props.instanceCapacity,
            removalPolicy: core_1.RemovalPolicy.DESTROY,
        });
        // Workaround for bug where TargetGroupName is not set but required
        let cfnDbInstance = dbCluster.node.children.find((child) => {
            return child instanceof rds.CfnDBInstance;
        });
        // enable the RDS proxy by default
        if (!props.rdsProxy === false) {
            // create iam role for RDS proxy
            const rdsProxyRole = new iam.Role(this, 'RdsProxyRole', {
                assumedBy: new iam.ServicePrincipal('rds.amazonaws.com'),
            });
            // see: https://aws.amazon.com/tw/blogs/compute/using-amazon-rds-proxy-with-aws-lambda/
            rdsProxyRole.addToPolicy(new iam.PolicyStatement({
                actions: [
                    'secretsmanager:GetResourcePolicy',
                    'secretsmanager:GetSecretValue',
                    'secretsmanager:DescribeSecret',
                    'secretsmanager:ListSecretVersionIds',
                ],
                resources: [masterUserSecret.secretArn],
            }));
            const proxyOptions = {
                vpc: props.vpc,
                secrets: [masterUserSecret],
                iamAuth: true,
                dbProxyName: `${cdk.Stack.of(this).stackName}-RDSProxy`,
                securityGroups: [dbConnectionGroup],
                role: rdsProxyRole,
            };
            // create the RDS proxy
            this.rdsProxy = dbCluster.addProxy('RDSProxy', proxyOptions);
            // ensure DB instance is ready before creating the proxy
            (_c = this.rdsProxy) === null || _c === void 0 ? void 0 : _c.node.addDependency(cfnDbInstance);
        }
    }
}
exports.DatabaseCluster = DatabaseCluster;
//# sourceMappingURL=data:application/json;base64,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