# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['rst2html5']

package_data = \
{'': ['*']}

install_requires = \
['Genshi>=0.7', 'Pygments>=2.5', 'docutils>=0.13.1']

entry_points = \
{'console_scripts': ['rst2html5 = rst2html5_:main']}

setup_kwargs = {
    'name': 'rst2html5',
    'version': '2.0',
    'description': 'Generates (X)HTML5 documents from standalone reStructuredText sources',
    'long_description': '=========\nrst2html5\n=========\n\n*rst2html5* generates (X)HTML5 documents from standalone reStructuredText sources.\nIt is a complete rewrite of the docutils\' *rst2html* and uses new HTML5 constructs such as\n:code:`<section>` and :code:`<aside>`.\n\n\nInstallation\n============\n\n.. code-block:: bash\n\n    $ pip install rst2html5\n\n\nUsage\n=====\n\n.. code-block:: bash\n\n\t$ rst2html5 [options] SOURCE [DEST]\n\nOptions:\n\n--no-indent             Don\'t indent output\n--stylesheet=<URL or path>\n                        Specify a stylesheet URL to be included.\n                        (This option can be used multiple times)\n--script=<URL or path>  Specify a script URL to be included.\n                        (This option can be used multiple times)\n--script-defer=<URL or path>\n                        Specify a script URL with a defer attribute\n                        to be included in the output HTML file.\n                        (This option can be used multiple times)\n--script-async=<URL or path>\n                        Specify a script URL with a async attribute\n                        to be included in the output HTML file.\n                        (This option can be used multiple times)\n--html-tag-attr=<attribute>\n                        Specify a html tag attribute.\n                        (This option can be used multiple times)\n--template=<filename or text>\n                        Specify a filename or text to be used as the HTML5\n                        output template. The template must have the {head} and\n                        {body} placeholders. The "<html{html_attr}>"\n                        placeholder is recommended.\n--define=<identifier>   Define a case insensitive identifier to be used with\n                        ifdef and ifndef directives. There is no value\n                        associated with an identifier. (This option can be\n                        used multiple times)\n\n\nIf ``DEST`` is not provided, the output is send to ``stdout``.\n\n\nExample\n-------\n\nConsider a file called ``example.rst`` that contains:\n\n.. code-block:: rst\n\n    Title\n    =====\n\n    Some text and a target to `Title 2`_. **strong emphasis**:\n\n    * item 1\n    * item 2\n\n    Title 2\n    =======\n\n    .. parsed-literal::\n\n        Inline markup is supported, e.g. *emphasis*, **strong**, ``literal\n        text``,\n        _`hyperlink targets`, and `references <http://www.python.org/>`_\n\n\nThe command to produce an ``example.html`` output file is::\n\n    $ rst2html5 example.rst example.html\n\n\nThe HTML5 produced is clean and tidy:\n\n.. code-block:: html\n\n    <!DOCTYPE html>\n    <html>\n    <head>\n        <meta charset="utf-8" />\n    </head>\n    <body>\n        <section id="title">\n            <h1>Title</h1>\n            <p>Some text and a target to <a href="#title-2">Title 2</a>. <strong>strong emphasis</strong>:</p>\n            <ul>\n                <li>item 1</li>\n                <li>item 2</li>\n            </ul>\n        </section>\n        <section id="title-2">\n            <h1>Title 2</h1>\n            <pre>Inline markup is supported, e.g. <em>emphasis</em>, <strong>strong</strong>, <code>literal\n    text</code>,\n    <a id="hyperlink-targets">hyperlink targets</a>, and <a href="http://www.python.org/">references</a></pre>\n        </section>\n    </body>\n    </html>\n\n\nStylesheets and Scripts\n-----------------------\n\nNo stylesheets or scripts are spread over the HTML5 by default.\nHowever stylesheets and javascripts URLs or paths can be included through ``stylesheet`` and ``script`` options:\n\n.. parsed-literal::\n\n    $ rst2html5 example.rst \\\\\n    **--stylesheet** https://example.com/css/default.css \\\\\n    **--stylesheet-inline** css/simple.css \\\\\n    **--script** ``https://ajax.googleapis.com/ajax/libs/jquery/1.7.2/jquery.min.js``\n    **--script-defer** ``js/test1.js``\n    **--script-async** ``js/test2.js``\n\n\n.. code-block:: html\n\n    <!DOCTYPE html>\n    <html>\n    <head>\n        <meta charset="utf-8" />\n        <link rel="stylesheet" href="https://example.com/css/default.css" />\n        <style>h1 {font-size: 20em}\n    img.icon {\n        width: 48px;\n        height: 48px;\n    }\n    h2 {color: red}\n    </style>\n        <script src="https://ajax.googleapis.com/ajax/libs/jquery/1.7.2/jquery.min.js"></script>\n        <script src="js/test1.js" defer="defer"></script>\n        <script src="js/test2.js" async="async"></script>\n    </head>\n    ...\n\n\nHTML tag attributes can be included through ``html-tag-attr`` option:\n\n.. parsed-literal::\n\n    $ rst2html5 **--html-tag-attr** \'lang="pt-BR"\' example.rst\n\n.. code-block:: html\n\n    <!DOCTYPE html>\n    <html lang="pt-BR">\n    ...\n\n\nTemplates\n---------\n\nCustom HTML5 template via the :literal:`--template` option. Example:\n\n.. parsed-literal::\n\n    $ template=\'<!DOCTYPE html>\n    <html{html_attr}>\n    <head>{head}    <!-- custom links and scripts -->\n        <link href="css/default.css" rel="stylesheet" />\n        <link href="css/pygments.css" rel="stylesheet" />\n        <script src="http\\://code.jquery.com/jquery-latest.min.js"></script>\n    </head>\n    <body>{body}</body>\n    </html>\'\n\n    $ echo \'one line\' > example.rst\n\n    $ rst2html5 **--template "$template"** example.rst\n\n\n.. code-block:: html\n\n    <!DOCTYPE html>\n    <html>\n    <head>\n        <meta charset="utf-8" />\n        <!-- custom links and scripts -->\n        <link href="css/default.css" rel="stylesheet" />\n        <link href="css/pygments.css" rel="stylesheet" />\n        <script src="http://code.jquery.com/jquery-latest.min.js"></script>\n    </head>\n    <body>\n        <p>one line</p>\n    </body>\n    </html>\n\n\nNew Directives\n==============\n\n``define``, ``undef``, ``ifdef`` and ``ifndef``\n-----------------------------------------------\n\n:code:`rst2html5` provides some new directives: ``define``, ``undef``, ``ifdef`` and ``ifndef``,\nsimilar to those used in C++.\nThey allow to conditionally include (or not) some rst snippets:\n\n.. code-block:: rst\n\n    .. ifdef:: x\n\n        this line will be included if \'x\' was previously defined\n\n\nIn case of you check two or more identifiers,\nthere must be an operator (``[and | or]``) defined:\n\n.. code-block:: rst\n\n    .. ifdef:: x y z\n        :operator: or\n\n        This line will be included only if \'x\', \'y\' or \'z\' is defined.\n\n\n``stylesheet`` and ``script``\n-----------------------------\n\nFrom rst2html5 1.9, you can include stylesheets and scripts via directives inside a reStructuredText text:\n\n.. code-block:: rst\n\n    Just an ordinary paragraph.\n\n    .. stylesheet:: css/default.css\n    .. stylesheet:: https://pronus.io/css/standard.css\n\n    .. script:: http://code.jquery.com/jquery-latest.min.js\n    .. script:: slide.js\n        :defer:\n\n    .. script:: test/animations.js\n        :async:\n\n    Another paragraph\n\n\n.. code-block:: html\n\n    <!DOCTYPE html>\n    <html>\n    <head>\n        <meta charset="utf-8" />\n        <link href="css/default.css" rel="stylesheet" />\n        <link href="https://pronus.io/css/standard.css" rel="stylesheet" />\n        <script src="http://code.jquery.com/jquery-latest.min.js"></script>\n        <script src="slide.js" defer="defer"></script>\n        <script src="test/animations.js" async="async"></script>\n    </head>\n    <body>\n        <p>Just an ordinary paragraph.</p>\n        <p>Another paragraph</p>\n    </body>\n    </html>\n\n\n``template``\n------------\n\nThere also is a :code:`template` directive. The usage is:\n\n.. code-block:: rst\n\n    .. template:: filename\n\n    or\n\n    .. template::\n\n        template content here.\n\n\nNew Roles\n=========\n\n``:abbr:``\n----------\n\nFrom `MDN Web Docs <https://developer.mozilla.org/en-US/docs/Web/HTML/Element/abbr>`_:\n\n    The HTML Abbreviation element (:code:`<abbr>`) represents an abbreviation or acronym;\n    the optional title attribute can provide an expansion or description for the abbreviation.\n    If present, title must contain this full description and nothing else.\n\nTo create an abbreviation in ``rst2html5`` use the ``:abbr:`` role:\n\n.. code:: rst\n\n    * :abbr:`SPA (Single-Page Application)`\n    * :abbr:`ASGI (Asynchronous Server Gateway Interface)` is a spiritual successor to :abbr:`WSGI`\n    * :abbr:`WSGI (Web Server Gateway Interface)`\n\n\nResulting in:\n\n.. code:: html\n\n    <ul>\n        <li>\n            <abbr title="Single-Page Application">SPA</abbr>\n        </li>\n        <li>\n            <abbr title="Asynchronous Server Gateway Interface">ASGI</abbr>\n        is a spiritual successor to\n            <abbr>WSGI</abbr>\n        </li>\n        <li>\n            <abbr title="Web Server Gateway Interface">WSGI</abbr>\n        </li>\n    </ul>\n\n\nNote that if the abbreviation follows the pattern ``ABBR (Description for the abbreviation)``,\nthe description is extracted and becomes the ``title``.\n\n\nHow To Use rst2html5 Programmatically\n=====================================\n\nYou should use ``rst2html5.HTML5Writer`` with one of the ``publish_*` methods available in ``docutils.core``.\nIn the case that the input and output will be in memory,\n``publish_parts`` is the best fit:\n\n.. code:: python\n\n    from docutils.core import publish_parts\n\n    from rst2html5 import HTML5Writer\n\n    text = r\'\'\'The area of a circle is :math:`A_\\text{c} = (\\pi/4) d^2`.\n\n    .. math::\n\n        \\frac{ \\sum_{t=0}^{N}f(t,k) }{N}\n    \'\'\'\n\n    body = publish_parts(writer=HTML5Writer(), source=text)[\'body\']\n    print(body)\n\n\nResulting in:\n\n.. code:: html\n\n    <p>The area of a circle is\n        <span class="math">\\(A_\\text{c} = (\\pi/4) d^2\\)</span>\n    .</p>\n    <div class="math">\\(\\frac{ \\sum_{t=0}^{N}f(t,k) }{N}\\)</div>\n\n\n.. attention::\n\n    Version 2.0 renames the module ``rst2html5_`` back to ``rst2html5``\n    since the conflict with docutils installation is solved.\n    Importing ``rst2html5_.HTML5Writer`` still works though.\n    See the section "**Workaround to Conflicts with Docutils**"\n    on ``docs/design_notes.rst`` for more information.\n\n\nSee also: `The Docutils Publisher <https://docutils.sourceforge.io/docs/api/publisher.html>`_\n\n\nLinks\n=====\n\n* `Documentation <https://rst2html5.readthedocs.org/>`_\n* `Project page at Heptapod <https://foss.heptapod.net/doc-utils/rst2html5>`_\n',
    'author': 'André Felipe Dias',
    'author_email': 'andref.dias@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://foss.heptapod.net/doc-utils/rst2html5',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.6,<4',
}


setup(**setup_kwargs)
