# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['file_reader', 'file_reader.hosts']

package_data = \
{'': ['*'], 'file_reader': ['assets/*']}

install_requires = \
['requests>=2.24.0,<3.0.0']

extras_require = \
{'all': ['boto3>=1.16.0,<2.0.0',
         'smbprotocol>=1.2.0,<2.0.0',
         'paramiko>=2.7.2,<3.0.0',
         'pyarrow>=2.0.0,<3.0.0'],
 'hdfs': ['pyarrow>=2.0.0,<3.0.0'],
 's3': ['boto3>=1.16.0,<2.0.0'],
 'sftp': ['paramiko>=2.7.2,<3.0.0'],
 'smb': ['smbprotocol>=1.2.0,<2.0.0'],
 'ssh': ['paramiko>=2.7.2,<3.0.0']}

setup_kwargs = {
    'name': 'file-reader',
    'version': '0.2.1',
    'description': 'A tool for reading file from different sources with a single interface',
    'long_description': '# file_reader\n\n![Build](https://github.com/mrijken/file_reader/workflows/CI/badge.svg)\n![Hits](https://hitcounter.pythonanywhere.com/count/tag.svg?url=https%3A%2F%2Fgithub.com%2Fmrijken%2Ffile_reader)\n\n## Read almost all files from almost anywhere with one API\n\nUniform file reader for a lot of different file storages, like\n\n- local filesystem\n- http(s)\n- ftp(s)\n- smb\n- sftp\n\nImport dependencies for doctest\n\n    >>> import file_reader\n    >>> import file_reader.hosts\n    >>> import pytest\n\n## Usage\n\n### Construct a Host and a Path\n\nTo read a file, you have to instantiate a host (like FTPHost) and get a path from it\n\n    >>> host = file_reader.hosts.ftp.FTPHost(\'ftp.nluug.nl\')\n    >>> path = host / \'pub\' / \'os\' / \'Linux\' / \'distr\' / \'ubuntu-releases\' / \'FOOTER.html\'\n\n    or\n\n    >>> path = host.get_path("pub/os/Linux/distr/ubuntu-releases/FOOTER.html")\n\n    >>> path\n    Path(FTPHost(ftp.nluug.nl:21)/pub/os/Linux/distr/ubuntu-releases/FOOTER.html)\n\nYou can use that path to read the contents.\n\n\n### Accces the conents like a file\n\nUse it as a regular file object:\n\n- with or without a context managers\n\n    >>> with path.open() as f:\n    ...     f.read()\n    b\'\\n</div></body></html>\\n\'\n\n    >>> with path.open(\'b\') as f:\n    ...     f.read()\n    b\'\\n</div></body></html>\\n\'\n\n    >>> with path.open(\'t\') as f:\n    ...     f.read()\n    \'\\n</div></body></html>\\n\'\n\n### Access the contents directly\n\nRead the content direct from the path as text:\n\n    >>> path.read_text()\n    \'\\n</div></body></html>\\n\'\n\nor as binary:\n\n    >>> path.read_bytes()\n    b\'\\n</div></body></html>\\n\'\n\n### Create a Path from an url\n\nYou can construct a Host and Path by parsing an url, like:\n\n    >>> file_reader.base.Host.from_url("ftp://marc:secret@ftp.nluug.nl/pub/os/Linux/distr/ubuntu-releases/FOOTER.html")\n    Path(FTPHost(ftp.nluug.nl:21)/pub/os/Linux/distr/ubuntu-releases/FOOTER.html)\n\n    >>> file_reader.base.Host.from_url("http://marc:secret@nu.nl/robots.txt")\n    Path(HttpHost(nu.nl:80)/robots.txt)\n\n    >>> file_reader.base.Host.from_url("package://file_reader/__init__.py")\n    Path(Package(file_reader)/__init__.py)\n\n## Possible hosts\n\nFor every supported file location, a host is defined which knows how to access the file. The following\nhosts are supported\n\n### System\n\nThis is the local filesystem of your machine. It can access all files from all already mounted drives.\nIt will use the credentials of the user who is running the python process.\n\nYou can use a path relative to the working directory\n\n    >>> file_reader.hosts.local.LocalHost() / "file_reader" / "__init__.py"\n    Path(LocalHost(.)/file_reader/__init__.py)\n\nA path relative to the home dir of the current user can be used\n\n    >>> file_reader.hosts.local.LocalHost(home_dir=True) / ".ssh" / "id_rsa.pub"\n    Path(LocalHost(/home/...)/.ssh/id_rsa.pub)\n\nOr an absolute path can be used:\n\n    >>> file_reader.hosts.local.LocalHost(root=True) / "etc" / "hosts"\n    Path(LocalHost(/)/etc/hosts)\n\n\n### HTTP(S)\n\nVia the GET method a file from a HTTP(S) location will be get.\n\n    >>> path = file_reader.hosts.http.HttpHost("nu.nl") / "robots.txt"\n    >>> path\n    Path(HttpHost(nu.nl:80)/robots.txt)\n\n    >>> file_reader.base.Host.from_url("http://nu.nl/robots.txt")\n    Path(HttpHost(nu.nl:80)/robots.txt)\n\n    >>> "User-agent" in path.read_text()\n    True\n    >>> path = file_reader.hosts.http.HttpsHost("nu.nl") / "robots.txt"\n    >>> path\n    Path(HttpsHost(nu.nl:443)/robots.txt)\n    >>> "User-agent" in path.read_text()\n    True\n\nThe ssl certificate of sites will be checked unless you disable it.\n\n    >>> path = file_reader.hosts.http.HttpsHost("expired.badssl.com", verify_ssl=True)\n    >>> import requests.exceptions\n    >>> with pytest.raises(requests.exceptions.SSLError):\n    ...     path.read_text()\n\n    >>> path = file_reader.hosts.http.HttpsHost("expired.badssl.com", verify_ssl=False)\n    >>> "expired.<br>badssl.com" in path.read_text()\n    True\n\nYou can also specify an optional username and password for basic authentication.\nLater on, we will add other authentication providers, like certificate or (Authroization) header.\n\n    >>> path = file_reader.hosts.http.HttpsHost("nu.nl", auth=file_reader.auth.UsernamePassword("name", "secret")) / "robots.txt"\n\n\n### FTP(S)\n\nYou can access ftp(s) sites:\n\n    >>> path = file_reader.hosts.ftp.FTPHost("ftp.nluug.nl") / "pub" / "os" / "Linux" / "distr" / "ubuntu-releases" / "FOOTER.html"\n    >>> path\n    Path(FTPHost(ftp.nluug.nl:21)/pub/os/Linux/distr/ubuntu-releases/FOOTER.html)\n\n    >>> file_reader.base.Host.from_url("ftp://ftp.nluug.nl/pub/os/Linux/distr/ubuntu-releases/FOOTER.html")\n    Path(FTPHost(ftp.nluug.nl:21)/pub/os/Linux/distr/ubuntu-releases/FOOTER.html)\n\n    >>> "</div></body></html>" in path.read_text()\n    True\n\n    >>> path = file_reader.hosts.ftp.FTPHost("test.rebex.net", auth=file_reader.auth.UsernamePassword("demo", "password")) / "pub" / "example" / "readme.txt"\n    >>> "Welcome" in path.read_text()\n    True\n\n    >>> path = file_reader.hosts.ftp.FTPSHost("test.rebex.net", port=990, auth=file_reader.auth.UsernamePassword("demo", "password")) / "pub" / "example" / "readme.txt"\n    >>> path\n    Path(FTPSHost(test.rebex.net:990)/pub/example/readme.txt)\n\n    >>> file_reader.base.Host.from_url("ftps://test.rebex.net:990/pub/example/readme.txt")\n    Path(FTPSHost(test.rebex.net:990)/pub/example/readme.txt)\n\n    >>> "Welcome" in path.read_text()\n    True\n\n\n### SFTP\n\n    Note: Install with `pip install file_reader[ssh] to actually use SFTP\n\n    >>> file_reader.hosts.sftp.SFTPHost("test.rebex.net", auth=file_reader.auth.UsernamePassword("demo", "password"), auto_add_host_key=True) / "pub" / "example" / "readme.txt"\n    Path(SFTPHost(test.rebex.net:22)/pub/example/readme.txt)\n\n    >>> file_reader.base.Host.from_url("sftp://test.rebex.net/pub/example/readme.txt")\n    Path(SFTPHost(test.rebex.net:22)/pub/example/readme.txt)\n\n\n### SMB\n\n    Note: Install with `pip install file_reader[smb] to actually use SMB\n\n    >>> file_reader.hosts.smb.SmbHost("server") / "share" / "folder" / "readme.txt"\n    Path(SmbHost(server)/share/folder/readme.txt)\n\n    >>> file_reader.base.Host.from_url("smb://server/share/folder/readme.txt")\n    Path(SmbHost(server)/share/folder/readme.txt)\n\n\n### S3\n\n    Note: Install with `pip install file_reader[s3] to actually use S3\n\n    >>> file_reader.hosts.s3.S3Host("filereaderpublic") / "test_folder" / "test_folder_2" / "test.txt"\n    Path(S3Host(filereaderpublic)/test_folder/test_folder_2/test.txt)\n\n    >>> file_reader.base.Host.from_url("s3://filereaderpublic/test_folder/test_folder_2/test.txt")\n    Path(S3Host(filereaderpublic)/test_folder/test_folder_2/test.txt)\n\n### HDFS\n\n    Note: Install with `pip install file_reader[hdfs] to actually use HDFS\n\n    >>> file_reader.hosts.hdfs.HdfsHost("localhost") / "pub" / "example" / "readme.txt"\n    Path(HdfsHost(localhost)/pub/example/readme.txt)\n\n    >>> file_reader.base.Host.from_url("hdfs://localhost/pub/example/readme.txt")\n    Path(HdfsHost(localhost)/pub/example/readme.txt)\n\n\n### Package\n\nYou can load every file within an installed Python Package, whether it is a Python or distributed data file.\n\n    >>> path = file_reader.hosts.package.PythonPackage("file_reader") / "assets" / "test.txt"\n    >>> "test" in path.read_text()\n    True\n\n## Archives\n\nAlso files in archives can be accessed.\n\n    >>> path = file_reader.hosts.package.PythonPackage("file_reader") / "assets" / "test.zip" / "folder" / "file3.txt"\n    >>> "file3" in path.read_text()\n    True\n\nWhen a path element has one of the known archive extensions, it will be read as an archive:\n\n- .tar (tar)\n- .tgz (tar)\n- .tar.gz (tar)\n- .zip (zip)\n- .dep (zip)\n\n\n## FAQ\n\nQ Why do you only support reading files?\n\nA For a lot of use cases reading files is sufficient. When you want to do more, like making directories, adding files,\nremoving files and changing permissions, the differences between the hosts became very big. Too big to use just\none API.\n',
    'author': 'Marc Rijken',
    'author_email': 'marc@rijken.org',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/mrijken/file_reader',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
