"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseCluster = exports.DatabaseClusterBase = exports.EngineVersion = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("aws-cdk-lib/aws-ec2");
const iam = require("aws-cdk-lib/aws-iam");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const endpoint_1 = require("./endpoint");
const aws_neptune_1 = require("aws-cdk-lib/aws-neptune");
const subnet_group_1 = require("./subnet-group");
/**
 * Possible Instances Types to use in Neptune cluster
 * used for defining {@link DatabaseClusterProps.engineVersion}.
 */
class EngineVersion {
    /**
     * Constructor for specifying a custom engine version
     * @param version the engine version of Neptune
     */
    constructor(version) {
        this.version = version;
    }
}
exports.EngineVersion = EngineVersion;
_a = JSII_RTTI_SYMBOL_1;
EngineVersion[_a] = { fqn: "@aws-cdk/aws-neptune-alpha.EngineVersion", version: "2.20.0-alpha.0" };
/**
 * Neptune engine version 1.0.1.0
 */
EngineVersion.V1_0_1_0 = new EngineVersion('1.0.1.0');
/**
 * Neptune engine version 1.0.1.1
 */
EngineVersion.V1_0_1_1 = new EngineVersion('1.0.1.1');
/**
 * Neptune engine version 1.0.1.2
 */
EngineVersion.V1_0_1_2 = new EngineVersion('1.0.1.2');
/**
 * Neptune engine version 1.0.2.1
 */
EngineVersion.V1_0_2_1 = new EngineVersion('1.0.2.1');
/**
 * Neptune engine version 1.0.2.2
 */
EngineVersion.V1_0_2_2 = new EngineVersion('1.0.2.2');
/**
 * Neptune engine version 1.0.3.0
 */
EngineVersion.V1_0_3_0 = new EngineVersion('1.0.3.0');
/**
 * Neptune engine version 1.0.4.0
 */
EngineVersion.V1_0_4_0 = new EngineVersion('1.0.4.0');
/**
 * Neptune engine version 1.0.4.1
 */
EngineVersion.V1_0_4_1 = new EngineVersion('1.0.4.1');
/**
 * Neptune engine version 1.0.5.0
 */
EngineVersion.V1_0_5_0 = new EngineVersion('1.0.5.0');
/**
 * Neptune engine version 1.1.0.0
 */
EngineVersion.V1_1_0_0 = new EngineVersion('1.1.0.0');
/**
 * A new or imported database cluster.
 */
class DatabaseClusterBase extends aws_cdk_lib_1.Resource {
    /**
     * Import an existing DatabaseCluster from properties
     */
    static fromDatabaseClusterAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings._aws_cdk_aws_neptune_alpha_DatabaseClusterAttributes(attrs);
        class Import extends DatabaseClusterBase {
            constructor() {
                super(...arguments);
                this.defaultPort = ec2.Port.tcp(attrs.port);
                this.connections = new ec2.Connections({
                    securityGroups: [attrs.securityGroup],
                    defaultPort: this.defaultPort,
                });
                this.clusterIdentifier = attrs.clusterIdentifier;
                this.clusterResourceIdentifier = attrs.clusterResourceIdentifier;
                this.clusterEndpoint = new endpoint_1.Endpoint(attrs.clusterEndpointAddress, attrs.port);
                this.clusterReadEndpoint = new endpoint_1.Endpoint(attrs.readerEndpointAddress, attrs.port);
                this.enableIamAuthentication = true;
            }
        }
        return new Import(scope, id);
    }
    grantConnect(grantee) {
        if (this.enableIamAuthentication === false) {
            throw new Error('Cannot grant connect when IAM authentication is disabled');
        }
        this.enableIamAuthentication = true;
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['neptune-db:*'],
            resourceArns: [
                [
                    'arn',
                    aws_cdk_lib_1.Aws.PARTITION,
                    'neptune-db',
                    aws_cdk_lib_1.Aws.REGION,
                    aws_cdk_lib_1.Aws.ACCOUNT_ID,
                    `${this.clusterResourceIdentifier}/*`,
                ].join(':'),
            ],
        });
    }
}
exports.DatabaseClusterBase = DatabaseClusterBase;
_b = JSII_RTTI_SYMBOL_1;
DatabaseClusterBase[_b] = { fqn: "@aws-cdk/aws-neptune-alpha.DatabaseClusterBase", version: "2.20.0-alpha.0" };
/**
 * Create a clustered database with a given number of instances.
 *
 * @resource AWS::Neptune::DBCluster
 */
class DatabaseCluster extends DatabaseClusterBase {
    constructor(scope, id, props) {
        var _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p;
        super(scope, id);
        /**
         * Identifiers of the instance
         */
        this.instanceIdentifiers = [];
        /**
         * Endpoints which address each individual instance.
         */
        this.instanceEndpoints = [];
        jsiiDeprecationWarnings._aws_cdk_aws_neptune_alpha_DatabaseClusterProps(props);
        this.vpc = props.vpc;
        this.vpcSubnets = (_d = props.vpcSubnets) !== null && _d !== void 0 ? _d : { subnetType: ec2.SubnetType.PRIVATE };
        // Determine the subnet(s) to deploy the Neptune cluster to
        const { subnetIds, internetConnectivityEstablished } = this.vpc.selectSubnets(this.vpcSubnets);
        // Cannot test whether the subnets are in different AZs, but at least we can test the amount.
        if (subnetIds.length < 2) {
            throw new Error(`Cluster requires at least 2 subnets, got ${subnetIds.length}`);
        }
        this.subnetGroup = (_e = props.subnetGroup) !== null && _e !== void 0 ? _e : new subnet_group_1.SubnetGroup(this, 'Subnets', {
            description: `Subnets for ${id} database`,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            removalPolicy: props.removalPolicy === aws_cdk_lib_1.RemovalPolicy.RETAIN ? props.removalPolicy : undefined,
        });
        const securityGroups = (_f = props.securityGroups) !== null && _f !== void 0 ? _f : [
            new ec2.SecurityGroup(this, 'SecurityGroup', {
                description: 'Neptune security group',
                vpc: this.vpc,
            }),
        ];
        // Default to encrypted storage
        const storageEncrypted = (_g = props.storageEncrypted) !== null && _g !== void 0 ? _g : true;
        if (props.kmsKey && !storageEncrypted) {
            throw new Error('KMS key supplied but storageEncrypted is false');
        }
        const deletionProtection = (_h = props.deletionProtection) !== null && _h !== void 0 ? _h : (props.removalPolicy === aws_cdk_lib_1.RemovalPolicy.RETAIN ? true : undefined);
        this.enableIamAuthentication = props.iamAuthentication;
        // Create the Neptune cluster
        const cluster = new aws_neptune_1.CfnDBCluster(this, 'Resource', {
            // Basic
            engineVersion: (_j = props.engineVersion) === null || _j === void 0 ? void 0 : _j.version,
            dbClusterIdentifier: props.dbClusterName,
            dbSubnetGroupName: this.subnetGroup.subnetGroupName,
            port: props.port,
            vpcSecurityGroupIds: securityGroups.map(sg => sg.securityGroupId),
            dbClusterParameterGroupName: (_k = props.clusterParameterGroup) === null || _k === void 0 ? void 0 : _k.clusterParameterGroupName,
            deletionProtection: deletionProtection,
            associatedRoles: props.associatedRoles ? props.associatedRoles.map(role => ({ roleArn: role.roleArn })) : undefined,
            iamAuthEnabled: aws_cdk_lib_1.Lazy.any({ produce: () => this.enableIamAuthentication }),
            // Backup
            backupRetentionPeriod: (_l = props.backupRetention) === null || _l === void 0 ? void 0 : _l.toDays(),
            preferredBackupWindow: props.preferredBackupWindow,
            preferredMaintenanceWindow: props.preferredMaintenanceWindow,
            // Encryption
            kmsKeyId: (_m = props.kmsKey) === null || _m === void 0 ? void 0 : _m.keyArn,
            storageEncrypted,
        });
        cluster.applyRemovalPolicy(props.removalPolicy, {
            applyToUpdateReplacePolicy: true,
        });
        this.clusterIdentifier = cluster.ref;
        this.clusterResourceIdentifier = cluster.attrClusterResourceId;
        const port = aws_cdk_lib_1.Token.asNumber(cluster.attrPort);
        this.clusterEndpoint = new endpoint_1.Endpoint(cluster.attrEndpoint, port);
        this.clusterReadEndpoint = new endpoint_1.Endpoint(cluster.attrReadEndpoint, port);
        // Create the instances
        const instanceCount = (_o = props.instances) !== null && _o !== void 0 ? _o : DatabaseCluster.DEFAULT_NUM_INSTANCES;
        if (instanceCount < 1) {
            throw new Error('At least one instance is required');
        }
        for (let i = 0; i < instanceCount; i++) {
            const instanceIndex = i + 1;
            const instanceIdentifier = props.instanceIdentifierBase != null ? `${props.instanceIdentifierBase}${instanceIndex}`
                : props.dbClusterName != null ? `${props.dbClusterName}instance${instanceIndex}` : undefined;
            const instance = new aws_neptune_1.CfnDBInstance(this, `Instance${instanceIndex}`, {
                // Link to cluster
                dbClusterIdentifier: cluster.ref,
                dbInstanceIdentifier: instanceIdentifier,
                // Instance properties
                dbInstanceClass: props.instanceType._instanceType,
                dbParameterGroupName: (_p = props.parameterGroup) === null || _p === void 0 ? void 0 : _p.parameterGroupName,
                autoMinorVersionUpgrade: props.autoMinorVersionUpgrade === true,
            });
            // We must have a dependency on the NAT gateway provider here to create
            // things in the right order.
            instance.node.addDependency(internetConnectivityEstablished);
            instance.applyRemovalPolicy(props.removalPolicy, {
                applyToUpdateReplacePolicy: true,
            });
            this.instanceIdentifiers.push(instance.ref);
            this.instanceEndpoints.push(new endpoint_1.Endpoint(instance.attrEndpoint, port));
        }
        this.connections = new ec2.Connections({
            defaultPort: ec2.Port.tcp(port),
            securityGroups: securityGroups,
        });
    }
}
exports.DatabaseCluster = DatabaseCluster;
_c = JSII_RTTI_SYMBOL_1;
DatabaseCluster[_c] = { fqn: "@aws-cdk/aws-neptune-alpha.DatabaseCluster", version: "2.20.0-alpha.0" };
/**
 * The default number of instances in the Neptune cluster if none are
 * specified
 */
DatabaseCluster.DEFAULT_NUM_INSTANCES = 1;
//# sourceMappingURL=data:application/json;base64,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