import { Duration, IResource, Resource } from '@aws-cdk/core';
import { Construct } from 'constructs';
import { AccessLogFormat, IAccessLogDestination } from './access-log';
import { Deployment } from './deployment';
import { IRestApi } from './restapi';
/**
 * Represents an APIGateway Stage.
 *
 * @stability stable
 */
export interface IStage extends IResource {
    /**
     * Name of this stage.
     *
     * @stability stable
     * @attribute true
     */
    readonly stageName: string;
    /**
     * RestApi to which this stage is associated.
     *
     * @stability stable
     */
    readonly restApi: IRestApi;
}
/**
 * @stability stable
 */
export interface StageOptions extends MethodDeploymentOptions {
    /**
     * The name of the stage, which API Gateway uses as the first path segment in the invoked Uniform Resource Identifier (URI).
     *
     * @default - "prod"
     * @stability stable
     */
    readonly stageName?: string;
    /**
     * The CloudWatch Logs log group.
     *
     * @default - No destination
     * @stability stable
     */
    readonly accessLogDestination?: IAccessLogDestination;
    /**
     * A single line format of access logs of data, as specified by selected $content variables.
     *
     * The format must include at least `AccessLogFormat.contextRequestId()`.
     *
     * @default - Common Log Format
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-mapping-template-reference.html#context-variable-reference
     * @stability stable
     */
    readonly accessLogFormat?: AccessLogFormat;
    /**
     * Specifies whether Amazon X-Ray tracing is enabled for this method.
     *
     * @default false
     * @stability stable
     */
    readonly tracingEnabled?: boolean;
    /**
     * Indicates whether cache clustering is enabled for the stage.
     *
     * @default - Disabled for the stage.
     * @stability stable
     */
    readonly cacheClusterEnabled?: boolean;
    /**
     * The stage's cache cluster size.
     *
     * @default 0.5
     * @stability stable
     */
    readonly cacheClusterSize?: string;
    /**
     * The identifier of the client certificate that API Gateway uses to call your integration endpoints in the stage.
     *
     * @default - None.
     * @stability stable
     */
    readonly clientCertificateId?: string;
    /**
     * A description of the purpose of the stage.
     *
     * @default - No description.
     * @stability stable
     */
    readonly description?: string;
    /**
     * The version identifier of the API documentation snapshot.
     *
     * @default - No documentation version.
     * @stability stable
     */
    readonly documentationVersion?: string;
    /**
     * A map that defines the stage variables.
     *
     * Variable names must consist of
     * alphanumeric characters, and the values must match the following regular
     * expression: [A-Za-z0-9-._~:/?#&amp;=,]+.
     *
     * @default - No stage variables.
     * @stability stable
     */
    readonly variables?: {
        [key: string]: string;
    };
    /**
     * Method deployment options for specific resources/methods.
     *
     * These will
     * override common options defined in `StageOptions#methodOptions`.
     *
     * @default - Common options will be used.
     * @stability stable
     */
    readonly methodOptions?: {
        [path: string]: MethodDeploymentOptions;
    };
}
/**
 * @stability stable
 */
export interface StageProps extends StageOptions {
    /**
     * The deployment that this stage points to [disable-awslint:ref-via-interface].
     *
     * @stability stable
     */
    readonly deployment: Deployment;
}
/**
 * @stability stable
 */
export declare enum MethodLoggingLevel {
    /**
     * @stability stable
     */
    OFF = "OFF",
    /**
     * @stability stable
     */
    ERROR = "ERROR",
    /**
     * @stability stable
     */
    INFO = "INFO"
}
/**
 * @stability stable
 */
export interface MethodDeploymentOptions {
    /**
     * Specifies whether Amazon CloudWatch metrics are enabled for this method.
     *
     * @default false
     * @stability stable
     */
    readonly metricsEnabled?: boolean;
    /**
     * Specifies the logging level for this method, which effects the log entries pushed to Amazon CloudWatch Logs.
     *
     * @default - Off
     * @stability stable
     */
    readonly loggingLevel?: MethodLoggingLevel;
    /**
     * Specifies whether data trace logging is enabled for this method, which effects the log entries pushed to Amazon CloudWatch Logs.
     *
     * @default false
     * @stability stable
     */
    readonly dataTraceEnabled?: boolean;
    /**
     * Specifies the throttling burst limit.
     *
     * The total rate of all requests in your AWS account is limited to 5,000 requests.
     *
     * @default - No additional restriction.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-request-throttling.html
     * @stability stable
     */
    readonly throttlingBurstLimit?: number;
    /**
     * Specifies the throttling rate limit.
     *
     * The total rate of all requests in your AWS account is limited to 10,000 requests per second (rps).
     *
     * @default - No additional restriction.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-request-throttling.html
     * @stability stable
     */
    readonly throttlingRateLimit?: number;
    /**
     * Specifies whether responses should be cached and returned for requests.
     *
     * A
     * cache cluster must be enabled on the stage for responses to be cached.
     *
     * @default - Caching is Disabled.
     * @stability stable
     */
    readonly cachingEnabled?: boolean;
    /**
     * Specifies the time to live (TTL), in seconds, for cached responses.
     *
     * The
     * higher the TTL, the longer the response will be cached.
     *
     * @default Duration.minutes(5)
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-caching.html
     * @stability stable
     */
    readonly cacheTtl?: Duration;
    /**
     * Indicates whether the cached responses are encrypted.
     *
     * @default false
     * @stability stable
     */
    readonly cacheDataEncrypted?: boolean;
}
/**
 * @stability stable
 */
export declare class Stage extends Resource implements IStage {
    /**
     * Name of this stage.
     *
     * @stability stable
     */
    readonly stageName: string;
    /**
     * RestApi to which this stage is associated.
     *
     * @stability stable
     */
    readonly restApi: IRestApi;
    private enableCacheCluster?;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: StageProps);
    /**
     * Returns the invoke URL for a certain path.
     *
     * @param path The resource path.
     * @stability stable
     */
    urlForPath(path?: string): string;
    private renderMethodSettings;
}
