"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.findAlarmThresholds = exports.normalizeIntervals = void 0;
/**
 * Normalize the given interval set to cover the complete number line and make sure it has at most one gap
 */
function normalizeIntervals(intervals, changesAreAbsolute) {
    // Make intervals a complete numberline
    const full = orderAndCompleteIntervals(intervals);
    // Add 'undefined's in uncovered areas of the number line
    makeGapsUndefined(full);
    // In case of relative changes, treat 0-change also as 'undefined' (= no change action)
    if (!changesAreAbsolute) {
        makeZerosUndefined(full);
    }
    // Combine adjacent undefines and make sure there's at most one of them
    combineUndefineds(full);
    validateAtMostOneUndefined(full);
    return full;
}
exports.normalizeIntervals = normalizeIntervals;
/**
 * Completely order scaling intervals, making their lower and upper bounds concrete.
 */
function orderAndCompleteIntervals(intervals) {
    if (intervals.length < 2) {
        throw new Error('Require at least 2 intervals');
    }
    for (const interval of intervals) {
        if (interval.lower === undefined && interval.upper === undefined) {
            throw new Error(`Must supply at least one of 'upper' or 'lower', got: ${JSON.stringify(interval)}`);
        }
    }
    // Make a copy
    intervals = intervals.map(x => ({ ...x }));
    // Sort by whatever number we have for each interval
    intervals.sort(comparatorFromKey((x) => x.lower !== undefined ? x.lower : x.upper));
    // Propagate boundaries until no more change
    while (propagateBounds(intervals)) { /* Repeat */ }
    const lastIndex = intervals.length - 1;
    // Validate that no intervals have undefined bounds now, which must mean they're complete.
    if (intervals[0].lower === undefined) {
        intervals[0] = { ...intervals[0], lower: 0 };
    }
    if (intervals[lastIndex].upper === undefined) {
        intervals[lastIndex] = { ...intervals[lastIndex], upper: Infinity };
    }
    for (const interval of intervals) {
        if (interval.lower === undefined || interval.upper === undefined) {
            throw new Error(`Could not determine the lower and upper bounds for ${JSON.stringify(interval)}`);
        }
    }
    const completeIntervals = intervals;
    // Validate that we have nonoverlapping intervals now.
    for (let i = 0; i < completeIntervals.length - 1; i++) {
        if (overlap(completeIntervals[i], completeIntervals[i + 1])) {
            throw new Error(`Two intervals overlap: ${JSON.stringify(completeIntervals[i])} and ${JSON.stringify(completeIntervals[i + 1])}`);
        }
    }
    // Fill up the gaps
    return completeIntervals;
}
/**
 * Make the intervals cover the complete number line
 *
 * This entails adding intervals with an 'undefined' change to fill up the gaps.
 *
 * Since metrics have a halfopen interval, the first one will get a lower bound
 * of 0, the last one will get an upper bound of +Infinity.
 *
 * In case of absolute adjustments, the lower number of the adjacent bound will
 * be used, which means conservative change. In case of relative adjustments,
 * we'll use relative adjusment 0 (which means no change).
 */
function makeGapsUndefined(intervals) {
    // Add edge intervals if necessary, but only for relative adjustments. Since we're
    // going to make scaling intervals extend all the way out to infinity on either side,
    // the result is the same for absolute adjustments anyway.
    if (intervals[0].lower !== 0) {
        intervals.splice(0, 1, {
            lower: 0,
            upper: intervals[0].lower,
            change: undefined,
        });
    }
    if (last(intervals).upper !== Infinity) {
        intervals.push({
            lower: last(intervals).upper,
            upper: Infinity,
            change: undefined,
        });
    }
    let i = 1;
    while (i < intervals.length) {
        if (intervals[i - 1].upper < intervals[i].lower) {
            intervals.splice(i, 0, {
                lower: intervals[i - 1].upper,
                upper: intervals[i].lower,
                change: undefined,
            });
        }
        else {
            i++;
        }
    }
}
/**
 * Turn zero changes into undefined, in-place
 */
function makeZerosUndefined(intervals) {
    for (let i = 0; i < intervals.length; ++i) {
        const interval = intervals[i];
        if (interval.change === 0) {
            intervals[i] = { ...interval, change: undefined };
        }
    }
}
/**
 * If there are adjacent "undefined" intervals, combine them
 */
function combineUndefineds(intervals) {
    let i = 0;
    while (i < intervals.length - 1) {
        if (intervals[i].change === undefined && intervals[i + 1].change === undefined) {
            intervals[i] = { ...intervals[i], upper: intervals[i + 1].upper };
            intervals.splice(i + 1, 1);
        }
        else {
            i++;
        }
    }
}
function validateAtMostOneUndefined(intervals) {
    const undef = intervals.filter(x => x.change === undefined);
    if (undef.length > 1) {
        throw new Error(`Can have at most one no-change interval, got ${JSON.stringify(undef)}`);
    }
}
function comparatorFromKey(keyFn) {
    return (a, b) => {
        const keyA = keyFn(a);
        const keyB = keyFn(b);
        if (keyA < keyB) {
            return -1;
        }
        if (keyA === keyB) {
            return 0;
        }
        return 1;
    };
}
function propagateBounds(intervals) {
    let ret = false;
    // Propagate upper bounds upwards
    for (let i = 0; i < intervals.length - 1; i++) {
        if (intervals[i].upper !== undefined && intervals[i + 1].lower === undefined) {
            intervals[i + 1] = { ...intervals[i + 1], lower: intervals[i].upper };
            ret = true;
        }
    }
    // Propagate lower bounds downwards
    for (let i = intervals.length - 1; i >= 1; i--) {
        if (intervals[i].lower !== undefined && intervals[i - 1].upper === undefined) {
            intervals[i - 1] = { ...intervals[i - 1], upper: intervals[i].lower };
            ret = true;
        }
    }
    return ret;
}
/**
 * Whether two intervals overlap
 */
function overlap(a, b) {
    return a.lower < b.upper && a.upper > b.lower;
}
function last(xs) {
    return xs[xs.length - 1];
}
/**
 * Locate the intervals that should have the alarm thresholds, by index.
 *
 * Pick the intervals on either side of the singleton "undefined" interval, or
 * pick the middle interval if there's no such interval.
 */
function findAlarmThresholds(intervals) {
    const gapIndex = intervals.findIndex(x => x.change === undefined);
    if (gapIndex !== -1) {
        return {
            lowerAlarmIntervalIndex: gapIndex > 0 ? gapIndex - 1 : undefined,
            upperAlarmIntervalIndex: gapIndex < intervals.length - 1 ? gapIndex + 1 : undefined,
        };
    }
    if (intervals.length === 1) {
        return { upperAlarmIntervalIndex: 0 };
    }
    const middleIndex = Math.floor(intervals.length / 2);
    return {
        lowerAlarmIntervalIndex: middleIndex - 1,
        upperAlarmIntervalIndex: middleIndex,
    };
}
exports.findAlarmThresholds = findAlarmThresholds;
//# sourceMappingURL=data:application/json;base64,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