"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateArbitraryIntervals = void 0;
/**
 * Arbitrary (valid) array of intervals
 *
 * There are many invalid combinations of interval arrays, so we have
 * to be very specific about generating arrays that are valid. We do this
 * by taking a full, valid interval schedule and progressively stripping parts
 * away from it.
 *
 * Some of the changes may change its meaning, but we take care to never leave
 * a schedule with insufficient information so that the parser will error out.
 */
function generateArbitraryIntervals(mrng) {
    const ret = new Array();
    const absolute = mrng.nextBoolean();
    // Ascending or descending scaling
    const factor = (mrng.nextBoolean() ? 1 : -1) * (absolute ? 10 : 1);
    const bias = absolute ? 50 : 0;
    // Begin with a full schedule
    ret.push({ lower: 0, upper: 10, change: -2 * factor + bias });
    ret.push({ lower: 10, upper: 20, change: -1 * factor + bias });
    ret.push({ lower: 20, upper: 60, change: 0 + bias });
    ret.push({ lower: 60, upper: 80, change: 0 + bias });
    ret.push({ lower: 80, upper: 90, change: 1 * factor + bias });
    ret.push({ lower: 90, upper: Infinity, change: 2 * factor + bias });
    // Take away parts from this. First we see if we do something to the 0-change alarms.
    // The actions can be: remove it OR turn it into a regular change value.
    const noChanges = ret.filter(x => x.change === bias);
    if (!absolute) {
        if (mrng.nextBoolean()) {
            if (mrng.nextBoolean()) {
                ret.splice(ret.indexOf(noChanges[0]), 1);
            }
            else {
                noChanges[0] = { ...noChanges[0], change: -1 * factor + bias };
            }
        }
        if (mrng.nextBoolean()) {
            if (mrng.nextBoolean()) {
                ret.splice(ret.indexOf(noChanges[1]), 1);
            }
            else {
                noChanges[1] = { ...noChanges[1], change: 1 * factor + bias };
            }
        }
    }
    else {
        // In absolute mode both have to get the same treatment at the same time
        // otherwise we'll end up with a timeline with two gaps
        if (mrng.nextBoolean()) {
            ret.splice(ret.indexOf(noChanges[0]), 1);
            ret.splice(ret.indexOf(noChanges[1]), 1);
        }
        else {
            noChanges[0] = { ...noChanges[0], change: -1 * factor + bias };
            noChanges[1] = { ...noChanges[1], change: 1 * factor + bias };
        }
    }
    // We might also take away either the bottom or the upper half
    if (mrng.nextInt(0, 2) === 0) {
        const signToStrip = mrng.nextBoolean() ? -1 : 1;
        let ix = ret.findIndex(x => Math.sign(x.change - bias) === signToStrip);
        while (ix >= 0) {
            ret.splice(ix, 1);
            ix = ret.findIndex(x => Math.sign(x.change - bias) === signToStrip);
        }
    }
    // Then we're going to arbitrarily get rid of bounds in the most naive way possible
    const iterations = mrng.nextInt(0, 10);
    for (let iter = 0; iter < iterations; iter++) {
        const i = mrng.nextInt(0, ret.length - 1);
        if (mrng.nextBoolean()) {
            // scrap lower bound
            // okay if current interval has an upper bound AND the preceding interval has an upper bound
            if (ret[i].upper !== undefined && (i === 0 || ret[i - 1].upper !== undefined)) {
                ret[i] = { ...ret[i], lower: undefined };
            }
        }
        else {
            // scrap upper bound
            // okay if current interval has a lower bound AND the succeeding interval has a lower bound
            if (ret[i].lower !== undefined && (i === ret.length - 1 || ret[i + 1].lower !== undefined)) {
                ret[i] = { ...ret[i], upper: undefined };
            }
        }
    }
    return { absolute, intervals: ret };
}
exports.generateArbitraryIntervals = generateArbitraryIntervals;
//# sourceMappingURL=data:application/json;base64,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