"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProxyResource = exports.Resource = exports.ResourceBase = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("./apigateway.generated");
const cors_1 = require("./cors");
const integrations_1 = require("./integrations");
const method_1 = require("./method");
/**
 * @stability stable
 */
class ResourceBase extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id) {
        super(scope, id);
        this.children = {};
    }
    /**
     * Defines a new child resource where this resource is the parent.
     *
     * @stability stable
     */
    addResource(pathPart, options) {
        return new Resource(this, pathPart, { parent: this, pathPart, ...options });
    }
    /**
     * Defines a new method for this resource.
     *
     * @stability stable
     */
    addMethod(httpMethod, integration, options) {
        return new method_1.Method(this, httpMethod, { resource: this, httpMethod, integration, options });
    }
    /**
     * Adds a greedy proxy resource ("{proxy+}") and an ANY method to this route.
     *
     * @stability stable
     */
    addProxy(options) {
        return new ProxyResource(this, '{proxy+}', { parent: this, ...options });
    }
    /**
     * Adds an OPTIONS method to this resource which responds to Cross-Origin Resource Sharing (CORS) preflight requests.
     *
     * Cross-Origin Resource Sharing (CORS) is a mechanism that uses additional
     * HTTP headers to tell browsers to give a web application running at one
     * origin, access to selected resources from a different origin. A web
     * application executes a cross-origin HTTP request when it requests a
     * resource that has a different origin (domain, protocol, or port) from its
     * own.
     *
     * @stability stable
     */
    addCorsPreflight(options) {
        var _d;
        const headers = {};
        //
        // Access-Control-Allow-Headers
        const allowHeaders = options.allowHeaders || cors_1.Cors.DEFAULT_HEADERS;
        headers['Access-Control-Allow-Headers'] = `'${allowHeaders.join(',')}'`;
        //
        // Access-Control-Allow-Origin
        if (options.allowOrigins.length === 0) {
            throw new Error('allowOrigins must contain at least one origin');
        }
        if (options.allowOrigins.includes('*') && options.allowOrigins.length > 1) {
            throw new Error(`Invalid "allowOrigins" - cannot mix "*" with specific origins: ${options.allowOrigins.join(',')}`);
        }
        // we use the first origin here and if there are more origins in the list, we
        // will match against them in the response velocity template
        const initialOrigin = options.allowOrigins[0];
        headers['Access-Control-Allow-Origin'] = `'${initialOrigin}'`;
        // the "Vary" header is required if we allow a specific origin
        // https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Origin#CORS_and_caching
        if (initialOrigin !== '*') {
            headers.Vary = '\'Origin\'';
        }
        //
        // Access-Control-Allow-Methods
        let allowMethods = options.allowMethods || cors_1.Cors.ALL_METHODS;
        if (allowMethods.includes('ANY')) {
            if (allowMethods.length > 1) {
                throw new Error(`ANY cannot be used with any other method. Received: ${allowMethods.join(',')}`);
            }
            allowMethods = cors_1.Cors.ALL_METHODS;
        }
        headers['Access-Control-Allow-Methods'] = `'${allowMethods.join(',')}'`;
        //
        // Access-Control-Allow-Credentials
        if (options.allowCredentials) {
            headers['Access-Control-Allow-Credentials'] = '\'true\'';
        }
        //
        // Access-Control-Max-Age
        let maxAgeSeconds;
        if (options.maxAge && options.disableCache) {
            throw new Error('The options "maxAge" and "disableCache" are mutually exclusive');
        }
        if (options.maxAge) {
            maxAgeSeconds = options.maxAge.toSeconds();
        }
        if (options.disableCache) {
            maxAgeSeconds = -1;
        }
        if (maxAgeSeconds) {
            headers['Access-Control-Max-Age'] = `'${maxAgeSeconds}'`;
        }
        //
        // Access-Control-Expose-Headers
        //
        if (options.exposeHeaders) {
            headers['Access-Control-Expose-Headers'] = `'${options.exposeHeaders.join(',')}'`;
        }
        //
        // statusCode
        const statusCode = (_d = options.statusCode) !== null && _d !== void 0 ? _d : 204;
        //
        // prepare responseParams
        const integrationResponseParams = {};
        const methodResponseParams = {};
        for (const [name, value] of Object.entries(headers)) {
            const key = `method.response.header.${name}`;
            integrationResponseParams[key] = value;
            methodResponseParams[key] = true;
        }
        return this.addMethod('OPTIONS', new integrations_1.MockIntegration({
            requestTemplates: { 'application/json': '{ statusCode: 200 }' },
            integrationResponses: [
                { statusCode: `${statusCode}`, responseParameters: integrationResponseParams, responseTemplates: renderResponseTemplate() },
            ],
        }), {
            methodResponses: [
                { statusCode: `${statusCode}`, responseParameters: methodResponseParams },
            ],
        });
        // renders the response template to match all possible origins (if we have more than one)
        function renderResponseTemplate() {
            const origins = options.allowOrigins.slice(1);
            if (origins.length === 0) {
                return undefined;
            }
            const template = new Array();
            template.push('#set($origin = $input.params("Origin"))');
            template.push('#if($origin == "") #set($origin = $input.params("origin")) #end');
            const condition = origins.map(o => `$origin.matches("${o}")`).join(' || ');
            template.push(`#if(${condition})`);
            template.push('  #set($context.responseOverride.header.Access-Control-Allow-Origin = $origin)');
            template.push('#end');
            return {
                'application/json': template.join('\n'),
            };
        }
    }
    /**
     * Retrieves a child resource by path part.
     *
     * @stability stable
     */
    getResource(pathPart) {
        return this.children[pathPart];
    }
    /**
     * @internal
     */
    _trackChild(pathPart, resource) {
        this.children[pathPart] = resource;
    }
    /**
     * Gets or create all resources leading up to the specified path.
     *
     * - Path may only start with "/" if this method is called on the root resource.
     * - All resources are created using default options.
     *
     * @stability stable
     */
    resourceForPath(path) {
        if (!path) {
            return this;
        }
        if (path.startsWith('/')) {
            if (this.path !== '/') {
                throw new Error(`Path may start with "/" only for the resource, but we are at: ${this.path}`);
            }
            // trim trailing "/"
            return this.resourceForPath(path.substr(1));
        }
        const parts = path.split('/');
        const next = parts.shift();
        if (!next || next === '') {
            throw new Error('resourceForPath cannot be called with an empty path');
        }
        let resource = this.getResource(next);
        if (!resource) {
            resource = this.addResource(next);
        }
        return resource.resourceForPath(parts.join('/'));
    }
    /**
     * @deprecated - Throws error in some use cases that have been enabled since this deprecation notice. Use `RestApi.urlForPath()` instead.
     */
    get url() {
        return this.restApi.urlForPath(this.path);
    }
}
exports.ResourceBase = ResourceBase;
_a = JSII_RTTI_SYMBOL_1;
ResourceBase[_a] = { fqn: "@aws-cdk/aws-apigateway.ResourceBase", version: "1.101.0" };
/**
 * @stability stable
 */
class Resource extends ResourceBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        validateResourcePathPart(props.pathPart);
        this.parentResource = props.parent;
        if (props.parent instanceof ResourceBase) {
            props.parent._trackChild(props.pathPart, this);
        }
        const resourceProps = {
            restApiId: props.parent.api.restApiId,
            parentId: props.parent.resourceId,
            pathPart: props.pathPart,
        };
        const resource = new apigateway_generated_1.CfnResource(this, 'Resource', resourceProps);
        this.resourceId = resource.ref;
        this.api = props.parent.api;
        // render resource path (special case for root)
        this.path = props.parent.path;
        if (!this.path.endsWith('/')) {
            this.path += '/';
        }
        this.path += props.pathPart;
        const deployment = props.parent.api.latestDeployment;
        if (deployment) {
            deployment.node.addDependency(resource);
            deployment.addToLogicalId({ resource: resourceProps });
        }
        // setup defaults based on properties and inherit from parent. method defaults
        // are inherited per property, so children can override piecemeal.
        this.defaultIntegration = props.defaultIntegration || props.parent.defaultIntegration;
        this.defaultMethodOptions = {
            ...props.parent.defaultMethodOptions,
            ...props.defaultMethodOptions,
        };
        this.defaultCorsPreflightOptions = props.defaultCorsPreflightOptions || props.parent.defaultCorsPreflightOptions;
        if (this.defaultCorsPreflightOptions) {
            this.addCorsPreflight(this.defaultCorsPreflightOptions);
        }
    }
    /**
     * Import an existing resource.
     *
     * @stability stable
     */
    static fromResourceAttributes(scope, id, attrs) {
        class Import extends ResourceBase {
            constructor() {
                super(...arguments);
                this.api = attrs.restApi;
                this.resourceId = attrs.resourceId;
                this.path = attrs.path;
                this.defaultIntegration = undefined;
                this.defaultMethodOptions = undefined;
                this.defaultCorsPreflightOptions = undefined;
            }
            get parentResource() {
                throw new Error('parentResource is not configured for imported resource.');
            }
            get restApi() {
                throw new Error('restApi is not configured for imported resource.');
            }
        }
        return new Import(scope, id);
    }
    /**
     * (deprecated) The RestApi associated with this Resource.
     *
     * @deprecated - Throws an error if this Resource is not associated with an instance of `RestApi`. Use `api` instead.
     */
    get restApi() {
        if (!this.parentResource) {
            throw new Error('parentResource was unexpectedly not defined');
        }
        return this.parentResource.restApi;
    }
}
exports.Resource = Resource;
_b = JSII_RTTI_SYMBOL_1;
Resource[_b] = { fqn: "@aws-cdk/aws-apigateway.Resource", version: "1.101.0" };
/**
 * Defines a {proxy+} greedy resource and an ANY method on a route.
 *
 * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-set-up-simple-proxy.html
 * @stability stable
 */
class ProxyResource extends Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _d;
        super(scope, id, {
            parent: props.parent,
            pathPart: '{proxy+}',
            defaultIntegration: props.defaultIntegration,
            defaultMethodOptions: props.defaultMethodOptions,
        });
        const anyMethod = (_d = props.anyMethod) !== null && _d !== void 0 ? _d : true;
        if (anyMethod) {
            this.anyMethod = this.addMethod('ANY');
        }
    }
    /**
     * Defines a new method for this resource.
     *
     * @stability stable
     */
    addMethod(httpMethod, integration, options) {
        // In case this proxy is mounted under the root, also add this method to
        // the root so that empty paths are proxied as well.
        if (this.parentResource && this.parentResource.path === '/') {
            // skip if the root resource already has this method defined
            if (!(this.parentResource.node.tryFindChild(httpMethod) instanceof method_1.Method)) {
                this.parentResource.addMethod(httpMethod, integration, options);
            }
        }
        return super.addMethod(httpMethod, integration, options);
    }
}
exports.ProxyResource = ProxyResource;
_c = JSII_RTTI_SYMBOL_1;
ProxyResource[_c] = { fqn: "@aws-cdk/aws-apigateway.ProxyResource", version: "1.101.0" };
function validateResourcePathPart(part) {
    // strip {} which indicate this is a parameter
    if (part.startsWith('{') && part.endsWith('}')) {
        part = part.substr(1, part.length - 2);
        // proxy resources are allowed to end with a '+'
        if (part.endsWith('+')) {
            part = part.substr(0, part.length - 1);
        }
    }
    if (!/^[a-zA-Z0-9\.\_\-]+$/.test(part)) {
        throw new Error(`Resource's path part only allow [a-zA-Z0-9._-], an optional trailing '+'
      and curly braces at the beginning and the end: ${part}`);
    }
}
//# sourceMappingURL=data:application/json;base64,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