# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['django_pint_field', 'django_pint_field.migrations']

package_data = \
{'': ['*']}

install_requires = \
['django-cachalot>=2.5.2,<3.0.0',
 'django-extensions>=3.2.1,<4.0.0',
 'django-redis>=5.2.0,<6.0.0',
 'django>=4.1.4,<5.0.0',
 'pint>=0.20.1,<0.21.0',
 'psycopg2-binary>=2.9.5,<3.0.0',
 'pydantic>=1.10.4,<2.0.0',
 'pytest>=7.2.0,<8.0.0',
 'redis[hiredis]>=4.4,<5.0',
 'werkzeug>=2.2.2,<3.0.0']

setup_kwargs = {
    'name': 'django-pint-field',
    'version': '0.2.3',
    'description': 'Pint Fields for Django and Postgres',
    'long_description': '# django-pint-field\n\nUse [pint](https://pint.readthedocs.io/en/stable/) with Django\'s ORM.\n\nIf you want to store quantities (1 gram, 3 miles, 8.120391 angstroms, etc) in a model, edit them in forms, and have the ability to convert to other quantities in your django projects, this is the package for you!\n\nThis package is modified from the fantastic [django-pint](https://github.com/CarliJoy/django-pint) with different goals. Unlike django-pint, in this project we use a composite Postgres field to store both the magnitude and the user\'s desired units, along with the equivalent value in base units. This third piece of date - the base units - makes it possible to conduct lookups comparing one instance that might be specified in "grams" with another that may be specified in "pounds", but display each instance in the units that the user desires. The units your users want to use are the units they see, while still allowing accurate comparisons of one quantity to another.\n\nFor this reason, the project only works with Postgresql databases.\n\n\n## Install\n\n`pip install django_pint_field`\n\nAdd `"django_pint_field",` to your list of installed apps.\n\nRun `python manage.py migrate django_pint_field`\n\n\n## Usage\n\n```python\nfrom decimal import Decimal\nfrom django_pint_field.units import ureg\nQuantity = ureg.Quantity\n\n# Start by creating a few Pint Quantity objects\nextra_small = Quantity(1 * ureg.gram)\nsmall = Quantity(10 * ureg.gram)\nmedium = Quantity(100 * ureg.gram)\nlarge = Quantity(1000 * ureg.gram)\nextra_large = Quantity(10000 * ureg.gram)\n\n# Create a couple objects\nIntegerPintFieldSaveModel.objects.create(name="small", weight=small)\nIntegerPintFieldSaveModel.objects.create(name="large", weight=large)\n\nIn [1]: IntegerPintFieldSaveModel.objects.first()\nOut[1]: <IntegerPintFieldSaveModel: small>\n\nIn [2]: IntegerPintFieldSaveModel.objects.first().weight\nOut[2]: 10 <Unit(\'gram\')>\n\nIn [3]: IntegerPintFieldSaveModel.objects.first().weight.magnitude\nOut[3]: 10\n\nIn [4]: IntegerPintFieldSaveModel.objects.first().weight.units\nOut[4]: <Unit(\'gram\')>\n```\n\n\n## Valid Lookups\n\nOther lookups will be added in the future. Currently available are:\n\n- exact\n- gt\n- gte\n- lt\n- lte\n- range\n- isnull\n\n```python\n# Perform some queries\nIntegerPintFieldSaveModel.objects.filter(weight__gt=medium)\n<QuerySet [<IntegerPintFieldSaveModel: large>]>\n\nIntegerPintFieldSaveModel.objects.filter(weight__gt=extra_small)\n<QuerySet [<IntegerPintFieldSaveModel: small>, <IntegerPintFieldSaveModel: large>]>\n\nIntegerPintFieldSaveModel.objects.filter(weight__gte=small)\n<QuerySet [<IntegerPintFieldSaveModel: small>, <IntegerPintFieldSaveModel: large>]>\n\nIntegerPintFieldSaveModel.objects.filter(weight__range=(small, medium))\n<QuerySet [<IntegerPintFieldSaveModel: small>]>\n```\n\n\n## Aggregates\n\nA number of aggregates have been implemented for the Django Pint Fields. Functionally they perform for Pint Fields the same way django\'s default aggregates work for other field types, and each is prepended with "Pint". The aggregates include:\n\n- PintAvg\n- PintCount\n- PintMax\n- PintMin\n- PintStdDev\n- PintSum\n- PintVariance\n\n```python\nfrom django_pint_field.aggregates import PintAvg, PintCount, PintMax, PintMin, PintStdDev, PintSum, PintVariance\n\n# Perform some queries\nIntegerPintFieldSaveModel.objects.aggregate(PintAvg(\'weight\'))\n{\'weight__pintavg\': Decimal(\'0.50500000000000000000\') <Unit(\'kilogram\')>}\n\nIntegerPintFieldSaveModel.objects.aggregate(PintCount(\'weight\'))\n{\'weight__pintcount\': 2}\n\nIntegerPintFieldSaveModel.objects.aggregate(PintMax(\'weight\'))\n{\'weight__pintmax\': Decimal(\'1.0\') <Unit(\'kilogram\')>}\n\nIntegerPintFieldSaveModel.objects.aggregate(PintMin(\'weight\'))\n{\'weight__pintmin\': Decimal(\'0.01\') <Unit(\'kilogram\')>}\n\nIntegerPintFieldSaveModel.objects.aggregate(PintStdDev(\'weight\'))\n{\'weight__pintstddev\': Decimal(\'0.49500000000000000000\') <Unit(\'kilogram\')>}\n\nIntegerPintFieldSaveModel.objects.aggregate(PintSum(\'weight\'))\n{\'weight__pintsum\': Decimal(\'1.01\') <Unit(\'kilogram\')>}\n\nIntegerPintFieldSaveModel.objects.aggregate(PintVariance(\'weight\'))\n{\'weight__pintvariance\': Decimal(\'0.24502500000000000000\') <Unit(\'kilogram\')>}\n```\n\n## Creating your own units\n\nYou can [create your own pint units](https://pint.readthedocs.io/en/stable/advanced/defining.html) if the [default units](https://github.com/hgrecco/pint/blob/master/pint/default_en.txt) in pint are not sufficient.\n\nAnywhere within your project (ideally in settings or a file adjacent to settings), define the custom unit registry by importing Pint\'s default UnitRegistry and extending it:\n\n```python\nfrom pint import UnitRegistry\n\ncustom_ureg = UnitRegistry()\ncustom_ureg.define("custom = [custom]")\ncustom_ureg.define("kilocustom = 1000 * custom")\n```\n\nThen add the custom registry to settings:\n\n`DJANGO_PINT_FIELD_UNIT_REGISTER = custom_ureg`\n\n\n## Model Fields\n\n- **IntegerPintField**: Stores a pint measurement as an integer (-2147483648 to 2147483647).\n- **BigIntegerPintField**: Stores a pint measurement as a big integer (-9223372036854775808 to 9223372036854775807).\n- **DecimalPintField**: Stores a pint measurement as a decimal.\n\n\n## Form Fields\n\n- **IntegerPintFormField**: Used in forms with IntegerPintField and BigIntegerPintField.\n- **DecimalPintFormField**: Used in forms with DecimalPintField.\n\n\n## Widgets\n\n- **PintFieldWidget**: Default widget for all django pint field types.\n\n\n## Settings\n\n<dl>\n  <dt><code>DJANGO_PINT_FIELD_DECIMAL_PRECISION</code></dt>\n  <dd>\n    Determines whether django_pint_field should automatically set the python decimal precision for the project. If an integer greater than 0 is provided, the decimal context precision for the entire project will be set to that value. Otherwise, the precision remains at the default (usually 28).<br>\n    <em>* Type: int</em>\n    <em>* Default: 0</em>\n  </dd>\n\n  <dt><code>DJANGO_PINT_FIELD_UNIT_REGISTER</code></dt>\n  <dd>\n    The Unit Registry to use in the project. Defaults to pint.UnitRegistry.<br>\n    <em>* Type: int</em>\n    <em>* Default: 0</em>\n  </dd>\n</dl>\n\n\n## Rounding modes (upcoming feature)\n\n**decimal.ROUND_CEILING**\nRound towards Infinity.\n\n**decimal.ROUND_DOWN**\nRound towards zero.\n\n**decimal.ROUND_FLOOR**\nRound towards -Infinity.\n\n**decimal.ROUND_HALF_DOWN**\nRound to nearest with ties going towards zero.\n\n**decimal.ROUND_HALF_EVEN**\nRound to nearest with ties going to nearest even integer.\n\n**decimal.ROUND_HALF_UP**\nRound to nearest with ties going away from zero.\n\n**decimal.ROUND_UP**\nRound away from zero.\n\n**decimal.ROUND_05UP**\nRound away from zero if last digit after rounding towards zero would have been 0 or 5; otherwise round towards zero.\n\nRead more about rounding modes for decimals at the [decimal docs](https://docs.python.org/3/library/decimal.html#rounding-modes)\n\n\n## Use the test app with docker compose\n\n### Build and bring up\n\n```\ndocker compose build\ndocker compose run django python manage.py migrate\ndocker compose run django python manage.py createsuperuser\ndocker compose up -d\n```\n\nNavigate to `127.0.0.1:8000`\n\n### Test (assuming you have already performed `build`)\n\n`docker compose run django python manage.py test`\n\n\n## Run psql on the Postgres database\n\n`docker compose exec postgres psql -U postgres`\n\n\n## ToDos:\n- If a unit_choices value is an alias (e.g. pounds vs pound), the form widget will show the incorrect item selected. The correct value is saved in db, though.\n- Implement rounding modes\n- ',
    'author': 'Jack Linke',
    'author_email': 'jack@watervize.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/jacklinke/django-pint-field',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
