"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.findAlarmThresholds = exports.normalizeIntervals = void 0;
/**
 * Normalize the given interval set to cover the complete number line and make sure it has at most one gap
 */
function normalizeIntervals(intervals, changesAreAbsolute) {
    // Make intervals a complete numberline
    const full = orderAndCompleteIntervals(intervals);
    // Add 'undefined's in uncovered areas of the number line
    makeGapsUndefined(full);
    // In case of relative changes, treat 0-change also as 'undefined' (= no change action)
    if (!changesAreAbsolute) {
        makeZerosUndefined(full);
    }
    // Combine adjacent undefines and make sure there's at most one of them
    combineUndefineds(full);
    validateAtMostOneUndefined(full);
    return full;
}
exports.normalizeIntervals = normalizeIntervals;
/**
 * Completely order scaling intervals, making their lower and upper bounds concrete.
 */
function orderAndCompleteIntervals(intervals) {
    if (intervals.length < 2) {
        throw new Error('Require at least 2 intervals');
    }
    for (const interval of intervals) {
        if (interval.lower === undefined && interval.upper === undefined) {
            throw new Error(`Must supply at least one of 'upper' or 'lower', got: ${JSON.stringify(interval)}`);
        }
    }
    // Make a copy
    intervals = intervals.map(x => ({ ...x }));
    // Sort by whatever number we have for each interval
    intervals.sort(comparatorFromKey((x) => x.lower !== undefined ? x.lower : x.upper));
    // Propagate boundaries until no more change
    while (propagateBounds(intervals)) { /* Repeat */ }
    const lastIndex = intervals.length - 1;
    // Validate that no intervals have undefined bounds now, which must mean they're complete.
    if (intervals[0].lower === undefined) {
        intervals[0] = { ...intervals[0], lower: 0 };
    }
    if (intervals[lastIndex].upper === undefined) {
        intervals[lastIndex] = { ...intervals[lastIndex], upper: Infinity };
    }
    for (const interval of intervals) {
        if (interval.lower === undefined || interval.upper === undefined) {
            throw new Error(`Could not determine the lower and upper bounds for ${JSON.stringify(interval)}`);
        }
    }
    const completeIntervals = intervals;
    // Validate that we have nonoverlapping intervals now.
    for (let i = 0; i < completeIntervals.length - 1; i++) {
        if (overlap(completeIntervals[i], completeIntervals[i + 1])) {
            throw new Error(`Two intervals overlap: ${JSON.stringify(completeIntervals[i])} and ${JSON.stringify(completeIntervals[i + 1])}`);
        }
    }
    // Fill up the gaps
    return completeIntervals;
}
/**
 * Make the intervals cover the complete number line
 *
 * This entails adding intervals with an 'undefined' change to fill up the gaps.
 *
 * Since metrics have a halfopen interval, the first one will get a lower bound
 * of 0, the last one will get an upper bound of +Infinity.
 *
 * In case of absolute adjustments, the lower number of the adjacent bound will
 * be used, which means conservative change. In case of relative adjustments,
 * we'll use relative adjusment 0 (which means no change).
 */
function makeGapsUndefined(intervals) {
    // Add edge intervals if necessary, but only for relative adjustments. Since we're
    // going to make scaling intervals extend all the way out to infinity on either side,
    // the result is the same for absolute adjustments anyway.
    if (intervals[0].lower !== 0) {
        intervals.splice(0, 1, {
            lower: 0,
            upper: intervals[0].lower,
            change: undefined,
        });
    }
    if (last(intervals).upper !== Infinity) {
        intervals.push({
            lower: last(intervals).upper,
            upper: Infinity,
            change: undefined,
        });
    }
    let i = 1;
    while (i < intervals.length) {
        if (intervals[i - 1].upper < intervals[i].lower) {
            intervals.splice(i, 0, {
                lower: intervals[i - 1].upper,
                upper: intervals[i].lower,
                change: undefined,
            });
        }
        else {
            i++;
        }
    }
}
/**
 * Turn zero changes into undefined, in-place
 */
function makeZerosUndefined(intervals) {
    for (let i = 0; i < intervals.length; ++i) {
        const interval = intervals[i];
        if (interval.change === 0) {
            intervals[i] = { ...interval, change: undefined };
        }
    }
}
/**
 * If there are adjacent "undefined" intervals, combine them
 */
function combineUndefineds(intervals) {
    let i = 0;
    while (i < intervals.length - 1) {
        if (intervals[i].change === undefined && intervals[i + 1].change === undefined) {
            intervals[i] = { ...intervals[i], upper: intervals[i + 1].upper };
            intervals.splice(i + 1, 1);
        }
        else {
            i++;
        }
    }
}
function validateAtMostOneUndefined(intervals) {
    const undef = intervals.filter(x => x.change === undefined);
    if (undef.length > 1) {
        throw new Error(`Can have at most one no-change interval, got ${JSON.stringify(undef)}`);
    }
}
function comparatorFromKey(keyFn) {
    return (a, b) => {
        const keyA = keyFn(a);
        const keyB = keyFn(b);
        if (keyA < keyB) {
            return -1;
        }
        if (keyA === keyB) {
            return 0;
        }
        return 1;
    };
}
function propagateBounds(intervals) {
    let ret = false;
    // Propagate upper bounds upwards
    for (let i = 0; i < intervals.length - 1; i++) {
        if (intervals[i].upper !== undefined && intervals[i + 1].lower === undefined) {
            intervals[i + 1] = { ...intervals[i + 1], lower: intervals[i].upper };
            ret = true;
        }
    }
    // Propagate lower bounds downwards
    for (let i = intervals.length - 1; i >= 1; i--) {
        if (intervals[i].lower !== undefined && intervals[i - 1].upper === undefined) {
            intervals[i - 1] = { ...intervals[i - 1], upper: intervals[i].lower };
            ret = true;
        }
    }
    return ret;
}
/**
 * Whether two intervals overlap
 */
function overlap(a, b) {
    return a.lower < b.upper && a.upper > b.lower;
}
function last(xs) {
    return xs[xs.length - 1];
}
/**
 * Locate the intervals that should have the alarm thresholds, by index.
 *
 * Pick the intervals on either side of the singleton "undefined" interval, or
 * pick the middle interval if there's no such interval.
 */
function findAlarmThresholds(intervals) {
    const gapIndex = intervals.findIndex(x => x.change === undefined);
    if (gapIndex !== -1) {
        return {
            lowerAlarmIntervalIndex: gapIndex > 0 ? gapIndex - 1 : undefined,
            upperAlarmIntervalIndex: gapIndex < intervals.length - 1 ? gapIndex + 1 : undefined,
        };
    }
    if (intervals.length === 1) {
        return { upperAlarmIntervalIndex: 0 };
    }
    const middleIndex = Math.floor(intervals.length / 2);
    return {
        lowerAlarmIntervalIndex: middleIndex - 1,
        upperAlarmIntervalIndex: middleIndex,
    };
}
exports.findAlarmThresholds = findAlarmThresholds;
//# sourceMappingURL=data:application/json;base64,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