"use strict";
const fc = require("fast-check");
const interval_utils_1 = require("../lib/interval-utils");
const util_1 = require("./util");
function realisticRelativeIntervals() {
    // Function so we don't have to worry about cloning
    return [
        { upper: 10, change: -2 },
        { upper: 20, change: -1 },
        { lower: 80, change: +1 },
        { lower: 90, change: +2 },
    ];
}
module.exports = {
    'test bounds propagation'(test) {
        const intervals = interval_utils_1.normalizeIntervals(realisticRelativeIntervals(), false);
        test.deepEqual(intervals, [
            { lower: 0, upper: 10, change: -2 },
            { lower: 10, upper: 20, change: -1 },
            { lower: 20, upper: 80, change: undefined },
            { lower: 80, upper: 90, change: +1 },
            { lower: 90, upper: Infinity, change: +2 },
        ]);
        test.done();
    },
    'bounds propagation fails if middle boundary missing'(test) {
        test.throws(() => {
            interval_utils_1.normalizeIntervals([
                { lower: 0, change: -2 },
                { upper: 20, change: -1 },
            ], false);
        });
        test.done();
    },
    'lower alarm index is lower than higher alarm index'(test) {
        fc.assert(fc.property(util_1.arbitrary_complete_intervals(), (intervals) => {
            const alarms = interval_utils_1.findAlarmThresholds(intervals);
            return (alarms.lowerAlarmIntervalIndex === undefined
                || alarms.upperAlarmIntervalIndex === undefined
                || alarms.lowerAlarmIntervalIndex < alarms.upperAlarmIntervalIndex);
        }));
        test.done();
    },
    'never pick undefined intervals for relative alarms'(test) {
        fc.assert(fc.property(util_1.arbitrary_complete_intervals(), (intervals) => {
            const alarms = interval_utils_1.findAlarmThresholds(intervals);
            return (alarms.lowerAlarmIntervalIndex === undefined || intervals[alarms.lowerAlarmIntervalIndex].change !== undefined)
                && (alarms.upperAlarmIntervalIndex === undefined || intervals[alarms.upperAlarmIntervalIndex].change !== undefined);
        }));
        test.done();
    },
    'pick intervals on either side of the undefined interval, if present'(test) {
        fc.assert(fc.property(util_1.arbitrary_complete_intervals(), (intervals) => {
            // There must be an undefined interval and it must not be at the edges
            const i = intervals.findIndex(x => x.change === undefined);
            fc.pre(i > 0 && i < intervals.length - 1);
            const alarms = interval_utils_1.findAlarmThresholds(intervals);
            return (alarms.lowerAlarmIntervalIndex === i - 1 && alarms.upperAlarmIntervalIndex === i + 1);
        }));
        test.done();
    },
    'no picking upper bound infinity for lower alarm'(test) {
        fc.assert(fc.property(util_1.arbitrary_complete_intervals(), (intervals) => {
            const alarms = interval_utils_1.findAlarmThresholds(intervals);
            fc.pre(alarms.lowerAlarmIntervalIndex !== undefined);
            return intervals[alarms.lowerAlarmIntervalIndex].upper !== Infinity;
        }));
        test.done();
    },
    'no picking lower bound 0 for upper alarm'(test) {
        fc.assert(fc.property(util_1.arbitrary_complete_intervals(), (intervals) => {
            const alarms = interval_utils_1.findAlarmThresholds(intervals);
            fc.pre(alarms.upperAlarmIntervalIndex !== undefined);
            return intervals[alarms.upperAlarmIntervalIndex].lower !== 0;
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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