"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AccessLogFormat = exports.AccessLogField = exports.LogGroupLogDestination = void 0;
/**
 * Use CloudWatch Logs as a custom access log destination for API Gateway.
 */
class LogGroupLogDestination {
    /**
     *
     */
    constructor(logGroup) {
        this.logGroup = logGroup;
    }
    /**
     * Binds this destination to the CloudWatch Logs.
     */
    bind(_stage) {
        return {
            destinationArn: this.logGroup.logGroupArn,
        };
    }
}
exports.LogGroupLogDestination = LogGroupLogDestination;
/**
 * $context variables that can be used to customize access log pattern.
 */
class AccessLogField {
    /**
     * The API owner's AWS account ID.
     */
    static contextAccountId() {
        return '$context.identity.accountId';
    }
    /**
     * The identifier API Gateway assigns to your API.
     */
    static contextApiId() {
        return '$context.apiId';
    }
    /**
     * A property of the claims returned from the Amazon Cognito user pool after the method caller is successfully authenticated.
     *
     * @param property A property key of the claims.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-integrate-with-cognito.html
     */
    static contextAuthorizerClaims(property) {
        return `$context.authorizer.claims.${property}`;
    }
    /**
     * The principal user identification associated with the token sent by the client and returned from an API Gateway Lambda authorizer (formerly known as a custom authorizer).
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-use-lambda-authorizer.html
     */
    static contextAuthorizerPrincipalId() {
        return '$context.authorizer.principalId';
    }
    /**
     * The stringified value of the specified key-value pair of the `context` map returned from an API Gateway Lambda authorizer function.
     *
     * @param property key of the context map.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-use-lambda-authorizer.html
     */
    static contextAuthorizer(property) {
        return `$context.authorizer.${property}`;
    }
    /**
     * The AWS endpoint's request ID.
     */
    static contextAwsEndpointRequestId() {
        return '$context.awsEndpointRequestId';
    }
    /**
     * The full domain name used to invoke the API.
     *
     * This should be the same as the incoming `Host` header.
     */
    static contextDomainName() {
        return '$context.domainName';
    }
    /**
     * The first label of the `$context.domainName`. This is often used as a caller/customer identifier.
     */
    static contextDomainPrefix() {
        return '$context.domainPrefix';
    }
    /**
     * A string containing an API Gateway error message.
     */
    static contextErrorMessage() {
        return '$context.error.message';
    }
    /**
     * The quoted value of $context.error.message, namely "$context.error.message".
     */
    static contextErrorMessageString() {
        return '$context.error.messageString';
    }
    /**
     * A type of GatewayResponse.
     *
     * This variable can only be used for simple variable substitution in a GatewayResponse body-mapping template,
     * which is not processed by the Velocity Template Language engine, and in access logging.
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/customize-gateway-responses.html
     */
    static contextErrorResponseType() {
        return '$context.error.responseType';
    }
    /**
     * A string containing a detailed validation error message.
     */
    static contextErrorValidationErrorString() {
        return '$context.error.validationErrorString';
    }
    /**
     * The extended ID that API Gateway assigns to the API request, which contains more useful information for debugging/troubleshooting.
     */
    static contextExtendedRequestId() {
        return '$context.extendedRequestId';
    }
    /**
     * The HTTP method used.
     *
     * Valid values include: `DELETE`, `GET`, `HEAD`, `OPTIONS`, `PATCH`, `POST`, and `PUT`.
     */
    static contextHttpMethod() {
        return '$context.httpMethod';
    }
    /**
     * The AWS account ID associated with the request.
     */
    static contextIdentityAccountId() {
        return '$context.identity.accountId';
    }
    /**
     * For API methods that require an API key, this variable is the API key associated with the method request.
     *
     * For methods that don't require an API key, this variable is
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-api-usage-plans.html
     */
    static contextIdentityApiKey() {
        return '$context.identity.apiKey';
    }
    /**
     * The API key ID associated with an API request that requires an API key.
     */
    static contextIdentityApiKeyId() {
        return '$context.identity.apiKeyId';
    }
    /**
     * The principal identifier of the caller making the request.
     */
    static contextIdentityCaller() {
        return '$context.identity.caller';
    }
    /**
     * The Amazon Cognito authentication provider used by the caller making the request.
     *
     * Available only if the request was signed with Amazon Cognito credentials.
     *
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-identity.html
     */
    static contextIdentityCognitoAuthenticationProvider() {
        return '$context.identity.cognitoAuthenticationProvider';
    }
    /**
     * The Amazon Cognito authentication type of the caller making the request.
     *
     * Available only if the request was signed with Amazon Cognito credentials.
     */
    static contextIdentityCognitoAuthenticationType() {
        return '$context.identity.cognitoAuthenticationType';
    }
    /**
     * The Amazon Cognito identity ID of the caller making the request.
     *
     * Available only if the request was signed with Amazon Cognito credentials.
     */
    static contextIdentityCognitoIdentityId() {
        return '$context.identity.cognitoIdentityId';
    }
    /**
     * The Amazon Cognito identity pool ID of the caller making the request.
     *
     * Available only if the request was signed with Amazon Cognito credentials.
     */
    static contextIdentityCognitoIdentityPoolId() {
        return '$context.identity.cognitoIdentityPoolId';
    }
    /**
     * The AWS organization ID.
     */
    static contextIdentityPrincipalOrgId() {
        return '$context.identity.principalOrgId';
    }
    /**
     * The source IP address of the TCP connection making the request to API Gateway.
     *
     * Warning: You should not trust this value if there is any chance that the `X-Forwarded-For` header could be forged.
     */
    static contextIdentitySourceIp() {
        return '$context.identity.sourceIp';
    }
    /**
     * The principal identifier of the user making the request.
     *
     * Used in Lambda authorizers.
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-lambda-authorizer-output.html
     */
    static contextIdentityUser() {
        return '$context.identity.user';
    }
    /**
     * The User-Agent header of the API caller.
     */
    static contextIdentityUserAgent() {
        return '$context.identity.userAgent';
    }
    /**
     * The Amazon Resource Name (ARN) of the effective user identified after authentication.
     *
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/id_users.html
     */
    static contextIdentityUserArn() {
        return '$context.identity.userArn';
    }
    /**
     * The request path.
     *
     * For example, for a non-proxy request URL of https://{rest-api-id.execute-api.{region}.amazonaws.com/{stage}/root/child,
     * this value is /{stage}/root/child.
     */
    static contextPath() {
        return '$context.path';
    }
    /**
     * The request protocol, for example, HTTP/1.1.
     */
    static contextProtocol() {
        return '$context.protocol';
    }
    /**
     * The ID that API Gateway assigns to the API request.
     */
    static contextRequestId() {
        return '$context.requestId';
    }
    /**
     * The request header override.
     *
     * If this parameter is defined, it contains the headers to be used instead of the HTTP Headers that are defined in the Integration Request pane.
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-override-request-response-parameters.html
     */
    static contextRequestOverrideHeader(headerName) {
        return `$context.requestOverride.header.${headerName}`;
    }
    /**
     * The request path override.
     *
     * If this parameter is defined,
     * it contains the request path to be used instead of the URL Path Parameters that are defined in the Integration Request pane.
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-override-request-response-parameters.html
     */
    static contextRequestOverridePath(pathName) {
        return `$context.requestOverride.path.${pathName}`;
    }
    /**
     * The request query string override.
     *
     * If this parameter is defined, it contains the request query strings to be used instead
     * of the URL Query String Parameters that are defined in the Integration Request pane.
     */
    static contextRequestOverrideQuerystring(querystringName) {
        return `$context.requestOverride.querystring.${querystringName}`;
    }
    /**
     * The response header override.
     *
     * If this parameter is defined, it contains the header to be returned instead of the Response header
     * that is defined as the Default mapping in the Integration Response pane.
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-override-request-response-parameters.html
     */
    static contextResponseOverrideHeader(headerName) {
        return `$context.responseOverride.header.${headerName}`;
    }
    /**
     * The response status code override.
     *
     * If this parameter is defined, it contains the status code to be returned instead of the Method response status
     * that is defined as the Default mapping in the Integration Response pane.
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-override-request-response-parameters.html
     */
    static contextResponseOverrideStatus() {
        return '$context.responseOverride.status';
    }
    /**
     * The CLF-formatted request time (dd/MMM/yyyy:HH:mm:ss +-hhmm).
     */
    static contextRequestTime() {
        return '$context.requestTime';
    }
    /**
     * The Epoch-formatted request time.
     */
    static contextRequestTimeEpoch() {
        return '$context.requestTimeEpoch';
    }
    /**
     * The identifier that API Gateway assigns to your resource.
     */
    static contextResourceId() {
        return '$context.resourceId';
    }
    /**
     * The path to your resource.
     *
     * For example, for the non-proxy request URI of `https://{rest-api-id.execute-api.{region}.amazonaws.com/{stage}/root/child`,
     * The $context.resourcePath value is `/root/child`.
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-create-api-step-by-step.html
     */
    static contextResourcePath() {
        return '$context.resourcePath';
    }
    /**
     * The deployment stage of the API request (for example, `Beta` or `Prod`).
     */
    static contextStage() {
        return '$context.stage';
    }
    /**
     * The response received from AWS WAF: `WAF_ALLOW` or `WAF_BLOCK`.
     *
     * Will not be set if the stage is not associated with a web ACL.
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-control-access-aws-waf.html
     */
    static contextWafResponseCode() {
        return '$context.wafResponseCode';
    }
    /**
     * The complete ARN of the web ACL that is used to decide whether to allow or block the request.
     *
     * Will not be set if the stage is not associated with a web ACL.
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-control-access-aws-waf.html
     */
    static contextWebaclArn() {
        return '$context.webaclArn';
    }
    /**
     * The trace ID for the X-Ray trace.
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-enabling-xray.html
     */
    static contextXrayTraceId() {
        return '$context.xrayTraceId';
    }
    /**
     * The authorizer latency in ms.
     */
    static contextAuthorizerIntegrationLatency() {
        return '$context.authorizer.integrationLatency';
    }
    /**
     * The integration latency in ms.
     */
    static contextIntegrationLatency() {
        return '$context.integrationLatency';
    }
    /**
     * For Lambda proxy integration, this parameter represents the status code returned from AWS Lambda, not from the backend Lambda function.
     */
    static contextIntegrationStatus() {
        return '$context.integrationStatus.';
    }
    /**
     * The response latency in ms.
     */
    static contextResponseLatency() {
        return '$context.responseLatency';
    }
    /**
     * The response payload length.
     */
    static contextResponseLength() {
        return '$context.responseLength';
    }
    /**
     * The method response status.
     */
    static contextStatus() {
        return '$context.status';
    }
}
exports.AccessLogField = AccessLogField;
/**
 * factory methods for access log format.
 */
class AccessLogFormat {
    constructor(format) {
        this.format = format;
    }
    /**
     * Custom log format.
     *
     * You can create any log format string. You can easily get the $ context variable by using the methods of AccessLogField.
     *
     * @example
     *
     * custom(JSON.stringify({
     * requestId: AccessLogField.contextRequestId(),
     * sourceIp: AccessLogField.contextIdentitySourceIp(),
     * method: AccessLogFiled.contextHttpMethod(),
     * userContext: {
     * sub: AccessLogField.contextAuthorizerClaims('sub'),
     * email: AccessLogField.contextAuthorizerClaims('email')
     * }
     * }))
     */
    static custom(format) {
        return new AccessLogFormat(format);
    }
    /**
     * Generate Common Log Format.
     */
    static clf() {
        const requester = [AccessLogField.contextIdentitySourceIp(), AccessLogField.contextIdentityCaller(), AccessLogField.contextIdentityUser()].join(' ');
        const requestTime = AccessLogField.contextRequestTime();
        const request = [AccessLogField.contextHttpMethod(), AccessLogField.contextResourcePath(), AccessLogField.contextProtocol()].join(' ');
        const status = [AccessLogField.contextStatus(), AccessLogField.contextResponseLength(), AccessLogField.contextRequestId()].join(' ');
        return new AccessLogFormat(`${requester} [${requestTime}] "${request}" ${status}`);
    }
    /**
     * Access log will be produced in the JSON format with a set of fields most useful in the access log.
     *
     * All fields are turned on by default with the
     * option to turn off specific fields.
     */
    static jsonWithStandardFields(fields = {
        ip: true,
        user: true,
        caller: true,
        requestTime: true,
        httpMethod: true,
        resourcePath: true,
        status: true,
        protocol: true,
        responseLength: true,
    }) {
        return this.custom(JSON.stringify({
            requestId: AccessLogField.contextRequestId(),
            ip: fields.ip ? AccessLogField.contextIdentitySourceIp() : undefined,
            user: fields.user ? AccessLogField.contextIdentityUser() : undefined,
            caller: fields.caller ? AccessLogField.contextIdentityCaller() : undefined,
            requestTime: fields.requestTime ? AccessLogField.contextRequestTime() : undefined,
            httpMethod: fields.httpMethod ? AccessLogField.contextHttpMethod() : undefined,
            resourcePath: fields.resourcePath ? AccessLogField.contextResourcePath() : undefined,
            status: fields.status ? AccessLogField.contextStatus() : undefined,
            protocol: fields.protocol ? AccessLogField.contextProtocol() : undefined,
            responseLength: fields.responseLength ? AccessLogField.contextResponseLength() : undefined,
        }));
    }
    /**
     * Output a format string to be used with CloudFormation.
     */
    toString() {
        return this.format;
    }
}
exports.AccessLogFormat = AccessLogFormat;
//# sourceMappingURL=data:application/json;base64,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