"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ResponseType = exports.GatewayResponse = void 0;
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("./apigateway.generated");
/**
 * Configure the response received by clients, produced from the API Gateway backend.
 *
 * @resource AWS::ApiGateway::GatewayResponse
 */
class GatewayResponse extends core_1.Resource {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id);
        const resource = new apigateway_generated_1.CfnGatewayResponse(this, 'Resource', {
            restApiId: props.restApi.restApiId,
            responseType: props.type.responseType,
            responseParameters: this.buildResponseParameters(props.responseHeaders),
            responseTemplates: props.templates,
            statusCode: props.statusCode,
        });
        this.node.defaultChild = resource;
    }
    buildResponseParameters(responseHeaders) {
        if (!responseHeaders) {
            return undefined;
        }
        const responseParameters = {};
        for (const [header, value] of Object.entries(responseHeaders)) {
            responseParameters[`gatewayresponse.header.${header}`] = value;
        }
        return responseParameters;
    }
}
exports.GatewayResponse = GatewayResponse;
/**
 * Supported types of gateway responses.
 *
 * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/supported-gateway-response-types.html
 */
class ResponseType {
    constructor(type) {
        this.responseType = type;
    }
    /**
     * A custom response type to suppport future cases.
     */
    static of(type) {
        return new ResponseType(type.toUpperCase());
    }
}
exports.ResponseType = ResponseType;
/**
 * The gateway response for authorization failure.
 */
ResponseType.ACCESS_DENIED = new ResponseType('ACCESS_DENIED');
/**
 * The gateway response for an invalid API configuration.
 */
ResponseType.API_CONFIGURATION_ERROR = new ResponseType('API_CONFIGURATION_ERROR');
/**
 * The gateway response when a custom or Amazon Cognito authorizer failed to authenticate the caller.
 */
ResponseType.AUTHORIZER_FAILURE = new ResponseType('AUTHORIZER_FAILURE');
/**
 * The gateway response for failing to connect to a custom or Amazon Cognito authorizer.
 */
ResponseType.AUTHORIZER_CONFIGURATION_ERROR = new ResponseType('AUTHORIZER_CONFIGURATION_ERROR');
/**
 * The gateway response when the request parameter cannot be validated according to an enabled request validator.
 */
ResponseType.BAD_REQUEST_PARAMETERS = new ResponseType('BAD_REQUEST_PARAMETERS');
/**
 * The gateway response when the request body cannot be validated according to an enabled request validator.
 */
ResponseType.BAD_REQUEST_BODY = new ResponseType('BAD_REQUEST_BODY');
/**
 * The default gateway response for an unspecified response type with the status code of 4XX.
 */
ResponseType.DEFAULT_4XX = new ResponseType('DEFAULT_4XX');
/**
 * The default gateway response for an unspecified response type with a status code of 5XX.
 */
ResponseType.DEFAULT_5XX = new ResponseType('DEFAULT_5XX');
/**
 * The gateway response for an AWS authentication token expired error.
 */
ResponseType.EXPIRED_TOKEN = new ResponseType('EXPIRED_TOKEN');
/**
 * The gateway response for an invalid AWS signature error.
 */
ResponseType.INVALID_SIGNATURE = new ResponseType('INVALID_SIGNATURE');
/**
 * The gateway response for an integration failed error.
 */
ResponseType.INTEGRATION_FAILURE = new ResponseType('INTEGRATION_FAILURE');
/**
 * The gateway response for an integration timed out error.
 */
ResponseType.INTEGRATION_TIMEOUT = new ResponseType('INTEGRATION_TIMEOUT');
/**
 * The gateway response for an invalid API key submitted for a method requiring an API key.
 */
ResponseType.INVALID_API_KEY = new ResponseType('INVALID_API_KEY');
/**
 * The gateway response for a missing authentication token error, including the cases when the client attempts to invoke an unsupported API method or resource.
 */
ResponseType.MISSING_AUTHENTICATION_TOKEN = new ResponseType('MISSING_AUTHENTICATION_TOKEN');
/**
 * The gateway response for the usage plan quota exceeded error.
 */
ResponseType.QUOTA_EXCEEDED = new ResponseType('QUOTA_EXCEEDED');
/**
 * The gateway response for the request too large error.
 */
ResponseType.REQUEST_TOO_LARGE = new ResponseType('REQUEST_TOO_LARGE');
/**
 * The gateway response when API Gateway cannot find the specified resource after an API request passes authentication and authorization.
 */
ResponseType.RESOURCE_NOT_FOUND = new ResponseType('RESOURCE_NOT_FOUND');
/**
 * The gateway response when usage plan, method, stage, or account level throttling limits exceeded.
 */
ResponseType.THROTTLED = new ResponseType('THROTTLED');
/**
 * The gateway response when the custom or Amazon Cognito authorizer failed to authenticate the caller.
 */
ResponseType.UNAUTHORIZED = new ResponseType('UNAUTHORIZED');
/**
 * The gateway response when a payload is of an unsupported media type, if strict passthrough behavior is enabled.
 */
ResponseType.UNSUPPORTED_MEDIA_TYPE = new ResponseType('UNSUPPORTED_MEDIA_TYPE');
/**
 * The gateway response when a request is blocked by AWS WAF.
 */
ResponseType.WAF_FILTERED = new ResponseType('WAF_FILTERED');
//# sourceMappingURL=data:application/json;base64,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