"""
Shell cloud API module, documentation:
https://shelly-api-docs.shelly.cloud/cloud-control-api/communication
"""

from typing import Optional, Union
import requests

from .validations import validate_empty_string


class ShellyApi:
    """
    Class providing methods for getting devices, their status and ids
    via API calls.
    """

    def __init__(self, host: str, token: str, timeout: int = 5) -> None:
        validate_empty_string(host, "host")
        validate_empty_string(token, "token")
        self.host: str = host
        self.token: str = token
        self.timeout: int = timeout

    def __query_status_api(self, path: str, id: Optional[str | int] = None) -> dict:
        try:
            payload = {"auth_key": self.token}

            if id:
                payload["id"] = id

            status_request = requests.post(
                f"{self.host}/device/{path}", data=payload, timeout=self.timeout
            )
            status = status_request.json()
            return status
        except (
            requests.exceptions.ConnectTimeout,
            requests.exceptions.ConnectionError,
        ):
            return {"success": False, "msg": "Request couldn't connect or timed out"}

    def get_devices(self) -> dict:
        """Get the status of ALL devices"""
        response = self.__query_status_api("all_status")

        if response is None or response.get("success") is False:
            raise RuntimeError(f"Request failed with: {response}")

        return response

    def get_device_status(self, id: Union[str, int]) -> dict:
        """Get the status of ONE device"""
        response = self.__query_status_api("status", id)

        if response is None or response.get("success") is False:
            raise RuntimeError(f"Request failed with: {response}")

        return response

    def get_device_ids(self) -> list[dict]:
        """Get the ids of all devices"""
        response = self.get_devices()
        devices = response.get("data", {}).get("devices_status")

        return list(devices)
