# -*- coding: utf-8 -*-

# PLEASE DO NOT EDIT THIS FILE, IT IS GENERATED AND WILL BE OVERWRITTEN:
# https://github.com/ccxt/ccxt/blob/master/CONTRIBUTING.md#how-to-contribute-code

from ccxt.async_support.base.exchange import Exchange
import hashlib
from ccxt.base.errors import ExchangeError
from ccxt.base.errors import PermissionDenied
from ccxt.base.errors import ArgumentsRequired
from ccxt.base.errors import BadRequest
from ccxt.base.errors import InsufficientFunds
from ccxt.base.errors import InvalidOrder
from ccxt.base.errors import OrderNotFound
from ccxt.base.errors import NotSupported
from ccxt.base.decimal_to_precision import TICK_SIZE
from ccxt.base.precise import Precise


class coinflex(Exchange):

    def describe(self):
        return self.deep_extend(super(coinflex, self).describe(), {
            'id': 'coinflex',
            'name': 'CoinFLEX',
            'countries': ['SC'],  # Seychelles
            'rateLimit': 120,  # 2500 requests per 5 minutes, 100 requests per minute
            'version': 'v3',
            'certified': False,
            'pro': True,
            'userAgent': self.userAgents['chrome100'],
            'hostname': 'coinflex.com',
            'has': {
                'CORS': None,
                'spot': True,
                'margin': None,
                'swap': True,
                'future': True,
                'option': None,
                'addMargin': None,
                'cancelAllOrders': True,
                'cancelOrder': True,
                'cancelOrders': True,
                'createDepositAddress': None,
                'createLimitOrder': True,
                'createMarketOrder': True,
                'createOrder': True,
                'createPostOnlyOrder': None,
                'createStopLimitOrder': True,
                'createStopMarketOrder': True,
                'createStopOrder': True,
                'editOrder': 'emulated',
                'fetchAccounts': True,
                'fetchBalance': True,
                'fetchBidsAsks': None,
                'fetchBorrowInterest': None,
                'fetchBorrowRate': None,
                'fetchBorrowRateHistory': None,
                'fetchBorrowRates': None,
                'fetchBorrowRatesPerSymbol': None,
                'fetchCanceledOrders': False,
                'fetchClosedOrder': None,
                'fetchClosedOrders': False,
                'fetchCurrencies': True,
                'fetchDeposit': None,
                'fetchDepositAddress': True,
                'fetchDepositAddresses': False,
                'fetchDepositAddressesByNetwork': False,
                'fetchDeposits': True,
                'fetchFundingHistory': True,
                'fetchFundingRate': 'emulated',
                'fetchFundingRateHistory': 'emulated',
                'fetchFundingRates': True,
                'fetchFundingRatesHistory': True,
                'fetchIndexOHLCV': None,
                'fetchL2OrderBook': None,
                'fetchLedger': None,
                'fetchLedgerEntry': None,
                'fetchLeverageTiers': None,
                'fetchMarketLeverageTiers': None,
                'fetchMarkets': True,
                'fetchMarkOHLCV': None,
                'fetchMyTrades': True,
                'fetchOHLCV': True,
                'fetchOpenOrder': None,
                'fetchOpenOrders': True,
                'fetchOrder': True,  # or maybe emulated, actually getting it from fetchOrders
                'fetchOrderBook': True,
                'fetchOrderBooks': None,
                'fetchOrders': True,
                'fetchOrderTrades': None,
                'fetchPermissions': None,
                'fetchPosition': True,
                'fetchPositions': True,
                'fetchPositionsRisk': None,
                'fetchPremiumIndexOHLCV': None,
                'fetchStatus': True,
                'fetchTicker': True,
                'fetchTickers': True,
                'fetchTime': None,
                'fetchTrades': True,
                'fetchTradingFee': None,
                'fetchTradingFees': None,
                'fetchTradingLimits': None,
                'fetchTransactionFee': True,
                'fetchTransactionFees': False,
                'fetchTransactions': None,
                'fetchTransfers': True,
                'fetchWithdrawal': True,
                'fetchWithdrawals': True,
                'loadMarkets': True,
                'privateAPI': True,
                'publicAPI': True,
                'reduceMargin': None,
                'setLeverage': None,
                'setMarginMode': None,
                'setPositionMode': None,
                'signIn': None,
                'transfer': True,
                'withdraw': True,
            },
            'timeframes': {
                '1m': '60s',
                '5m': '300s',
                '15m': '900s',
                '30m': '1800s',
                '1h': '3600s',
                '2h': '7200s',
                '4h': '14400s',
                '1d': '86400s',
            },
            'urls': {
                'logo': 'https://user-images.githubusercontent.com/1294454/168937923-80d6af4a-43b5-4ed9-9d53-31065656be4f.jpg',
                'api': {
                    'rest': 'https://v2api.{hostname}',
                },
                'www': 'https://coinflex.com/',
                'doc': [
                    'https://docs.coinflex.com/',
                ],
                'fees': [
                    'https://coinflex.com/fees/',
                ],
                'test': {
                    'rest': 'https://v2stgapi.{hostname}',
                },
                'referral': 'https://coinflex.com/user-console/register?shareAccountId=S6Y87a8P',
            },
            'api': {
                'public': {
                    'get': {
                        'v2/all/markets': 1,  # superceded by v3/markets
                        'v2/all/assets': 1,  # superceded by v3/assets
                        'v2/publictrades/{marketCode}': 1,
                        'v2/ticker': 1,  # superceded by v3/tickers
                        'v2/delivery/public/funding': 1,  # superceded by v3/funding-rates
                        'v2.1/deliver-auction/{instrumentId}': 1,
                        'v2/candles/{marketCode}': 1,  # superceded by v3/candles
                        'v2/funding-rates/{marketCode}': 1,  # historical funding rates only for 'REPO' symbols(i.e. BTC-USD-REPO-LIN )
                        'v2/depth/{marketCode}/{level}': 1,  # superceded by v3/depth
                        'v2/ping': 1,
                        'v2/flex-protocol/balances/{flexProtocol}': 1,
                        'v2/flex-protocol/positions/{flexProtocol}': 1,
                        'v2/flex-protocol/orders/{flexProtocol}': 1,
                        'v2/flex-protocol/trades/{flexProtocol}/{marketCode}': 1,
                        'v2/flex-protocol/delivery/orders/{flexProtocol}': 1,
                        'v3/markets': 1,
                        'v3/assets': 1,
                        'v3/tickers': 1,
                        'v3/auction': 1,
                        'v3/funding-rates': 1,
                        'v3/candles': 1,
                        'v3/depth': 1,
                        'v3/flexasset/balances': 1,
                        'v3/flexasset/positions': 1,
                        'v3/flexasset/yields': 1,
                    },
                    'post': {
                        'v3/flexasset/redeem': 1,
                        'v3/AMM/redeem': 1,
                    },
                },
                'private': {
                    'get': {
                        'v2/accountinfo': 1,  # superceded by v3/account
                        'v2/balances': 1,  # superceded by v3/account
                        'v2/balances/{instrumentId}': 1,  # superceded by v3/account
                        'v2/positions': 1,
                        'v2/positions/{instrumentId}': 1,
                        'v2/trades/{marketCode}': 1,
                        'v2/orders': 1,
                        'v2.1/orders': 1,
                        'v2.1/delivery/orders': 1,
                        'v2/mint/{asset}': 1,
                        'v2/redeem/{asset}': 1,
                        'v2/funding-payments': 1,
                        'v2/AMM': 1,
                        'v3/account': 1,
                        'v3/deposit-addresses': 1,
                        'v3/deposit': 1,
                        'v3/withdrawal-addresses': 1,
                        'v3/withdrawal': 1,
                        'v3/withdrawal-fee': 1,
                        'v3/transfer': 1,
                        'v3/flexasset/mint': 1,
                        'v3/flexasset/redeem': 1,
                        'v3/flexasset/earned': 1,
                        'v3/AMM': 1,
                        'v3/AMM/balances': 1,
                        'v3/AMM/positions': 1,
                        'v3/AMM/orders': 1,
                        'v3/AMM/trades': 1,
                        'v3/AMM/hash-token': 1,
                        'v2/borrow/{asset}': 1,  # TO_DO : Get borrow history by asset(but doesn't match directly to any unified method)
                        'v2/repay/{asset}': 1,  # TO_DO: Get repay history by asset(but doesn't match directly to any unified method)
                        'v2/borrowingSummary': 1,  # TO_DO : Get borrowing summary
                    },
                    'post': {
                        'v2.1/delivery/orders': 1,
                        'v2/orders/place': 1,  # Note: supports batch/bulk orders
                        'v2/orders/modify': 1,
                        'v2/mint': 1,
                        'v2/redeem': 1,
                        'v2/borrow': 1,  # TO_DO
                        'v2/repay': 1,  # TO_DO
                        'v2/borrow/close': 1,  # TO_DO
                        'v2/AMM/create': 1,
                        'v2/AMM/redeem': 1,
                        'v3/withdrawal': 1,
                        'v3/transfer': 1,
                        'v3/flexasset/mint': 1,
                        'v3/AMM/create': 1,
                    },
                    'delete': {
                        'v2/cancel/orders': 1,
                        'v2/cancel/orders/{marketCode}': 1,
                        'v2.1/delivery/orders/{deliveryOrderId}': 1,
                        'v2/orders/cancel': 1,
                    },
                },
            },
            'fees': {
                'trading': {
                    'tierBased': False,
                    'percentage': True,
                    'maker': self.parse_number('0.000'),
                    'taker': self.parse_number('0.008'),
                },
            },
            'precisionMode': TICK_SIZE,
            'options': {
                'defaultType': 'spot',  # spot, swap
                'networks': {
                    # 'SOLANA': 'SPL',
                    'BTC': 'BTC',
                    'ERC20': 'ERC20',
                    'BEP20': 'BEP20',
                },
                'networksByIds': {
                    # 'SPL': 'SOLANA',
                    'BTC': 'BTC',
                    'ERC20': 'ERC20',
                    'BEP20': 'BEP20',
                },
            },
            'commonCurrencies': {
            },
            'exceptions': {
                'exact': {
                    '40001': BadRequest,
                    '710003': InvalidOrder,
                    '25009': PermissionDenied,
                    '710006': InsufficientFunds,
                    '40035': OrderNotFound,
                    '20001': BadRequest,
                    '25030': BadRequest,
                    '35034': BadRequest,
                    '05001': PermissionDenied,
                    '20020': InvalidOrder,
                },
                'broad': {
                    'no result, please check your parameters': BadRequest,  # 40001
                    'sanity bound check as price': InvalidOrder,  # 710003
                    '2FA is not turned on': PermissionDenied,  # 25009
                    'balance check as balance': InsufficientFunds,  # 710006
                    'Open order not found with clientOrderId or orderId': OrderNotFound,  # 40035
                    'result not found, please check your parameters': BadRequest,  # 20001
                    'Invalid Code': BadRequest,  # 25030
                    'Wallet API is abnormal, please try again or contact customer service': BadRequest,  # 35034
                    'Unauthorized': PermissionDenied,  # 05001
                    'stopPrice or limitPrice is invalid': InvalidOrder,  # 20020
                },
            },
        })

    async def fetch_status(self, params={}):
        """
        the latest known information on the availability of the exchange API
        :param dict params: extra parameters specific to the coinflex api endpoint
        :returns dict: a `status structure <https://docs.ccxt.com/en/latest/manual.html#exchange-status-structure>`
        """
        response = await self.publicGetV2Ping(params)
        #
        #     {"success": "true"}
        #
        statusRaw = self.safe_string(response, 'success')
        status = self.safe_string({'true': 'ok', 'false': 'maintenance'}, statusRaw, statusRaw)
        return {
            'status': status,
            'updated': None,
            'eta': None,
            'url': None,
            'info': response,
        }

    async def fetch_markets(self, params={}):
        """
        retrieves data on all markets for coinflex
        :param dict params: extra parameters specific to the exchange api endpoint
        :returns [dict]: an array of objects representing market data
        """
        # v3 markets has a few less fields available for market-objects, but still enough to precede.
        response = await self.publicGetV3Markets(params)
        #
        #     {
        #         "success": True,
        #         "data": [
        #             {
        #                 "marketCode": "BTC-USD",
        #                 "name": "BTC/USD",
        #                 "referencePair": "BTC/USD",
        #                 "base": "BTC",
        #                 "counter": "USD",
        #                 "type": "SPOT",  # SPOT, FUTURE, REPO, SPREAD
        #                 "tickSize": "1",
        #                 "minSize": "0.001",
        #                 "listedAt": "1593316800000",
        #                 "upperPriceBound": "40632",
        #                 "lowerPriceBound": "37506",
        #                 "markPrice": "39069",
        #                 "lastUpdatedAt": "1651240365178"
        #             },
        #
        #         futures/spreads/repo markets just have the same structure, but different namings
        #
        #             {
        #                 "marketCode": "BTC-USD-SWAP-LIN",
        #                 "name": "BTC/USD Perp",
        #                 "referencePair": "BTC/USD",
        #                 "base": "BTC",
        #                 "counter": "USD",
        #                 "type": "FUTURE",
        #                 ...
        #             },
        #             {
        #                 "marketCode": "BTC-USD-220624-LIN",
        #                 "name": "BTC/USD Q220624",
        #                 "referencePair": "BTC/USD",
        #                 "base": "BTC",
        #                 "counter": "USD",
        #                 "type": "FUTURE",
        #                 "settlementAt": "1656072000000",
        #                 ...
        #             },
        #             {
        #                 "marketCode": "BTC-USD-REPO-LIN",
        #                 "name": "BTC/USD Repo",
        #                 "referencePair": "BTC/USD",
        #                 "base": "BTC-USD",
        #                 "counter": "BTC-USD-SWAP-LIN",
        #                 "type": "REPO",
        #                 ...
        #             },
        #             {
        #                 "marketCode": "BTC-USD-SPR-220624P-LIN",
        #                 "name": "BTC/USD SPR Q220624",
        #                 "referencePair": "BTC/USD",
        #                 "base": "BTC-USD-220624-LIN",
        #                 "counter": "BTC-USD-SWAP-LIN",
        #                 "type": "SPREAD",
        #                 "settlementAt": "1656072000000",
        #                 ...
        #             },
        #         ],
        #     }
        #
        data = self.safe_value(response, 'data', [])
        result = []
        for i in range(0, len(data)):
            market = data[i]
            id = self.safe_string(market, 'marketCode')
            baseId = self.safe_string(market, 'base')
            quoteId = self.safe_string(market, 'counter')
            settleId = self.safe_string(market, 'counter')
            base = self.safe_currency_code(baseId)
            quote = self.safe_currency_code(quoteId)
            settle = self.safe_currency_code(settleId)
            type = self.safe_string(market, 'type')
            settlementTime = self.safe_integer(market, 'settlementAt')
            symbol = base + '/' + quote
            marketType = None
            linear = None
            inverse = None
            if type == 'SPOT':
                marketType = 'spot'
            elif type == 'FUTURE':
                inverse = False
                linear = True
                if settlementTime is None:
                    marketType = 'swap'
                    symbol += ':' + settle
                else:
                    marketType = 'future'
                    symbol += ':' + settle + '-' + self.yymmdd(settlementTime)
            elif type == 'SPREAD' or type == 'REPO':
                symbol = id
            result.append({
                'id': id,
                'symbol': symbol,
                'base': base,
                'quote': quote,
                'settle': settle,
                'baseId': baseId,
                'quoteId': quoteId,
                'settleId': settleId,
                'type': marketType,
                'spot': marketType == 'spot',
                'margin': False,
                'future': marketType == 'future',
                'swap': marketType == 'swap',
                'option': False,
                'active': True,
                'contract': (marketType == 'future' or marketType == 'swap'),
                'linear': linear,
                'inverse': inverse,
                'contractSize': None,
                'expiry': settlementTime,
                'expiryDatetime': self.iso8601(settlementTime),
                'strike': None,
                'optionType': None,
                'precision': {
                    'amount': self.safe_number(market, 'minSize'),
                    'price': self.safe_number(market, 'tickSize'),
                },
                'limits': {
                    'leverage': {
                        'min': None,
                        'max': None,
                    },
                    'amount': {
                        'min': None,
                        'max': None,
                    },
                    'price': {
                        'min': self.safe_number(market, 'lowerPriceBound'),
                        'max': self.safe_number(market, 'upperPriceBound'),
                    },
                    'cost': {
                        'min': None,
                        'max': None,
                    },
                },
                'info': market,
            })
        return result

    async def fetch_currencies(self, params={}):
        """
        fetches all available currencies on an exchange
        :param dict params: extra parameters specific to the coinflex api endpoint
        :returns dict: an associative dictionary of currencies
        """
        response = await self.publicGetV3Assets(params)
        #
        #     {
        #         "success": True,
        #         "data": [
        #           {
        #             "asset": "BTC",
        #             "isCollateral": True,
        #             "loanToValue": "0.950000000",
        #             "networkList": [
        #               {
        #                 "network": "BTC",
        #                 "transactionPrecision": "8",
        #                 "isWithdrawalFeeChargedToUser": True,
        #                 "canDeposit": True,
        #                 "canWithdraw": True,
        #                 "minDeposit": "0.00010",
        #                 "minWithdrawal": "0.00010",
        #                 # "tokenId": "730136783b0cb167727361cd3cbe47bfe3a327e2e91850948d1cb5e2ca8ce7de",  # some coins have self prop
        #               }
        #             ]
        #           },
        #         ]
        #     }
        #
        data = self.safe_value(response, 'data', [])
        result = {}
        for i in range(0, len(data)):
            entry = data[i]
            id = self.safe_string(entry, 'asset')
            code = self.safe_currency_code(id)
            isWithdrawEnabled = True
            isDepositEnabled = True
            fees = {}
            networks = {}
            networkList = self.safe_value(entry, 'networkList', [])
            precisionString = None
            for j in range(0, len(networkList)):
                networkItem = networkList[j]
                networkId = self.safe_string(networkItem, 'network')
                # name = self.safe_string(networkItem, 'name')
                depositEnable = self.safe_value(networkItem, 'canDeposit')
                withdrawEnable = self.safe_value(networkItem, 'canWithdraw')
                isDepositEnabled = isDepositEnabled or depositEnable
                isWithdrawEnabled = isWithdrawEnabled or withdrawEnable
                fees[networkId] = None
                networkPrecisionString = self.safe_string(networkItem, 'transactionPrecision')
                if precisionString is None:
                    precisionString = networkPrecisionString
                else:
                    precisionString = Precise.string_min(precisionString, networkPrecisionString)
                networks[networkId] = {
                    'id': networkId,
                    'network': networkId,
                    'active': isDepositEnabled and isWithdrawEnabled,
                    'deposit': isDepositEnabled,
                    'withdraw': isWithdrawEnabled,
                    'fee': None,
                    'precision': self.parse_number(self.parse_precision(networkPrecisionString)),
                    'limits': {
                        'deposit': {
                            'min': self.safe_number(networkItem, 'minDeposit'),
                            'max': None,
                        },
                        'withdraw': {
                            'min': self.safe_number(networkItem, 'minWithdrawal'),
                            'max': None,
                        },
                    },
                    'info': networkItem,
                }
            result[code] = {
                'id': id,
                'name': code,
                'code': code,
                'precision': self.parse_number(self.parse_precision(precisionString)),
                'info': entry,
                'active': isWithdrawEnabled and isDepositEnabled,
                'deposit': isDepositEnabled,
                'withdraw': isWithdrawEnabled,
                'fee': None,
                'fees': None,
                'limits': {
                    'amount': {
                        'min': None,
                        'max': None,
                    },
                },
                'networks': networks,
            }
        return result

    def set_start_end_times(self, request, since):
        # exchange has 7 days maximum allowed distance between start/end times across its api endpoints
        distance = 7 * 24 * 60 * 60 * 1000  # 7 days
        if since is None:
            since = self.sum(self.milliseconds(), self.timeout) - distance  # don't directly set 7 days ago from self moment, as when request arrives at exchange, it will be more than 7 days from 'current time'. so, add timeout seconds to make sure we have enough time to reach exchange.
        request['startTime'] = since
        currentTs = self.milliseconds()
        sinceWithAddedDistance = self.sum(since, distance)
        if sinceWithAddedDistance < currentTs:
            request['endTime'] = sinceWithAddedDistance
        return request

    async def fetch_trades(self, symbol, since=None, limit=None, params={}):
        """
        get the list of most recent trades for a particular symbol
        :param str symbol: unified symbol of the market to fetch trades for
        :param int|None since: timestamp in ms of the earliest trade to fetch
        :param int|None limit: the maximum amount of trades to fetch
        :param dict params: extra parameters specific to the coinflex api endpoint
        :returns [dict]: a list of `trade structures <https://docs.ccxt.com/en/latest/manual.html?#public-trades>`
        """
        await self.load_markets()
        market = self.market(symbol)
        request = {
            'marketCode': market['id'],
        }
        if limit is not None:
            request['limit'] = limit
        request = self.set_start_end_times(request, since)
        response = await self.publicGetV2PublictradesMarketCode(self.extend(request, params))
        #
        #     {
        #         "event": "publicTrades",
        #         "timestamp": "1651312416050",
        #         "marketCode": "BTC-USD",
        #         "data": [
        #             {
        #                 "matchId": "304734619669458401",
        #                 "matchQuantity": "0.012",
        #                 "matchPrice": "38673",
        #                 "side": "BUY",
        #                 "matchTimestamp": "1651281046230"
        #             },
        #         ]
        #     }
        #
        trades = self.safe_value(response, 'data', [])
        return self.parse_trades(trades, market, since, limit, params)

    async def fetch_my_trades(self, symbol=None, since=None, limit=None, params={}):
        """
        fetch all trades made by the user
        :param str symbol: unified market symbol
        :param int|None since: the earliest time in ms to fetch trades for
        :param int|None limit: the maximum number of trades structures to retrieve
        :param dict params: extra parameters specific to the coinflex api endpoint
        :returns [dict]: a list of `trade structures <https://docs.ccxt.com/en/latest/manual.html#trade-structure>`
        """
        if symbol is None:
            raise ArgumentsRequired(self.id + ' fetchMyTrades() requires a symbol argument')
        await self.load_markets()
        market = self.market(symbol)
        request = {
            'marketCode': market['id'],
        }
        if limit is not None:
            request['limit'] = limit
        request = self.set_start_end_times(request, since)
        response = await self.privateGetV2TradesMarketCode(self.extend(request, params))
        #
        #     {
        #         "event": "trades",
        #         "timestamp": "1651402298537",
        #         "accountId": "38432",
        #         "data": [
        #             {
        #                 "matchId": "8375163007067643787",
        #                 "matchTimestamp": "1651342025862",
        #                 "marketCode": "SHIB-USD-SWAP-LIN",
        #                 "matchQuantity": "313546",
        #                 "matchPrice": "0.00002161",
        #                 "total": "6.77572906",
        #                 "orderMatchType": "TAKER",
        #                 "fees": "0.00542058",
        #                 "feeInstrumentId": "USD",
        #                 "orderId": "1002109741555",
        #                 "side": "BUY",
        #                 "clientOrderId": "1651342025382"
        #             }
        #         ]
        #     }
        #
        trades = self.safe_value(response, 'data', {})
        return self.parse_trades(trades, market, since, limit, params)

    def parse_trade(self, trade, market=None):
        #
        # fetchTrades
        #
        #     {
        #         "matchId": "304734619669458401",
        #         "matchQuantity": "0.012",
        #         "matchPrice": "38673",
        #         "side": "BUY",
        #         "matchTimestamp": "1651281046230"
        #     }
        #
        # fetchMyTrades
        #
        #     {
        #         "matchId": "8375163007067643787",
        #         "matchQuantity": "313546",
        #         "matchPrice": "0.00002161",
        #         "side": "BUY",
        #         "matchTimestamp": "1651342025862",
        #         "marketCode": "SHIB-USD-SWAP-LIN",
        #         "total": "6.77572906",
        #         "orderMatchType": "TAKER",
        #         "fees": "0.00542058",
        #         "feeInstrumentId": "USD",
        #         "orderId": "1002109741555",
        #         "clientOrderId": "1651342025382"
        #     }
        #
        # trades from order-object
        #
        #     {
        #         "8375163007067827477": {
        #             "matchQuantity": "334016",
        #             "matchPrice": "0.00002089",
        #             "timestamp": "1651410712318",
        #             "orderMatchType": "TAKER"
        #         }
        #     }
        #
        marketId = self.safe_string(trade, 'marketCode')
        market = self.safe_market(marketId, market)
        id = None
        timestamp = None
        priceString = None
        amountString = None
        side = None
        cost = None
        fee = None
        takerOrMakerRaw = None
        keys = list(trade.keys())
        length = len(keys)
        if length == 1:
            id = keys[0]
            tradeData = trade[id]
            amountString = self.safe_string(tradeData, 'matchQuantity')
            priceString = self.safe_string(tradeData, 'matchPrice')
            timestamp = self.safe_integer(tradeData, 'timestamp')
            takerOrMakerRaw = self.safe_string(trade, 'orderMatchType')
        else:
            id = self.safe_string(trade, 'matchId')
            timestamp = self.safe_integer(trade, 'matchTimestamp')
            priceString = self.safe_string(trade, 'matchPrice')
            amountString = self.safe_string(trade, 'matchQuantity')
            side = self.safe_string_lower(trade, 'side')
            takerOrMakerRaw = self.safe_string(trade, 'orderMatchType')
            cost = self.safe_number(trade, 'total')
            feeAmount = self.safe_string(trade, 'fees')
            if feeAmount is not None:
                feeCurrency = self.safe_string(trade, 'feeInstrumentId')
                fee = {
                    'currency': self.safe_currency_code(feeCurrency, None),
                    'cost': feeAmount,
                }
        takerOrMaker = self.safe_string({'TAKER': 'taker', 'MAKER': 'maker'}, takerOrMakerRaw, 'taker')
        return self.safe_trade({
            'id': id,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'symbol': market['symbol'],
            'order': self.safe_string(trade, 'orderId'),
            'type': None,
            'takerOrMaker': takerOrMaker,
            'side': side,
            'price': priceString,
            'amount': amountString,
            'cost': cost,
            'fee': fee,
            'info': trade,
        }, market)

    def convert_order_side(self, side):
        sides = {
            'buy': 'BUY',
            'sell': 'SELL',
        }
        return self.safe_string(sides, side, side)

    def parse_order_type(self, type):
        types = {
            'MARKET': 'market',
            'LIMIT': 'limit',
            # 'STOP_LIMIT': 'stop_limit',
        }
        return self.safe_string(types, type, type)

    def convert_order_type(self, type):
        types = {
            'market': 'MARKET',
            'limit': 'LIMIT',
        }
        return self.safe_string(types, type, type)

    def parse_time_in_force(self, status):
        statuses = {
            'FOK': 'FOK',
            'IOC': 'IOC',
            'GTC': 'GTC',
            'MAKER_ONLY': 'PO',
            'MAKER_ONLY_REPRICE': 'PO',
        }
        return self.safe_string(statuses, status, status)

    def parse_order_status(self, status):
        statuses = {
            # fetchOrders
            'OrderOpened': 'open',
            'OrderMatched': 'closed',
            'OrderClosed': 'canceled',
            # cancelOrder
            'CANCELED_BY_USER': 'canceled',
            'CANCELED_ALL_BY_IOC': 'canceled',
            'CANCELED_BY_MAKER_ONLY': 'canceled',
            'CANCELED_BY_FOK': 'canceled',
            'CANCELED_PARTIAL_BY_IOC': 'canceled',
            'CANCELED_BY_AMEND': 'canceled',
            # createOrder(during createOrder, cancelation statuses might also happen)
            'OPEN': 'open',
            'FILLED': 'closed',
        }
        return self.safe_string(statuses, status, status)

    async def fetch_tickers(self, symbols=None, params={}):
        """
        fetches price tickers for multiple markets, statistical calculations with the information calculated over the past 24 hours each market
        :param [str]|None symbols: unified symbols of the markets to fetch the ticker for, all market tickers are returned if not assigned
        :param dict params: extra parameters specific to the coinflex api endpoint
        :returns dict: an array of `ticker structures <https://docs.ccxt.com/en/latest/manual.html#ticker-structure>`
        """
        await self.load_markets()
        response = await self.publicGetV3Tickers(params)
        #
        #     {
        #         "success": True,
        #         "data": [
        #             {
        #                 "marketCode": "BTC-USD",
        #                 "markPrice": "38649",
        #                 "open24h": "38799",
        #                 "high24h": "39418.0",
        #                 "low24h": "38176.0",
        #                 "volume24h": "18650098.7500",
        #                 "currencyVolume24h": "481.898",
        #                 "openInterest": "0",
        #                 "lastTradedPrice": "38632.0",
        #                 "lastTradedQuantity": "0.001",
        #                 "lastUpdatedAt": "1651314699020"
        #             }
        #         ]
        #     }
        #
        data = self.safe_value(response, 'data', [])
        return self.parse_tickers(data, symbols, params)

    async def fetch_ticker(self, symbol, params={}):
        """
        fetches a price ticker, a statistical calculation with the information calculated over the past 24 hours for a specific market
        :param str symbol: unified symbol of the market to fetch the ticker for
        :param dict params: extra parameters specific to the coinflex api endpoint
        :returns dict: a `ticker structure <https://docs.ccxt.com/en/latest/manual.html#ticker-structure>`
        """
        await self.load_markets()
        market = self.market(symbol)
        request = {
            'marketCode': market['id'],
        }
        response = await self.publicGetV3Tickers(self.extend(request, params))
        #
        #     {
        #         "success": True,
        #         "data": [
        #             {
        #                 "marketCode": "BTC-USD",
        #                 "markPrice": "38649",
        #                 "open24h": "38799",
        #                 "high24h": "39418.0",
        #                 "low24h": "38176.0",
        #                 "volume24h": "18650098.7500",
        #                 "currencyVolume24h": "481.898",
        #                 "openInterest": "0",
        #                 "lastTradedPrice": "38632.0",
        #                 "lastTradedQuantity": "0.001",
        #                 "lastUpdatedAt": "1651314699020"
        #             }
        #         ]
        #     }
        #
        data = self.safe_value(response, 'data', [])
        ticker = self.safe_value(data, 0, {})
        return self.parse_ticker(ticker, market)

    def parse_ticker(self, ticker, market=None):
        #
        #     {
        #         "marketCode": "BTC-USD",
        #         "markPrice": "38649",
        #         "open24h": "38799",
        #         "high24h": "39418.0",
        #         "low24h": "38176.0",
        #         "volume24h": "18650098.7500",
        #         "currencyVolume24h": "481.898",
        #         "openInterest": "0",
        #         "lastTradedPrice": "38632.0",
        #         "lastTradedQuantity": "0.001",
        #         "lastUpdatedAt": "1651314699020"
        #     }
        #
        timestamp = self.safe_integer(ticker, 'lastUpdatedAt')
        marketId = self.safe_string(ticker, 'marketCode')
        market = self.safe_market(marketId, market)
        close = self.safe_string(ticker, 'lastTradedPrice')
        open = self.safe_string(ticker, 'open24h')
        return self.safe_ticker({
            'symbol': market['symbol'],
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'high': self.safe_string(ticker, 'high24h'),
            'low': self.safe_string(ticker, 'low24h'),
            'bid': None,
            'bidVolume': None,
            'ask': None,
            'askVolume': None,
            'vwap': None,
            'open': open,
            'close': close,
            'last': close,
            'previousClose': None,
            'change': None,
            'percentage': None,
            'average': None,
            'baseVolume': self.safe_string(ticker, 'currencyVolume24h'),
            'quoteVolume': None,
            'info': ticker,
        }, market)

    async def fetch_funding_history(self, symbol=None, since=None, limit=None, params={}):
        """
        fetch the history of funding payments paid and received on self account
        :param str|None symbol: unified market symbol
        :param int|None since: the earliest time in ms to fetch funding history for
        :param int|None limit: the maximum number of funding history structures to retrieve
        :param dict params: extra parameters specific to the coinflex api endpoint
        :returns dict: a `funding history structure <https://docs.ccxt.com/en/latest/manual.html#funding-history-structure>`
        """
        await self.load_markets()
        request = {}
        market = None
        if symbol is not None:
            market = self.market(symbol)
            request['marketCode'] = market['id']
        request = self.set_start_end_times(request, since)
        if limit is not None:
            request['limit'] = limit
        if since is not None:
            request['startTime'] = since
        response = await self.privateGetV2FundingPayments(self.extend(request, params))
        #
        #     {
        #         "event": "fundingPayments",
        #         "timestamp": "1651750925903",
        #         "accountId": "38422",
        #         "data": [
        #             {
        #                 "marketCode": "SHIB-USD-SWAP-LIN",
        #                 "payment": "-0.00007112",
        #                 "rate": "0.000005",
        #                 "position": "661287",
        #                 "markPrice": "0.00002151",
        #                 "timestamp": "1651420807679"
        #             },
        #         ]
        #     }
        #
        data = self.safe_value(response, 'data', [])
        result = []
        for i in range(0, len(data)):
            entry = data[i]
            marketId = self.safe_string(entry, 'marketCode')
            timestamp = self.safe_string(entry, 'timestamp')
            result.append({
                'symbol': self.safe_symbol(marketId, market),
                'code': None,
                'timestamp': self.parse8601(timestamp),
                'datetime': timestamp,
                'id': None,
                'amount': self.safe_number(entry, 'payment'),
                'info': entry,
            })
        sorted = self.sort_by(result, 'timestamp')
        return self.filter_by_symbol_since_limit(sorted, symbol, since, limit)

    async def fetch_funding_rate(self, symbol, params={}):
        """
        fetch the current funding rate
        :param str symbol: unified market symbol
        :param dict params: extra parameters specific to the coinflex api endpoint
        :returns dict: a `funding rate structure <https://docs.ccxt.com/en/latest/manual.html#funding-rate-structure>`
        """
        # TODO: self can be moved as emulated into base
        if self.has['fetchFundingRates']:
            response = await self.fetch_funding_rates([symbol], params)
            return self.safe_value(response, symbol)
        else:
            raise NotSupported(self.id + ' fetchFundingRate() not supported yet')

    async def fetch_funding_rates(self, symbols=None, params={}):
        """
        fetch the funding rate for multiple markets
        :param [str]|None symbols: list of unified market symbols
        :param dict params: extra parameters specific to the coinflex api endpoint
        :returns dict: a dictionary of `funding rates structures <https://docs.ccxt.com/en/latest/manual.html#funding-rates-structure>`, indexe by market symbols
        """
        await self.load_markets()
        request = {}
        market = None
        if isinstance(symbols, list) and len(symbols) == 1:
            market = self.market(symbols[0])
            request['marketCode'] = market['id']
        response = await self.publicGetV3FundingRates(self.extend(request, params))
        #
        #     {
        #         "success": True,
        #         "data": [
        #             {
        #                 "marketCode": "BTC-USD-SWAP-LIN",
        #                 "fundingRate": "0.000005000",
        #                 "netDelivered": "-18.676",
        #                 "createdAt": "1651312802926"
        #             },
        #             {
        #                 "marketCode": "BTC-USD-SWAP-LIN",
        #                 "fundingRate": "0.000005000",
        #                 "netDelivered": "-19.402",
        #                 "createdAt": "1651309202926"
        #             },
        #          ]
        #      }
        #
        data = self.safe_value(response, 'data', [])
        latestRates = {}
        resultedArray = []
        for i in range(0, len(data)):
            entry = data[i]
            marketId = self.safe_string(entry, 'marketCode')
            if not (marketId in latestRates):
                latestRates[marketId] = True
                resultedArray.append(entry)
        return self.parse_funding_rates(resultedArray, market)

    async def fetch_funding_rate_history(self, symbol, since=None, limit=None, params={}):
        """
        fetches historical funding rate prices
        :param str|None symbol: unified symbol of the market to fetch the funding rate history for
        :param int|None since: timestamp in ms of the earliest funding rate to fetch
        :param int|None limit: the maximum amount of `funding rate structures <https://docs.ccxt.com/en/latest/manual.html?#funding-rate-history-structure>` to fetch
        :param dict params: extra parameters specific to the coinflex api endpoint
        :returns [dict]: a list of `funding rate structures <https://docs.ccxt.com/en/latest/manual.html?#funding-rate-history-structure>`
        """
        # TODO: self can be moved as emulated into base
        if self.has['fetchFundingRatesHistory']:
            response = await self.fetch_funding_rates_history([symbol], since, limit, params)
            return self.filter_by_symbol_since_limit(response, symbol, since, limit)
        else:
            raise NotSupported(self.id + ' fetchFundingRateHistory() not supported yet')

    async def fetch_funding_rates_history(self, symbols=None, since=None, limit=None, params={}):
        await self.load_markets()
        request = {}
        market = None
        if isinstance(symbols, list) and len(symbols) == 1:
            market = self.market(symbols[0])
            request['marketCode'] = market['id']
        request = self.set_start_end_times(request, since)
        if limit is not None:
            request['limit'] = limit
        response = await self.publicGetV3FundingRates(self.extend(request, params))
        # same response/endtpoint as in fetchFundingRates
        data = self.safe_value(response, 'data', [])
        return self.parse_funding_rate_histories(data, market, since, limit)

    def parse_funding_rate_histories(self, response, market=None, since=None, limit=None):
        rates = []
        for i in range(0, len(response)):
            parsed = self.parse_funding_rate(response[i], market)
            rates.append(parsed)
        sorted = self.sort_by(rates, 'timestamp')
        symbol = self.safe_string(market, 'symbol')
        return self.filter_by_symbol_since_limit(sorted, symbol, since, limit)

    def parse_funding_rate(self, contract, market=None):
        #
        # fetchFundingRate, fetchFundingRates, fetchFundingRateHistory
        #
        #     {
        #         "marketCode": "BTC-USD-SWAP-LIN",
        #         "fundingRate": "0.000005000",
        #         "netDelivered": "-18.676",
        #         "createdAt": "1651312802926"
        #     }
        #
        marketId = self.safe_string(contract, 'marketCode')
        fundingDatetime = self.safe_string(contract, 'createdAt')
        return {
            'symbol': self.safe_symbol(marketId, market),
            'markPrice': None,
            'indexPrice': None,
            'interestRate': None,
            'estimatedSettlePrice': None,
            'timestamp': None,
            'datetime': None,
            'fundingRate': self.safe_number(contract, 'fundingRate'),
            'fundingTimestamp': fundingDatetime,
            'fundingDatetime': self.iso8601(fundingDatetime),
            'nextFundingRate': None,
            'nextFundingTimestamp': None,
            'nextFundingDatetime': None,
            'previousFundingRate': None,
            'previousFundingTimestamp': None,
            'previousFundingDatetime': None,
            'info': contract,
        }

    async def fetch_ohlcv(self, symbol, timeframe='1m', since=None, limit=None, params={}):
        """
        fetches historical candlestick data containing the open, high, low, and close price, and the volume of a market
        :param str symbol: unified symbol of the market to fetch OHLCV data for
        :param str timeframe: the length of time each candle represents
        :param int|None since: timestamp in ms of the earliest candle to fetch
        :param int|None limit: the maximum amount of candles to fetch
        :param dict params: extra parameters specific to the coinflex api endpoint
        :returns [[int]]: A list of candles ordered as timestamp, open, high, low, close, volume
        """
        await self.load_markets()
        market = self.market(symbol)
        request = {
            'marketCode': market['id'],
            'timeframe': self.timeframes[timeframe],
        }
        if limit is not None:
            request['limit'] = limit
        request = self.set_start_end_times(request, since)
        response = await self.publicGetV3Candles(self.extend(request, params))
        #
        #     {
        #         "success": True,
        #         "timeframe": "3600s",
        #         "data": [
        #             {
        #                 "open": "38571.00000000",
        #                 "high": "38604.00000000",
        #                 "low": "38570.00000000",
        #                 "close": "38602.00000000",
        #                 "volume": "722820.88000000",
        #                 "currencyVolume": "18.74000000",
        #                 "openedAt": "1651316400000"
        #             },
        #         ]
        #     }
        #
        data = self.safe_value(response, 'data', [])
        return self.parse_ohlcvs(data, market, timeframe, since, limit)

    def parse_ohlcv(self, ohlcv, market=None):
        #
        #     {
        #         "open": "38571.00000000",
        #         "high": "38604.00000000",
        #         "low": "38570.00000000",
        #         "close": "38602.00000000",
        #         "volume": "722820.88000000",
        #         "currencyVolume": "18.74000000",
        #         "openedAt": "1651316400000"
        #     }
        #
        return [
            self.safe_integer(ohlcv, 'openedAt'),
            self.safe_number(ohlcv, 'open'),
            self.safe_number(ohlcv, 'high'),
            self.safe_number(ohlcv, 'low'),
            self.safe_number(ohlcv, 'close'),
            self.safe_number(ohlcv, 'currencyVolume'),
        ]

    async def fetch_order_book(self, symbol, limit=None, params={}):
        """
        fetches information on open orders with bid(buy) and ask(sell) prices, volumes and other data
        :param str symbol: unified symbol of the market to fetch the order book for
        :param int|None limit: the maximum amount of order book entries to return
        :param dict params: extra parameters specific to the coinflex api endpoint
        :returns dict: A dictionary of `order book structures <https://docs.ccxt.com/en/latest/manual.html#order-book-structure>` indexed by market symbols
        """
        await self.load_markets()
        market = self.market(symbol)
        request = {
            'marketCode': market['id'],
        }
        if limit is not None:
            request['level'] = limit
        response = await self.publicGetV3Depth(self.extend(request, params))
        #
        #     {
        #         "success": True,
        #         "level": "5",
        #         "data": {
        #             "marketCode": "BTC-USD-SWAP-LIN",
        #             "lastUpdatedAt": "1651322410296",
        #             "asks": [[38563, 0.312], [38568, 0.001], [38570, 0.001], [38572, 0.002], [38574, 0.001]],
        #             "bids": [[38562, 0.001], [38558, 0.001], [38556, 0.1], [38555, 0.003], [38554, 0.445]]
        #         }
        #     }
        #
        orderbook = self.safe_value(response, 'data', {})
        timestamp = self.safe_integer(orderbook, 'lastUpdatedAt')
        return self.parse_order_book(orderbook, symbol, timestamp)

    async def get_account_data(self, params={}):
        response = await self.privateGetV3Account(params)
        #
        #     {
        #         "success": True,
        #         "data": [
        #             {
        #                 "accountId": "38432",
        #                 "name": "main",
        #                 "accountType": "LINEAR",
        #                 "balances": [
        #                     {
        #                         "asset": "USDT",
        #                         "total": "0.33",
        #                         "available": "0.33",
        #                         "reserved": "0",
        #                         "lastUpdatedAt": "1651233586761"
        #                     },
        #                 ],
        #                 "positions": [
        #                     {
        #                         "marketCode": "SHIB-USD-SWAP-LIN",
        #                         "baseAsset": "SHIB",
        #                         "counterAsset": "USD",
        #                         "position": "313546.0",
        #                         "entryPrice": "0.00002161",
        #                         "markPrice": "0.00002158",
        #                         "positionPnl": "-0.009406380",
        #                         "estLiquidationPrice": "0",
        #                         "lastUpdatedAt": "1651342025876"
        #                     },
        #                 ],
        #                 "collateral": "28.297558",
        #                 "notionalPositionSize": "6.82903188",
        #                 "portfolioVarMargin": "0.682589",
        #                 "riskRatio": "41.5216",
        #                 "maintenanceMargin": "0.34129450",
        #                 "marginRatio": "1.20",
        #                 "liquidating": False,
        #                 "feeTier": "0",
        #                 "createdAt": "1651232948406"
        #             }
        #         ]
        #     }
        #
        return self.safe_value(response, 'data', [])

    async def fetch_accounts(self, params={}):
        """
        fetch all the accounts associated with a profile
        :param dict params: extra parameters specific to the coinflex api endpoint
        :returns dict: a dictionary of `account structures <https://docs.ccxt.com/en/latest/manual.html#account-structure>` indexed by the account type
        """
        await self.load_markets()
        data = await self.get_account_data(params)
        result = []
        for i in range(0, len(data)):
            account = data[i]
            result.append({
                'id': self.safe_string(account, 'accountId'),
                'type': None,
                'code': None,
                'info': account,
            })
        return result

    async def fetch_balance(self, params={}):
        """
        query for balance and get the amount of funds available for trading or funds locked in orders
        :param dict params: extra parameters specific to the coinflex api endpoint
        :returns dict: a `balance structure <https://docs.ccxt.com/en/latest/manual.html?#balance-structure>`
        """
        await self.load_markets()
        data = await self.get_account_data(params)
        targetAccount = self.safe_value(data, 0)
        return self.parse_balance(targetAccount)

    def parse_balance(self, data):
        balances = self.safe_value(data, 'balances', [])
        result = {}
        for i in range(0, len(balances)):
            balance = balances[i]
            #
            #     {
            #         "asset": "USDT",
            #         "total": "0.33",
            #         "available": "0.33",
            #         "reserved": "0",
            #         "lastUpdatedAt": "1651233586761"
            #     }
            #
            currencyId = self.safe_string(balance, 'asset')
            code = self.safe_currency_code(currencyId)
            account = self.account()
            account['free'] = self.safe_string(balance, 'available')
            account['total'] = self.safe_string(balance, 'total')
            result[code] = account
        timestamp = self.safe_integer(data, 'createdAt')
        result['timestamp'] = timestamp
        result['datetime'] = self.iso8601(timestamp)
        return self.safe_balance(result)

    async def fetch_order(self, id, symbol=None, params={}):
        """
        fetches information on an order made by the user
        :param str symbol: unified symbol of the market the order was made in
        :param dict params: extra parameters specific to the coinflex api endpoint
        :returns dict: An `order structure <https://docs.ccxt.com/en/latest/manual.html#order-structure>`
        """
        if symbol is None:
            raise ArgumentsRequired(self.id + ' fetchOrder() requires a symbol argument')
        request = {
            'orderId': id,
        }
        results = await self.fetch_orders(symbol, None, None, self.extend(request, params))
        order = self.safe_value(results, 0)
        if order is None:
            raise OrderNotFound(self.id + ' order ' + id + ' not found')
        return order

    async def fetch_orders(self, symbol=None, since=None, limit=None, params={}):
        """
        fetches information on multiple orders made by the user
        :param str|None symbol: unified market symbol of the market orders were made in
        :param int|None since: the earliest time in ms to fetch orders for
        :param int|None limit: the maximum number of  orde structures to retrieve
        :param dict params: extra parameters specific to the coinflex api endpoint
        :returns [dict]: a list of [order structures]{@link https://docs.ccxt.com/en/latest/manual.html#order-structure
        """
        await self.load_markets()
        request = {}
        market = None
        if symbol is not None:
            market = self.market(symbol)
            request['marketCode'] = market['id']
        if limit is not None:
            request['limit'] = limit
        request = self.set_start_end_times(request, since)
        response = await self.privateGetV21Orders(self.extend(request, params))
        #
        #     {
        #         "event": "orders",
        #         "timestamp": "1651410725892",
        #         "accountId": "38432",
        #         "data": [
        #             {
        #                 "status": "OrderMatched",
        #                 "orderId": "1002113333774",
        #                 "clientOrderId": "1651410682769",
        #                 "marketCode": "SHIB-USD-SWAP-LIN",
        #                 "side": "BUY",
        #                 "orderType": "STOP_LIMIT",
        #                 "price": "0.00002100",
        #                 "lastTradedPrice": "0.00002089",
        #                 "avgFillPrice": "0.00002089",
        #                 "stopPrice": "0.00002055",
        #                 "limitPrice": "0.00002100",
        #                 "quantity": "334016",
        #                 "filledQuantity": "334016",
        #                 "remainQuantity": "0",
        #                 "matchIds": [
        #                     {
        #                         "8375163007067827477": {
        #                             "matchQuantity": "334016",
        #                             "matchPrice": "0.00002089",
        #                             "timestamp": "1651410712318",
        #                             "orderMatchType": "TAKER"
        #                         }
        #                     }
        #                 ],
        #                 "fees": {
        #                     "USD": "-0.00558207"
        #                 },
        #                 "timeInForce": "GTC",
        #                 "isTriggered": "false"
        #             },
        #         ]
        #     }
        data = self.safe_value(response, 'data', [])
        return self.parse_orders(data, market, since, limit, params)

    async def fetch_open_orders(self, symbol=None, since=None, limit=None, params={}):
        """
        fetch all unfilled currently open orders
        :param str|None symbol: unified market symbol
        :param int|None since: the earliest time in ms to fetch open orders for
        :param int|None limit: the maximum number of  open orders structures to retrieve
        :param dict params: extra parameters specific to the coinflex api endpoint
        :returns [dict]: a list of `order structures <https://docs.ccxt.com/en/latest/manual.html#order-structure>`
        """
        await self.load_markets()
        market = self.market(symbol) if (symbol is not None) else None
        response = await self.privateGetV2Orders(params)
        #
        #     {
        #         "event": "orders",
        #         "timestamp": "1651423023567",
        #         "accountId": "38432",
        #         "data": [
        #             {
        #                 "orderId": "1002114041607",
        #                 "marketCode": "SHIB-USD-SWAP-LIN",
        #                 "clientOrderId": "1651422991501",
        #                 "side": "BUY",
        #                 "orderType": "LIMIT",
        #                 "quantity": "400575.0",
        #                 "remainingQuantity": "400575.0",
        #                 "price": "0.000019",
        #                 "stopPrice": null,
        #                 "limitPrice": "0.000019",
        #                 "orderCreated": "1651422991179",
        #                 "lastModified": "1651422991186",
        #                 "lastTradeTimestamp": "1651422991181",
        #                 "timeInForce": "MAKER_ONLY"
        #             }
        #         ]
        #     }
        #
        data = self.safe_value(response, 'data', [])
        return self.parse_orders(data, market, since, limit, params)

    def parse_order(self, order, market=None):
        #
        # fetchOrders
        #
        #     {
        #         "orderId": "1002113333774",
        #         "clientOrderId": "1651410682769",
        #         "marketCode": "SHIB-USD-SWAP-LIN",
        #         "side": "BUY",
        #         "orderType": "STOP_LIMIT",  # MARKET, STOP_LIMIT, LIMIT
        #         "quantity": "334016",
        #         "price": "0.00002100",
        #         "limitPrice": "0.00002100",  # only available for limit types
        #         "stopPrice": "0.00002055",  # only available for stop types
        #         "timeInForce": "GTC",
        #         "lastTradedPrice": "0.00002089",
        #         "avgFillPrice": "0.00002089",
        #         "status": "OrderMatched",  # OrderOpened, OrderMatched, OrderClosed
        #         "filledQuantity": "334016",
        #         "remainQuantity": "0",
        #         "matchIds": [ # only available for filled order
        #             {
        #                 "8375163007067827477": {
        #                     "matchQuantity": "334016",
        #                     "matchPrice": "0.00002089",
        #                     "timestamp": "1651410712318",
        #                     "orderMatchType": "TAKER"
        #                 }
        #             }
        #         ],
        #         "fees": {
        #             "USD": "-0.00558207"
        #         },
        #         "isTriggered": "false"
        #     }
        #
        # fetchOpenOrders(last few props are different)
        #
        #     {
        #         "orderId": "1002114041607",
        #         "clientOrderId": "1651422991501",
        #         "marketCode": "SHIB-USD-SWAP-LIN",
        #         "side": "BUY",
        #         "orderType": "LIMIT",
        #         "quantity": "400575.0",
        #         "price": "0.000019",
        #         "limitPrice": "0.000019",
        #         "stopPrice": null,
        #         "timeInForce": "MAKER_ONLY"
        #         "lastTradeTimestamp": "1651422991181",
        #         "lastModified": "1651422991186",
        #         "orderCreated": "1651422991179",
        #         "remainingQuantity": "400575.0",
        #      }
        #
        # createOrder
        #
        #     {
        #         "success": "true",
        #         "timestamp": "1651620006856",
        #         "clientOrderId": "1651620006827000",
        #         "orderId": "1002124426084",
        #         "price": "0.65",
        #         "quantity": "8.0",
        #         "side": "BUY",
        #         "status": "FILLED",  # OPEN, FILLED
        #         "marketCode": "XRP-USD",
        #         "timeInForce": "GTC",
        #         "matchId": "5636974433720947783",  # zero if not filled
        #         "lastTradedPrice": "0.6028",  # field not present if order doesn't have any fills
        #         "matchQuantity": "8.0",  # field not present if order doesn't have any fills
        #         "orderMatchType": "TAKER",  # field not present if order doesn't have any fills
        #         "remainQuantity": "0.0",  # field not present if order doesn't have any fills
        #         "notice": "OrderMatched",
        #         "orderType": "LIMIT",
        #         "fees": "0.003857920",  # field not present if order doesn't have any fills
        #         "feeInstrumentId": "USD",  # field not present if order doesn't have any fills
        #         "isTriggered": "false"
        #     }
        #
        # cancelAllOrders
        #
        #     {
        #         "marketCode": "XRP-USD",
        #         "msg": "All open orders for the specified market have been queued for cancellation"
        #     }
        #
        marketId = self.safe_string(order, 'marketCode')
        market = self.safe_market(marketId, market)
        isCreateOrder = ('timestamp' in order)
        symbol = market['symbol']
        timestamp = self.safe_integer_2(order, 'timestamp', 'orderCreated')
        orderId = self.safe_string(order, 'orderId')
        clientOrderId = self.safe_string(order, 'clientOrderId')
        statusRaw = self.safe_string(order, 'status')
        status = self.parse_order_status(statusRaw)
        orderTypeRaw = self.safe_string(order, 'orderType')
        orderType = self.parse_order_type(orderTypeRaw)
        filledQuantityRaw = self.safe_string(order, 'filledQuantity')
        if isCreateOrder:
            filledQuantityRaw = self.safe_string(order, 'matchQuantity', '0')
        avgPriceRaw = self.safe_string(order, 'avgFillPrice')
        timeInForceRaw = self.safe_string(order, 'timeInForce')
        timeInForce = self.parse_time_in_force(timeInForceRaw)
        price = self.safe_string(order, 'price')
        limitPrice = self.safe_string(order, 'limitPrice')
        finalLimitPrice = None
        if isCreateOrder:
            finalLimitPrice = price
        else:
            finalLimitPrice = limitPrice
        trades = self.safe_value(order, 'matchIds')
        cost = None
        if avgPriceRaw is not None and filledQuantityRaw is not None:
            cost = Precise.string_mul(avgPriceRaw, filledQuantityRaw)
        feesRaw = self.safe_value(order, 'fees')
        fees = None
        if feesRaw is not None:
            if isCreateOrder:
                fees = {
                    'currency': self.safe_string(order, 'feeInstrumentId'),
                    'fee': self.safe_string(order, 'fees'),
                }
            else:
                feeKeys = list(feesRaw.keys())
                if len(feeKeys) > 0:
                    firstCurrencyId = feeKeys[0]
                    fees = {
                        'currency': self.safe_currency(firstCurrencyId),
                        'fee': feesRaw[firstCurrencyId],
                    }
        return self.safe_order({
            'id': orderId,
            'symbol': symbol,
            'clientOrderId': clientOrderId,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'lastTradeTimestamp': self.safe_integer(order, 'lastTradeTimestamp'),
            'timeInForce': timeInForce,
            'postOnly': timeInForce == 'PO',
            'status': status,
            'side': self.safe_string_lower(order, 'side'),
            'price': finalLimitPrice,
            'type': orderType,
            'stopPrice': self.safe_number(order, 'stopPrice'),
            'amount': self.safe_number(order, 'quantity'),
            'filled': self.parse_number(filledQuantityRaw),
            'remaining': self.safe_number_2(order, 'remainQuantity', 'remainingQuantity'),
            'average': self.parse_number(avgPriceRaw),
            'cost': self.parse_number(cost),
            'fee': fees,
            'trades': trades,
            'info': order,
        }, market)

    async def fetch_position(self, symbol, params={}):
        """
        fetch data on a single open contract trade position
        :param str symbol: unified market symbol of the market the position is held in, default is None
        :param dict params: extra parameters specific to the coinflex api endpoint
        :returns dict: a `position structure <https://docs.ccxt.com/en/latest/manual.html#position-structure>`
        """
        await self.load_markets()
        positions = await self.fetch_positions(None, params)
        array = self.filter_by_symbol(positions, symbol)
        return self.safe_value(array, 0)  # exchange doesn't seem to have hedge mode, so the array will contain only one position per symbol

    async def fetch_positions(self, symbols=None, params={}):
        """
        fetch all open positions
        :param [str]|None symbols: list of unified market symbols
        :param dict params: extra parameters specific to the coinflex api endpoint
        :returns [dict]: a list of `position structure <https://docs.ccxt.com/en/latest/manual.html#position-structure>`
        """
        data = await self.get_account_data(params)
        # response sample inside `getAccountData` method
        self.targetAccount = self.safe_value(data, 0)
        positions = self.safe_value(self.targetAccount, 'positions', [])
        return self.parse_positions(positions, symbols)

    def parse_position(self, position, market=None):
        #
        #     {
        #         "marketCode": "SHIB-USD-SWAP-LIN",
        #         "baseAsset": "SHIB",
        #         "counterAsset": "USD",
        #         "position": "313546.0",
        #         "entryPrice": "0.00002161",
        #         "markPrice": "0.00002158",
        #         "positionPnl": "-0.009406380",
        #         "estLiquidationPrice": "0",
        #         "lastUpdatedAt": "1651342025876"
        #     }
        #
        #    but self.targetAccount has also like:
        #
        #     {
        #         "collateral": "28.297558",
        #         "notionalPositionSize": "6.82903188",
        #         "portfolioVarMargin": "0.682589",
        #         "riskRatio": "41.5216",
        #         "maintenanceMargin": "0.34129450",
        #         "marginRatio": "1.20",
        #         "liquidating": False,
        #         "feeTier": "0",
        #         "createdAt": "1651232948406"
        #     }
        #
        marketId = self.safe_string(position, 'marketCode')
        market = self.safe_market(marketId, market)
        contractsString = self.safe_string(position, 'position')
        timestamp = None  # self.safe_integer(position, 'lastUpdatedAt')
        side = 'long' if Precise.string_gt(contractsString, '0') else 'short'
        liquidationPriceString = self.safe_string(position, 'estLiquidationPrice')
        entryPriceString = self.safe_string(position, 'entryPrice')
        unrealizedPnlString = self.safe_string(position, 'positionPnl')
        markPriceString = self.safe_string(position, 'markPrice')
        return {
            'id': None,
            'symbol': market['symbol'],
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'initialMargin': None,
            'initialMarginPercentage': None,
            'maintenanceMargin': None,
            'maintenanceMarginPercentage': None,
            'entryPrice': self.parse_number(entryPriceString),
            'notional': None,
            'leverage': None,
            'unrealizedPnl': self.parse_number(unrealizedPnlString),
            'contracts': self.parse_number(contractsString),
            'contractSize': None,
            'marginRatio': None,
            'liquidationPrice': self.parse_number(liquidationPriceString),
            'markPrice': self.parse_number(markPriceString),
            'collateral': None,
            'marginMode': 'cross',  # each account is cross : https://coinflex.com/support/3-4-margin-and-risk-management/
            'side': side,
            'percentage': None,
            'info': position,
        }

    async def fetch_deposit_address(self, code, params={}):
        """
        fetch the deposit address for a currency associated with self account
        :param str code: unified currency code
        :param dict params: extra parameters specific to the coinflex api endpoint
        :returns dict: an `address structure <https://docs.ccxt.com/en/latest/manual.html#address-structure>`
        """
        await self.load_markets()
        currency = self.currency(code)
        request = {
            'asset': currency['id'],
        }
        networks = self.safe_value(self.options, 'networks', {})
        network = self.safe_string_upper(params, 'network')
        params = self.omit(params, 'network')
        networkId = self.safe_string(networks, network, network)
        if networkId is None:
            raise ExchangeError(self.id + ' fetchDepositAddress() requires a `network` parameter')
        request['network'] = networkId
        response = await self.privateGetV3DepositAddresses(self.extend(request, params))
        #
        #     {
        #         "success": True,
        #         "data": {
        #             "address": "0x5D561479d9665E490894822896c9c45Ea63007EE"
        #         }
        #     }
        #
        data = self.safe_value(response, 'data')
        address = self.safe_value(data, 'address')
        self.check_address(address)
        return {
            'currency': code,
            'address': address,
            'tag': None,
            'network': network,
            'info': response,
        }

    async def fetch_deposits(self, code=None, since=None, limit=None, params={}):
        """
        fetch all deposits made to an account
        :param str|None code: unified currency code
        :param int|None since: the earliest time in ms to fetch deposits for
        :param int|None limit: the maximum number of deposits structures to retrieve
        :param dict params: extra parameters specific to the coinflex api endpoint
        :returns [dict]: a list of `transaction structures <https://docs.ccxt.com/en/latest/manual.html#transaction-structure>`
        """
        await self.load_markets()
        currency = None
        request = {}
        if code is not None:
            currency = self.currency(code)
            request['asset'] = currency['id']
        if limit is not None:
            request['limit'] = limit
        request = self.set_start_end_times(request, since)
        data = None
        try:
            response = await self.privateGetV3Deposit(self.extend(request, params))
            #
            #     {
            #         "success": True,
            #         "data": [
            #             {
            #                 "id": "757475245433389059",
            #                 "asset": "USDT",
            #                 "network": "ERC20",
            #                 "address": "0x5D561479d9665E490894822896c9c45Ea63007EE",
            #                 "quantity": "28.33",
            #                 "status": "COMPLETED",
            #                 "txId": "0x6a92c8190b4b56a56fed2f9a8d0d7afd01843c28d0c0a8a5607b974b2fab8b4a",
            #                 "creditedAt": "1651233499800"
            #             }
            #         ]
            #     }
            #
            # Note, when there are no deposit records, you might get:
            #
            #     {
            #         "success": False,
            #         "code": "20001",
            #         "message": "result not found, please check your parameters"
            #     }
            #
            data = self.safe_value(response, 'data', [])
        except Exception as e:
            #
            # Note, when there are no deposit records, you might get:
            #
            #     {
            #         "success": False,
            #         "code": "20001",  # the code is not documented, so might be different
            #         "message": "result not found, please check your parameters"
            #     }
            #
            if self.last_json_response:
                code = self.safe_string(self.last_json_response, 'code')
                message = self.safe_string(self.last_json_response, 'message')
                if code == '20001' or message.find('result not found') >= 0:
                    data = []
        return self.parse_transactions(data, currency, since, limit, params)

    async def fetch_withdrawals(self, code=None, since=None, limit=None, params={}):
        """
        fetch all withdrawals made from an account
        :param str|None code: unified currency code
        :param int|None since: the earliest time in ms to fetch withdrawals for
        :param int|None limit: the maximum number of withdrawals structures to retrieve
        :param dict params: extra parameters specific to the coinflex api endpoint
        :returns [dict]: a list of `transaction structures <https://docs.ccxt.com/en/latest/manual.html#transaction-structure>`
        """
        await self.load_markets()
        currency = None
        request = {}
        if code is not None:
            currency = self.currency(code)
            request['asset'] = currency['id']
        if limit is not None:
            request['limit'] = limit
        request = self.set_start_end_times(request, since)
        data = None
        try:
            response = await self.privateGetV3Withdrawal(self.extend(request, params))
            #
            #     {
            #         "success": True,
            #         "data": [
            #             {
            #                 "id": "759031388269150209",
            #                 "asset": "USDT",
            #                 "network": "ERC20",
            #                 "address": "0xe8c2d75e0392e32f36e541b868D8AC3148A4DDf8",
            #                 "quantity": "18",
            #                 "fee": "8.799600000",
            #                 "status": "COMPLETED",
            #                 "txId": "0xc071cd34cb2f60135e709c26219b39523addbb3599818d98dcd5db2bf0115c17",
            #                 "requestedAt": "1651708397398",
            #                 "completedAt": "1651708835000"
            #             }
            #         ]
            #     }
            #
            data = self.safe_value(response, 'data', [])
        except Exception as e:
            #
            # Note, when there are no withdrawal records, you might get:
            #
            #     {
            #         "success": False,
            #         "code": "20001",
            #         "message": "result not found, please check your parameters"
            #     }
            #
            if self.last_json_response:
                code = self.safe_string(self.last_json_response, 'code')
                if code == '20001':
                    data = []
        return self.parse_transactions(data, currency, since, limit, params)

    async def fetch_withdrawal(self, id, code=None, params={}):
        """
        fetch data on a currency withdrawal via the withdrawal id
        :param str id: withdrawal id
        :param str|None code: unified currency code of the currency withdrawn, default is None
        :param dict params: extra parameters specific to the coinflex api endpoint
        :returns dict: a `transaction structure <https://docs.ccxt.com/en/latest/manual.html#transaction-structure>`
        """
        request = {
            'id': id,
        }
        withdrawals = await self.fetch_withdrawals(code, None, None, self.extend(request, params))
        return self.safe_value(withdrawals, 0)  # the target transaction will be the only in array

    def parse_transaction(self, transaction, currency=None):
        #
        # fetchDeposits
        #
        #     {
        #         "id": "757475245433389059",
        #         "txId": "0x6a92c8190b4b56a56fed2f9a8d0d7afd01843c28d0c0a8a5607b974b2fab8b5a",
        #         "asset": "USDT",
        #         "network": "ERC20",
        #         "address": "0x5D561479d9665E490894822896c9c45Ea63007EE",
        #         "quantity": "28.33",
        #         "status": "COMPLETED",
        #         "creditedAt": "1651233499800"
        #    }
        #
        # fetchWithdrawals
        #
        #     {
        #         "id": "651573911056351237",
        #         "txId": "38c09755bff75d33304a3cb6ee839fcb78bbb38b6e3e16586f20852cdec4883d",
        #         "asset": "USDT",
        #         "network": "ERC20",
        #         "address": "0x5D561479d9665E490899382896c9c45Ea63007EE",
        #         "quantity": "36",
        #         "fee": "5.2324",
        #         "status": "COMPLETED",
        #         "requestedAt": "1617940893000",
        #         "completedAt": "1617940921123"
        #     }
        #
        # withdraw
        #
        #     {
        #         "id": "759031388269150209",
        #         "asset": "USDT",
        #         "network": "ERC20",
        #         "address": "0xe8c2d73e0312e32f98e541b813D8EC3148A4BAd5",
        #         "quantity": "18",
        #         "externalFee": False,
        #         "fee": "8.7996",
        #         "status": "PENDING",
        #         "requestedAt": "1651708397366"
        #     }
        #
        isDeposit = ('creditedAt' in transaction)
        id = self.safe_string(transaction, 'id')
        txId = self.safe_string(transaction, 'txId')
        currencyId = self.safe_string(transaction, 'asset')
        currency = self.safe_currency(currencyId, currency)
        networkId = self.safe_string(transaction, 'network')
        networkCode = self.safe_string(self.options['networksByIds'], networkId, networkId)
        addressTo = self.safe_string(transaction, 'address')
        statusRaw = self.safe_string(transaction, 'status')
        status = self.parse_transaction_status(statusRaw)
        timestamp = self.safe_integer_2(transaction, 'creditedAt', 'requestedAt')
        type = 'deposit' if isDeposit else 'withdrawal'
        return {
            'id': id,
            'txid': txId,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'network': networkCode,
            'address': None,
            'addressTo': addressTo,
            'addressFrom': None,
            'tag': None,
            'tagTo': None,
            'tagFrom': None,
            'type': type,
            'amount': self.safe_number(transaction, 'quantity'),
            'currency': currency['code'],
            'status': status,
            'updated': self.safe_integer(transaction, 'completedAt'),
            'internal': None,
            'fee': self.safe_number(transaction, 'fee'),
            'info': transaction,
        }

    def parse_transaction_status(self, status):
        statuses = {
            'PROCESSING': 'pending',
            'PENDING': 'pending',
            'COMPLETED': 'ok',
            'FAILED': 'failed',
            'CANCELED': 'canceled',
            # 'ON HOLD': 'pending',
        }
        return self.safe_string(statuses, status, status)

    async def fetch_transfers(self, code=None, since=None, limit=None, params={}):
        """
        fetch a history of internal transfers made on an account
        :param str|None code: unified currency code of the currency transferred
        :param int|None since: the earliest time in ms to fetch transfers for
        :param int|None limit: the maximum number of  transfers structures to retrieve
        :param dict params: extra parameters specific to the coinflex api endpoint
        :returns [dict]: a list of `transfer structures <https://docs.ccxt.com/en/latest/manual.html#transfer-structure>`
        """
        await self.load_markets()
        currency = None
        request = {}
        if code is not None:
            currency = self.currency(code)
            request['asset'] = currency['id']
        if limit is not None:
            request['limit'] = limit
        request = self.set_start_end_times(request, since)
        response = await self.privateGetV3Transfer(self.extend(request, params))
        #
        #     {
        #         "success": True,
        #         "data": [
        #             {
        #                 "asset": "USDT",
        #                 "quantity": "5",
        #                 "fromAccount": "38432",
        #                 "toAccount": "38774",
        #                 "id": "758113170128764931",
        #                 "status": "COMPLETED",
        #                 "transferredAt": "1651428178967"
        #              }
        #          ]
        #     }
        #
        transfers = self.safe_value(response, 'data', [])
        return self.parse_transfers(transfers, currency, since, limit)

    def parse_transfer(self, transfer, currency=None):
        #
        # fetchTransfers
        #
        #     {
        #         "asset": "USDT",
        #         "quantity": "5",
        #         "fromAccount": "38432",
        #         "toAccount": "38774",
        #         "id": "758113170128764931",
        #         "status": "COMPLETED",
        #         "transferredAt": "1651428178967"
        #      }
        #
        # transfer
        #
        #     {
        #         "asset": "USDT",
        #         "quantity": "5",
        #         "fromAccount": "38422",
        #         "toAccount": "38776",
        #         "transferredAt": "1651704762775"
        #     }
        #
        currencyId = self.safe_string(transfer, 'asset')
        timestamp = self.safe_string(transfer, 'transferredAt')
        fromAccount = self.safe_string(transfer, 'fromAccount')
        toAccount = self.safe_string(transfer, 'toAccount')
        status = self.parse_transaction_status(self.safe_string(transfer, 'status'))
        # if case of 'transfer', as status field is not available, 'transferredAt' prop should mean that transfer was done
        if status is None and timestamp is not None:
            status = 'ok'
        return {
            'id': self.safe_string(transfer, 'id'),
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'currency': self.safe_currency_code(currencyId, currency),
            'amount': self.safe_number(transfer, 'quantity'),
            'fromAccount': fromAccount,
            'toAccount': toAccount,
            'status': status,
            'info': transfer,
        }

    async def build_order_request(self, market, type, side, amount, price=None, params={}):
        clientOrderId = self.safe_string(params, 'clientOrderId')
        maxCOI = '9223372036854775807'
        if (clientOrderId is not None) and Precise.string_gt(clientOrderId, maxCOI):
            raise InvalidOrder(self.id + ' createOrder() param clientOrderId should not exceed ' + maxCOI)
        orderType = self.convert_order_type(type)
        stopPrice = self.safe_number(params, 'stopPrice')
        limitPrice = self.safe_number(params, 'limitPrice')
        stopPriceIsDefined = stopPrice is not None
        orderTypeIsStop = orderType == 'STOP'
        isStopOrder = stopPriceIsDefined or orderTypeIsStop
        if orderTypeIsStop:
            if not stopPriceIsDefined:
                raise ArgumentsRequired(self.id + ' createOrder() requires params["stopPrice"] for stop orders')
            elif limitPrice is None and price is None:
                raise ArgumentsRequired(self.id + ' createOrder() requires "price" argument or params["limitPrice"] as a limit price for stop orders, as stop-market orders are not supported on self exchange')
        await self.load_markets()
        order = {
            'marketCode': market['id'],
            'side': self.convert_order_side(side),
            'orderType': self.convert_order_type(type),
            'quantity': self.amount_to_precision(market['symbol'], amount),
        }
        if isStopOrder:
            order['stopPrice'] = self.price_to_precision(market['symbol'], stopPrice)
            params = self.omit(params, 'stopPrice')
            order['orderType'] = 'STOP'
        # stop orders have separate field for limit price, so make further checks
        if type == 'limit' or isStopOrder:
            if isStopOrder:
                if limitPrice is None:
                    order['limitPrice'] = self.price_to_precision(market['symbol'], price)
            else:
                order['price'] = self.price_to_precision(market['symbol'], price)
        request = {
            'responseType': 'FULL',  # FULL, ACK
            'orders': [order],
        }
        return [request, params]

    def check_order_response_for_exception(self, response, firstResult):
        success = self.safe_value(firstResult, 'success')
        if success == 'false':
            message = self.safe_string(firstResult, 'message')
            code = self.safe_string(firstResult, 'code')
            body = self.id + ' ' + self.json(response)
            self.throw_exactly_matched_exception(self.exceptions['exact'], code, body)
            self.throw_broadly_matched_exception(self.exceptions['broad'], message, body)
            raise ExchangeError(body)

    async def create_order(self, symbol, type, side, amount, price=None, params={}):
        """
        create a trade order
        :param str symbol: unified symbol of the market to create an order in
        :param str type: 'market' or 'limit'
        :param str side: 'buy' or 'sell'
        :param float amount: how much of currency you want to trade in units of base currency
        :param float|None price: the price at which the order is to be fullfilled, in units of the quote currency, ignored in market orders
        :param dict params: extra parameters specific to the coinflex api endpoint
        :returns dict: an `order structure <https://docs.ccxt.com/en/latest/manual.html#order-structure>`
        """
        market = self.market(symbol)
        self.check_order_arguments(market, type, side, amount, price, params)
        request, query = await self.build_order_request(market, type, side, amount, price, params)
        response = await self.privatePostV2OrdersPlace(self.extend(request, query))
        #
        #     {
        #         "event": "placeOrder",
        #         "timestamp": "1651619029297",
        #         "accountId": "38422",
        #         "data": [
        #           {
        #             "success": "true",
        #             "timestamp": "1651620006856",
        #             "clientOrderId": "1651620006827000",
        #             "orderId": "1002124426084",
        #             "price": "0.65",
        #             "quantity": "8.0",
        #             "side": "BUY",
        #             "status": "FILLED",  # OPEN, FILLED
        #             "marketCode": "XRP-USD",
        #             "timeInForce": "GTC",
        #             "matchId": "5636974433720947783",  # zero if not filled
        #             "lastTradedPrice": "0.6028",  # field not present if order doesn't have any fills
        #             "matchQuantity": "8.0",  # field not present if order doesn't have any fills
        #             "orderMatchType": "TAKER",  # field not present if order doesn't have any fills
        #             "remainQuantity": "0.0",  # field not present if order doesn't have any fills
        #             "notice": "OrderMatched",
        #             "orderType": "LIMIT",
        #             "fees": "0.003857920",  # field not present if order doesn't have any fills
        #             "feeInstrumentId": "USD",  # field not present if order doesn't have any fills
        #             "isTriggered": "false"
        #           }
        #         ]
        #     }
        #
        # Note, for failed order, the order-object might be like:
        #
        #     {
        #          "success": "false",
        #          "timestamp": "1651619029297",
        #          "code": "710003",
        #          "message": "FAILED sanity bound check as price(5.2) >  upper bound(4.1)",
        #          "price": "4.000",
        #          "quantity": "3.0",
        #          "side": "BUY",
        #          "marketCode": "BAND-USD",
        #          "orderType": "LIMIT"
        #     }
        #
        data = self.safe_value(response, 'data', [])
        firstOrder = self.safe_value(data, 0, {})
        self.check_order_response_for_exception(response, firstOrder)
        return self.parse_order(firstOrder, market)

    async def edit_order(self, id, symbol, type, side, amount=None, price=None, params={}):
        market = self.market(symbol)
        request, query = await self.build_order_request(market, type, side, amount, price, params)
        request['orders'][0]['orderId'] = id
        response = await self.privatePostV2OrdersModify(self.extend(request, query))
        #
        #     {
        #         "event": "modifyOrder",
        #         "timestamp": "1651695117070",
        #         "accountId": "38422",
        #         "data": [
        #             {
        #                 "success": "true",
        #                 "timestamp": "1651696843859",
        #                 "clientOrderId": "1651692616185",
        #                 "orderId": "1002128500938",
        #                 "price": "0.5600",
        #                 "quantity": "8.0",
        #                 "side": "BUY",
        #                 "marketCode": "XRP-USD",
        #                 "timeInForce": "GTC",
        #                 "notice": "OrderModified",
        #                 "orderType": "LIMIT"
        #             }
        #         ]
        #     }
        #
        #
        # Note, for inexistent order-id, the order-object might be like:
        #
        #             {
        #                 "success": "false",
        #                 "timestamp": "1651695117070",
        #                 "code": "40035",
        #                 "message": "Open order not found with clientOrderId or orderId",
        #                 "price": "0.56",
        #                 "quantity": "8",
        #                 "side": "BUY",
        #                 "marketCode": "XRP-USD",
        #                 "orderType": "LIMIT"
        #             }
        #
        data = self.safe_value(response, 'data', [])
        firstOrder = self.safe_value(data, 0, {})
        self.check_order_response_for_exception(response, firstOrder)
        return self.parse_order(firstOrder, market)

    async def cancel_orders(self, ids, symbol=None, params={}):
        """
        cancel multiple orders
        :param [str] ids: order ids
        :param str symbol: unified market symbol
        :param dict params: extra parameters specific to the coinflex api endpoint
        :returns dict: an list of `order structures <https://docs.ccxt.com/en/latest/manual.html#order-structure>`
        """
        if symbol is None:
            raise ArgumentsRequired(self.id + ' cancelOrders() requires a symbol argument')
        market = self.market(symbol)
        await self.load_markets()
        request = {
            'responseType': 'FULL',  # FULL, ACK
            'orders': [],
        }
        for i in range(0, len(ids)):
            request['orders'].append({
                'marketCode': market['id'],
                'orderId': ids[i],
            })
        response = await self.privateDeleteV2OrdersCancel(self.extend(request, params))
        #
        #     {
        #         "event": "cancelOrder",
        #         "timestamp": "1651697925019",
        #         "accountId": "38422",
        #         "data": [
        #             {
        #                 "success": "true",
        #                 "timestamp": "1651699447397",
        #                 "clientOrderId": "1651692616185",
        #                 "orderId": "1002128500938",
        #                 "price": "0.56",
        #                 "quantity": "8.0",
        #                 "side": "BUY",
        #                 "status": "CANCELED_BY_USER",
        #                 "marketCode": "XRP-USD",
        #                 "timeInForce": "GTC",
        #                 "matchId": "0",
        #                 "remainQuantity": "8.0",
        #                 "notice": "OrderClosed",
        #                 "orderType": "LIMIT",
        #                 "isTriggered": "false"
        #             }
        #         ]
        #     }
        #
        # Note, for inexistent order-id, the order-object might be like:
        #
        #             {
        #                 "success": "false",
        #                 "timestamp": "1651697925019",
        #                 "code": "40035",
        #                 "message": "Open order not found with clientOrderId or orderId",
        #                 "orderId": "1002128500938",
        #                 "marketCode": "XRP/USD"
        #             }
        #
        data = self.safe_value(response, 'data', [])
        return self.parse_orders(data, market)

    async def cancel_order(self, id, symbol=None, params={}):
        """
        cancels an open order
        :param str id: order id
        :param str|None symbol: unified symbol of the market the order was made in
        :param dict params: extra parameters specific to the coinflex api endpoint
        :returns dict: An `order structure <https://docs.ccxt.com/en/latest/manual.html#order-structure>`
        """
        orders = await self.cancel_orders([id], symbol, params)
        return self.safe_value(orders, 0)

    async def cancel_all_orders(self, symbol=None, params={}):
        """
        cancel all open orders
        :param str|None symbol: unified market symbol, only orders in the market of self symbol are cancelled when symbol is not None
        :param dict params: extra parameters specific to the coinflex api endpoint
        :returns [dict]: a list of `order structures <https://docs.ccxt.com/en/latest/manual.html#order-structure>`
        """
        await self.load_markets()
        market = None
        request = {}
        method = 'privateDeleteV2CancelOrders'
        if symbol is not None:
            market = self.market(symbol)
            request['marketCode'] = market['id']
            method = 'privateDeleteV2CancelOrdersMarketCode'
        response = await getattr(self, method)(self.extend(request, params))
        #
        #     {
        #         "event": "orders",
        #         "timestamp": "1651651758625",
        #         "accountId": "38422",
        #         "data": {
        #             "marketCode": "XRP-USD",
        #             "msg": "All open orders for the specified market have been queued for cancellation"
        #         }
        #     }
        #
        # Note1: when fired without symbol, the 'data' object only contains a msg property with '{"msg":"All open orders for the account have been queued for cancellation"}'
        # Note2: if there has been no orders pending, then data property will be null.
        #
        return response

    async def transfer(self, code, amount, fromAccount, toAccount, params={}):
        """
        transfer currency internally between wallets on the same account
        :param str code: unified currency code
        :param float amount: amount to transfer
        :param str fromAccount: account to transfer from
        :param str toAccount: account to transfer to
        :param dict params: extra parameters specific to the coinflex api endpoint
        :returns dict: a `transfer structure <https://docs.ccxt.com/en/latest/manual.html#transfer-structure>`
        """
        await self.load_markets()
        currency = self.currency(code)
        request = {
            'asset': currency['id'],
            'quantity': amount,
            'fromAccount': fromAccount,
            'toAccount': toAccount,
        }
        response = await self.privatePostV3Transfer(self.extend(request, params))
        #
        #     {
        #         "success": True,
        #         "data": {
        #             "asset": "USDT",
        #             "quantity": "5",
        #             "fromAccount": "38422",
        #             "toAccount": "38776",
        #             "transferredAt": "1651704762775"
        #         }
        #     }
        #
        data = self.safe_value(response, 'data', {})
        return self.parse_transfer(data, currency)

    async def withdraw(self, code, amount, address, tag=None, params={}):
        """
        make a withdrawal
        :param str code: unified currency code
        :param float amount: the amount to withdraw
        :param str address: the address to withdraw to
        :param str|None tag:
        :param dict params: extra parameters specific to the coinflex api endpoint
        :returns dict: a `transaction structure <https://docs.ccxt.com/en/latest/manual.html#transaction-structure>`
        """
        await self.load_markets()
        currency = self.currency(code)
        twoFaCode = self.safe_string(params, 'code')
        if twoFaCode is None:
            raise ArgumentsRequired(self.id + ' withdraw() requires two-factor "code" parameter for withdrawals')
        networks = self.safe_value(self.options, 'networks', {})
        networkName = self.safe_string_upper(params, 'network')
        networkId = self.safe_string_upper(networks, networkName, networkName)
        request = {
            'asset': currency['id'],
            'network': networkId,
            'quantity': self.currency_to_precision(code, amount),
            'address': address,
            # below are required params by self time
            'externalFee': False,
        }
        if self.twofa is not None:
            request['tfaType'] = 'GOOGLE'
            request['code'] = self.oath()
        response = await self.privatePostV3Withdrawal(self.extend(request, params))
        #
        #     {
        #         "success": True,
        #         "data": {
        #             "id": "759031388269150209",
        #             "asset": "USDT",
        #             "network": "ERC20",
        #             "address": "0xe8c2d73e0312e32f98e541b813D8EC3148A4BAd5",
        #             "quantity": "18",
        #             "externalFee": False,
        #             "fee": "8.7996",
        #             "status": "PENDING",
        #             "requestedAt": "1651708397366"
        #         }
        #     }
        #
        data = self.safe_value(response, 'data')
        return self.parse_transaction(data, currency)

    async def fetch_transaction_fee(self, code, params={}):
        """
        fetch the fee for a transaction
        :param str code: unified currency code
        :param dict params: extra parameters specific to the coinflex api endpoint
        :param str params['networkName']: the protocol for a transaction
        :param str params['address']: withdrawal address
        :returns dict: a `fee structure <https://docs.ccxt.com/en/latest/manual.html#fee-structure>`
        """
        networkName = self.safe_string_upper(params, 'network')
        if networkName is None:
            raise ArgumentsRequired(self.id + ' fetchTransactionFee() requires "network"  parameter')
        address = self.safe_string(params, 'address')
        if address is None:
            raise ArgumentsRequired(self.id + ' fetchTransactionFee() requires recipient "address"  param to calculate fee')
        amount = self.safe_number(params, 'quantity')
        params = self.omit(params, 'quantity')
        if address is None:
            raise ArgumentsRequired(self.id + ' fetchTransactionFee() requires "quantity"  param to calculate fee')
        await self.load_markets()
        currency = self.currency(code)
        networks = self.safe_value(self.options, 'networks', {})
        networkId = self.safe_string_upper(networks, networkName, networkName)
        request = {
            'asset': currency['id'],
            'network': networkId,
            'quantity': self.currency_to_precision(code, amount),
        }
        response = await self.privateGetV3WithdrawalFee(self.extend(request, params))
        #
        #     {
        #         "success": True,
        #         "data": {
        #             "asset": "USDT",
        #             "network": "ERC20",
        #             "address": "0x811bFBd9CfBB503c592842c11b89D2b1D65db32F",
        #             "quantity": "30",
        #             "externalFee": False,
        #             "estimatedFee": "9.2985"
        #         }
        #     }
        #
        data = self.safe_value(response, 'data', {})
        withdrawFees = {}
        withdrawFees[code] = self.safe_number(data, 'estimatedFee')
        return {
            'withdraw': withdrawFees,
            'deposit': {},
            'info': response,
        }

    def nonce(self):
        return self.milliseconds()

    def sign(self, path, api='public', method='GET', params={}, headers=None, body=None):
        finalPath, query = self.resolve_path(path, params)
        baseUrl = self.implode_hostname(self.urls['api']['rest'])
        url = baseUrl + '/' + finalPath
        encodedParams = ''
        isGetRequest = (method == 'GET')
        if query:
            encodedParams = self.rawencode(query)
            if isGetRequest:
                url += '?' + encodedParams
        if api == 'private':
            self.check_required_credentials()
            nonce = str(self.nonce())
            datetime = self.ymdhms(self.milliseconds(), 'T')
            baseUrlTrimmed = baseUrl.replace('https://', '')
            auth = datetime + "\n" + nonce + "\n" + method + "\n" + baseUrlTrimmed + "\n" + '/' + finalPath + "\n"  # eslint-disable-line quotes
            if isGetRequest:
                auth += encodedParams
            else:
                jsonified = self.json(query)
                auth += jsonified
                body = jsonified
            signature = self.hmac(self.encode(auth), self.encode(self.secret), hashlib.sha256, 'base64')
            headers = {}
            headers['Content-Type'] = 'application/json'
            headers['AccessKey'] = self.apiKey
            headers['Timestamp'] = datetime
            headers['Signature'] = signature
            headers['Nonce'] = nonce
        return {'url': url, 'method': method, 'body': body, 'headers': headers}

    def handle_errors(self, code, reason, url, method, headers, body, response, requestHeaders, requestBody):
        if response is None:
            return
        #
        # Success
        #
        #     {
        #         "success": "true"
        #     }
        #
        #   or
        #
        #     {
        #         "success":true,
        #         "data":[...]
        #     }
        #
        #   or
        #
        #     {
        #         "event": "publicTrades",
        #         "timestamp": "1651312416050",
        #         "marketCode": "BTC-USD",
        #         "data": [
        #             {
        #                 "matchId": "304734619669458401",
        #                 "matchQuantity": "0.012",
        #                 "matchPrice": "38673",
        #                 "side": "BUY",
        #                 "matchTimestamp": "1651281046230"
        #             },
        #         ]
        #     }
        #
        # Error
        #
        #     {
        #         "success":false,
        #         "code":"40001",
        #         "message":"no result, please check your parameters"
        #     }
        #
        responseCode = self.safe_string(response, 'code')
        if responseCode is not None and responseCode != '0':
            feedback = self.id + ' ' + body
            self.throw_exactly_matched_exception(self.exceptions['exact'], responseCode, feedback)
            self.throw_broadly_matched_exception(self.exceptions['broad'], body, feedback)
            raise ExchangeError(feedback)
