"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RequestAuthorizer = exports.TokenAuthorizer = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("../apigateway.generated");
const authorizer_1 = require("../authorizer");
class LambdaAuthorizer extends authorizer_1.Authorizer {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.handler = props.handler;
        this.role = props.assumeRole;
        if (props.resultsCacheTtl && ((_a = props.resultsCacheTtl) === null || _a === void 0 ? void 0 : _a.toSeconds()) > 3600) {
            throw new Error('Lambda authorizer property \'resultsCacheTtl\' must not be greater than 3600 seconds (1 hour)');
        }
    }
    /**
     * Attaches this authorizer to a specific REST API.
     * @internal
     */
    _attachToApi(restApi) {
        if (this.restApiId && this.restApiId !== restApi.restApiId) {
            throw new Error('Cannot attach authorizer to two different rest APIs');
        }
        this.restApiId = restApi.restApiId;
    }
    /**
     * Sets up the permissions necessary for the API Gateway service to invoke the Lambda function.
     */
    setupPermissions() {
        if (!this.role) {
            this.handler.addPermission(`${this.node.uniqueId}:Permissions`, {
                principal: new iam.ServicePrincipal('apigateway.amazonaws.com'),
                sourceArn: this.authorizerArn,
            });
        }
        else if (this.role instanceof iam.Role) { // i.e. not imported
            this.role.attachInlinePolicy(new iam.Policy(this, 'authorizerInvokePolicy', {
                statements: [
                    new iam.PolicyStatement({
                        resources: [this.handler.functionArn],
                        actions: ['lambda:InvokeFunction'],
                    }),
                ],
            }));
        }
    }
    /**
     * Returns a token that resolves to the Rest Api Id at the time of synthesis.
     * Throws an error, during token resolution, if no RestApi is attached to this authorizer.
     */
    lazyRestApiId() {
        return core_1.Lazy.stringValue({
            produce: () => {
                if (!this.restApiId) {
                    throw new Error(`Authorizer (${this.node.path}) must be attached to a RestApi`);
                }
                return this.restApiId;
            },
        });
    }
}
/**
 * Token based lambda authorizer that recognizes the caller's identity as a bearer token,
 * such as a JSON Web Token (JWT) or an OAuth token.
 * Based on the token, authorization is performed by a lambda function.
 *
 * @resource AWS::ApiGateway::Authorizer
 */
class TokenAuthorizer extends LambdaAuthorizer {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id, props);
        const restApiId = this.lazyRestApiId();
        const resource = new apigateway_generated_1.CfnAuthorizer(this, 'Resource', {
            name: (_a = props.authorizerName) !== null && _a !== void 0 ? _a : this.node.uniqueId,
            restApiId,
            type: 'TOKEN',
            authorizerUri: lambdaAuthorizerArn(props.handler),
            authorizerCredentials: (_b = props.assumeRole) === null || _b === void 0 ? void 0 : _b.roleArn,
            authorizerResultTtlInSeconds: (_c = props.resultsCacheTtl) === null || _c === void 0 ? void 0 : _c.toSeconds(),
            identitySource: props.identitySource || 'method.request.header.Authorization',
            identityValidationExpression: props.validationRegex,
        });
        this.authorizerId = resource.ref;
        this.authorizerArn = core_1.Stack.of(this).formatArn({
            service: 'execute-api',
            resource: restApiId,
            resourceName: `authorizers/${this.authorizerId}`,
        });
        this.setupPermissions();
    }
}
exports.TokenAuthorizer = TokenAuthorizer;
/**
 * Request-based lambda authorizer that recognizes the caller's identity via request parameters,
 * such as headers, paths, query strings, stage variables, or context variables.
 * Based on the request, authorization is performed by a lambda function.
 *
 * @resource AWS::ApiGateway::Authorizer
 */
class RequestAuthorizer extends LambdaAuthorizer {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id, props);
        if ((props.resultsCacheTtl === undefined || props.resultsCacheTtl.toSeconds() !== 0) && props.identitySources.length === 0) {
            throw new Error('At least one Identity Source is required for a REQUEST-based Lambda authorizer if caching is enabled.');
        }
        const restApiId = this.lazyRestApiId();
        const resource = new apigateway_generated_1.CfnAuthorizer(this, 'Resource', {
            name: (_a = props.authorizerName) !== null && _a !== void 0 ? _a : this.node.uniqueId,
            restApiId,
            type: 'REQUEST',
            authorizerUri: lambdaAuthorizerArn(props.handler),
            authorizerCredentials: (_b = props.assumeRole) === null || _b === void 0 ? void 0 : _b.roleArn,
            authorizerResultTtlInSeconds: (_c = props.resultsCacheTtl) === null || _c === void 0 ? void 0 : _c.toSeconds(),
            identitySource: props.identitySources.map(is => is.toString()).join(','),
        });
        this.authorizerId = resource.ref;
        this.authorizerArn = core_1.Stack.of(this).formatArn({
            service: 'execute-api',
            resource: restApiId,
            resourceName: `authorizers/${this.authorizerId}`,
        });
        this.setupPermissions();
    }
}
exports.RequestAuthorizer = RequestAuthorizer;
/**
 * constructs the authorizerURIArn.
 */
function lambdaAuthorizerArn(handler) {
    return `arn:${core_1.Stack.of(handler).partition}:apigateway:${core_1.Stack.of(handler).region}:lambda:path/2015-03-31/functions/${handler.functionArn}/invocations`;
}
//# sourceMappingURL=data:application/json;base64,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