"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Deployment = void 0;
const crypto = require("crypto");
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("./apigateway.generated");
const restapi_1 = require("./restapi");
/**
 * A Deployment of a REST API.
 *
 * An immutable representation of a RestApi resource that can be called by users
 * using Stages. A deployment must be associated with a Stage for it to be
 * callable over the Internet.
 *
 * Normally, you don't need to define deployments manually. The RestApi
 * construct manages a Deployment resource that represents the latest model. It
 * can be accessed through `restApi.latestDeployment` (unless `deploy: false` is
 * set when defining the `RestApi`).
 *
 * If you manually define this resource, you will need to know that since
 * deployments are immutable, as long as the resource's logical ID doesn't
 * change, the deployment will represent the snapshot in time in which the
 * resource was created. This means that if you modify the RestApi model (i.e.
 * add methods or resources), these changes will not be reflected unless a new
 * deployment resource is created.
 *
 * To achieve this behavior, the method `addToLogicalId(data)` can be used to
 * augment the logical ID generated for the deployment resource such that it
 * will include arbitrary data. This is done automatically for the
 * `restApi.latestDeployment` deployment.
 *
 * Furthermore, since a deployment does not reference any of the REST API
 * resources and methods, CloudFormation will likely provision it before these
 * resources are created, which means that it will represent a "half-baked"
 * model. Use the `node.addDependency(dep)` method to circumvent that. This is done
 * automatically for the `restApi.latestDeployment` deployment.
 */
class Deployment extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.resource = new LatestDeploymentResource(this, 'Resource', {
            description: props.description,
            restApi: props.api,
        });
        if (props.retainDeployments) {
            this.resource.applyRemovalPolicy(core_1.RemovalPolicy.RETAIN);
        }
        this.api = props.api;
        this.deploymentId = core_1.Lazy.stringValue({ produce: () => this.resource.ref });
        if (props.api instanceof restapi_1.RestApiBase) {
            props.api._attachDeployment(this);
        }
    }
    /**
     * Adds a component to the hash that determines this Deployment resource's
     * logical ID.
     *
     * This should be called by constructs of the API Gateway model that want to
     * invalidate the deployment when their settings change. The component will
     * be resolve()ed during synthesis so tokens are welcome.
     */
    addToLogicalId(data) {
        this.resource.addToLogicalId(data);
    }
    /**
     * Quoting from CloudFormation's docs:
     *
     *   If you create an AWS::ApiGateway::RestApi resource and its methods (using
     *   AWS::ApiGateway::Method) in the same template as your deployment, the
     *   deployment must depend on the RestApi's methods. To create a dependency,
     *   add a DependsOn attribute to the deployment. If you don't, AWS
     *   CloudFormation creates the deployment right after it creates the RestApi
     *   resource that doesn't contain any methods, and AWS CloudFormation
     *   encounters the following error: The REST API doesn't contain any methods.
     *
     * @param method The method to add as a dependency of the deployment
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-deployment.html
     * @see https://github.com/aws/aws-cdk/pull/6165
     * @internal
     */
    _addMethodDependency(method) {
        // adding a dependency between the constructs using `node.addDependency()`
        // will create additional dependencies between `AWS::ApiGateway::Deployment`
        // and the `AWS::Lambda::Permission` resources (children under Method),
        // causing cyclic dependency errors. Hence, falling back to declaring
        // dependencies between the underlying CfnResources.
        this.node.addDependency(method.node.defaultChild);
    }
}
exports.Deployment = Deployment;
class LatestDeploymentResource extends apigateway_generated_1.CfnDeployment {
    constructor(scope, id, props) {
        super(scope, id, {
            description: props.description,
            restApiId: props.restApi.restApiId,
        });
        this.hashComponents = new Array();
        this.api = props.restApi;
        this.originalLogicalId = this.stack.getLogicalId(this);
        this.overrideLogicalId(core_1.Lazy.stringValue({ produce: () => this.calculateLogicalId() }));
    }
    /**
     * Allows adding arbitrary data to the hashed logical ID of this deployment.
     * This can be used to couple the deployment to the API Gateway model.
     */
    addToLogicalId(data) {
        // if the construct is locked, it means we are already synthesizing and then
        // we can't modify the hash because we might have already calculated it.
        if (this.node.locked) {
            throw new Error('Cannot modify the logical ID when the construct is locked');
        }
        this.hashComponents.push(data);
    }
    calculateLogicalId() {
        const hash = [...this.hashComponents];
        if (this.api instanceof restapi_1.RestApi || this.api instanceof restapi_1.SpecRestApi) { // Ignore IRestApi that are imported
            // Add CfnRestApi to the logical id so a new deployment is triggered when any of its properties change.
            const cfnRestApiCF = this.api.node.defaultChild._toCloudFormation();
            hash.push(this.stack.resolve(cfnRestApiCF));
        }
        let lid = this.originalLogicalId;
        // if hash components were added to the deployment, we use them to calculate
        // a logical ID for the deployment resource.
        if (hash.length > 0) {
            const md5 = crypto.createHash('md5');
            hash.map(x => this.stack.resolve(x)).forEach(c => md5.update(JSON.stringify(c)));
            lid += md5.digest('hex');
        }
        return lid;
    }
}
//# sourceMappingURL=data:application/json;base64,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