"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConnectionType = exports.PassthroughBehavior = exports.IntegrationType = exports.ContentHandling = exports.Integration = void 0;
/**
 * Base class for backend integrations for an API Gateway method.
 *
 * Use one of the concrete classes such as `MockIntegration`, `AwsIntegration`, `LambdaIntegration`
 * or implement on your own by specifying the set of props.
 */
class Integration {
    constructor(props) {
        this.props = props;
        const options = this.props.options || {};
        if (options.credentialsPassthrough !== undefined && options.credentialsRole !== undefined) {
            throw new Error('\'credentialsPassthrough\' and \'credentialsRole\' are mutually exclusive');
        }
        if (options.connectionType === ConnectionType.VPC_LINK && options.vpcLink === undefined) {
            throw new Error('\'connectionType\' of VPC_LINK requires \'vpcLink\' prop to be set');
        }
        if (options.connectionType === ConnectionType.INTERNET && options.vpcLink !== undefined) {
            throw new Error('cannot set \'vpcLink\' where \'connectionType\' is INTERNET');
        }
    }
    /**
     * Can be overridden by subclasses to allow the integration to interact with the method
     * being integrated, access the REST API object, method ARNs, etc.
     */
    bind(_method) {
        return {
            options: this.props.options,
            type: this.props.type,
            uri: this.props.uri,
            integrationHttpMethod: this.props.integrationHttpMethod,
        };
    }
}
exports.Integration = Integration;
var ContentHandling;
(function (ContentHandling) {
    /**
     * Converts a request payload from a base64-encoded string to a binary blob.
     */
    ContentHandling["CONVERT_TO_BINARY"] = "CONVERT_TO_BINARY";
    /**
     * Converts a request payload from a binary blob to a base64-encoded string.
     */
    ContentHandling["CONVERT_TO_TEXT"] = "CONVERT_TO_TEXT";
})(ContentHandling = exports.ContentHandling || (exports.ContentHandling = {}));
var IntegrationType;
(function (IntegrationType) {
    /**
     * For integrating the API method request with an AWS service action,
     * including the Lambda function-invoking action. With the Lambda
     * function-invoking action, this is referred to as the Lambda custom
     * integration. With any other AWS service action, this is known as AWS
     * integration.
     */
    IntegrationType["AWS"] = "AWS";
    /**
     * For integrating the API method request with the Lambda function-invoking
     * action with the client request passed through as-is. This integration is
     * also referred to as the Lambda proxy integration
     */
    IntegrationType["AWS_PROXY"] = "AWS_PROXY";
    /**
     * For integrating the API method request with an HTTP endpoint, including a
     * private HTTP endpoint within a VPC. This integration is also referred to
     * as the HTTP custom integration.
     */
    IntegrationType["HTTP"] = "HTTP";
    /**
     * For integrating the API method request with an HTTP endpoint, including a
     * private HTTP endpoint within a VPC, with the client request passed
     * through as-is. This is also referred to as the HTTP proxy integration
     */
    IntegrationType["HTTP_PROXY"] = "HTTP_PROXY";
    /**
     * For integrating the API method request with API Gateway as a "loop-back"
     * endpoint without invoking any backend.
     */
    IntegrationType["MOCK"] = "MOCK";
})(IntegrationType = exports.IntegrationType || (exports.IntegrationType = {}));
var PassthroughBehavior;
(function (PassthroughBehavior) {
    /**
     * Passes the request body for unmapped content types through to the
     * integration back end without transformation.
     */
    PassthroughBehavior["WHEN_NO_MATCH"] = "WHEN_NO_MATCH";
    /**
     * Rejects unmapped content types with an HTTP 415 'Unsupported Media Type'
     * response
     */
    PassthroughBehavior["NEVER"] = "NEVER";
    /**
     * Allows pass-through when the integration has NO content types mapped to
     * templates. However if there is at least one content type defined,
     * unmapped content types will be rejected with the same 415 response.
     */
    PassthroughBehavior["WHEN_NO_TEMPLATES"] = "WHEN_NO_TEMPLATES";
})(PassthroughBehavior = exports.PassthroughBehavior || (exports.PassthroughBehavior = {}));
var ConnectionType;
(function (ConnectionType) {
    /**
     * For connections through the public routable internet
     */
    ConnectionType["INTERNET"] = "INTERNET";
    /**
     * For private connections between API Gateway and a network load balancer in a VPC
     */
    ConnectionType["VPC_LINK"] = "VPC_LINK";
})(ConnectionType = exports.ConnectionType || (exports.ConnectionType = {}));
//# sourceMappingURL=data:application/json;base64,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