"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaIntegration = void 0;
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const aws_1 = require("./aws");
/**
 * Integrates an AWS Lambda function to an API Gateway method.
 *
 * @example
 *
 *    const handler = new lambda.Function(this, 'MyFunction', ...);
 *    api.addMethod('GET', new LambdaIntegration(handler));
 *
 */
class LambdaIntegration extends aws_1.AwsIntegration {
    constructor(handler, options = {}) {
        const proxy = options.proxy === undefined ? true : options.proxy;
        super({
            proxy,
            service: 'lambda',
            path: `2015-03-31/functions/${handler.functionArn}/invocations`,
            options,
        });
        this.handler = handler;
        this.enableTest = options.allowTestInvoke === undefined ? true : false;
    }
    bind(method) {
        const bindResult = super.bind(method);
        const principal = new iam.ServicePrincipal('apigateway.amazonaws.com');
        const desc = `${method.api.node.uniqueId}.${method.httpMethod}.${method.resource.path.replace(/\//g, '.')}`;
        this.handler.addPermission(`ApiPermission.${desc}`, {
            principal,
            scope: method,
            sourceArn: core_1.Lazy.stringValue({ produce: () => method.methodArn }),
        });
        // add permission to invoke from the console
        if (this.enableTest) {
            this.handler.addPermission(`ApiPermission.Test.${desc}`, {
                principal,
                scope: method,
                sourceArn: method.testMethodArn,
            });
        }
        let functionName;
        if (this.handler instanceof lambda.Function) {
            // if not imported, extract the name from the CFN layer to reach
            // the literal value if it is given (rather than a token)
            functionName = this.handler.node.defaultChild.functionName;
        }
        else {
            // imported, just take the function name.
            functionName = this.handler.functionName;
        }
        let deploymentToken;
        if (!core_1.Token.isUnresolved(functionName)) {
            deploymentToken = JSON.stringify({ functionName });
        }
        return {
            ...bindResult,
            deploymentToken,
        };
    }
}
exports.LambdaIntegration = LambdaIntegration;
//# sourceMappingURL=data:application/json;base64,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