"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Model = exports.ErrorModel = exports.EmptyModel = void 0;
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("./apigateway.generated");
const restapi_1 = require("./restapi");
const util = require("./util");
/**
 * Represents a reference to a REST API's Empty model, which is available
 * as part of the model collection by default. This can be used for mapping
 * JSON responses from an integration to what is returned to a client,
 * where strong typing is not required. In the absence of any defined
 * model, the Empty model will be used to return the response payload
 * unmapped.
 *
 * Definition
 * {
 *   "$schema" : "http://json-schema.org/draft-04/schema#",
 *   "title" : "Empty Schema",
 *   "type" : "object"
 * }
 *
 * @see https://docs.amazonaws.cn/en_us/apigateway/latest/developerguide/models-mappings.html#models-mappings-models
 * @deprecated You should use @see Model.EMPTY_MODEL
 */
class EmptyModel {
    constructor() {
        this.modelId = 'Empty';
    }
}
exports.EmptyModel = EmptyModel;
/**
 * Represents a reference to a REST API's Error model, which is available
 * as part of the model collection by default. This can be used for mapping
 * error JSON responses from an integration to a client, where a simple
 * generic message field is sufficient to map and return an error payload.
 *
 * Definition
 * {
 *   "$schema" : "http://json-schema.org/draft-04/schema#",
 *   "title" : "Error Schema",
 *   "type" : "object",
 *   "properties" : {
 *     "message" : { "type" : "string" }
 *   }
 * }
 * @deprecated You should use @see Model.ERROR_MODEL
 */
class ErrorModel {
    constructor() {
        this.modelId = 'Error';
    }
}
exports.ErrorModel = ErrorModel;
class Model extends core_1.Resource {
    constructor(scope, id, props) {
        var _a;
        super(scope, id, {
            physicalName: props.modelName,
        });
        const modelProps = {
            name: this.physicalName,
            restApiId: props.restApi.restApiId,
            contentType: (_a = props.contentType) !== null && _a !== void 0 ? _a : 'application/json',
            description: props.description,
            schema: util.JsonSchemaMapper.toCfnJsonSchema(props.schema),
        };
        const resource = new apigateway_generated_1.CfnModel(this, 'Resource', modelProps);
        this.modelId = this.getResourceNameAttribute(resource.ref);
        const deployment = (props.restApi instanceof restapi_1.RestApi) ? props.restApi.latestDeployment : undefined;
        if (deployment) {
            deployment.node.addDependency(resource);
            deployment.addToLogicalId({ model: modelProps });
        }
    }
    static fromModelName(scope, id, modelName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.modelId = modelName;
            }
        }
        return new Import(scope, id);
    }
}
exports.Model = Model;
/**
 * Represents a reference to a REST API's Error model, which is available
 * as part of the model collection by default. This can be used for mapping
 * error JSON responses from an integration to a client, where a simple
 * generic message field is sufficient to map and return an error payload.
 *
 * Definition
 * {
 *   "$schema" : "http://json-schema.org/draft-04/schema#",
 *   "title" : "Error Schema",
 *   "type" : "object",
 *   "properties" : {
 *     "message" : { "type" : "string" }
 *   }
 * }
 */
Model.ERROR_MODEL = new ErrorModel();
/**
 * Represents a reference to a REST API's Empty model, which is available
 * as part of the model collection by default. This can be used for mapping
 * JSON responses from an integration to what is returned to a client,
 * where strong typing is not required. In the absence of any defined
 * model, the Empty model will be used to return the response payload
 * unmapped.
 *
 * Definition
 * {
 *   "$schema" : "http://json-schema.org/draft-04/schema#",
 *   "title" : "Empty Schema",
 *   "type" : "object"
 * }
 *
 * @see https://docs.amazonaws.cn/en_us/apigateway/latest/developerguide/models-mappings.html#models-mappings-models
 */
Model.EMPTY_MODEL = new EmptyModel();
//# sourceMappingURL=data:application/json;base64,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