"use strict";
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const apigw = require("../lib");
module.exports = {
    'ProxyResource defines a "{proxy+}" resource with ANY method'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        // WHEN
        new apigw.ProxyResource(stack, 'proxy', {
            parent: api.root,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Resource', {
            'ParentId': {
                'Fn::GetAtt': [
                    'apiC8550315',
                    'RootResourceId',
                ],
            },
            'PathPart': '{proxy+}',
            'RestApiId': {
                'Ref': 'apiC8550315',
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            'HttpMethod': 'ANY',
            'ResourceId': {
                'Ref': 'proxy3A1DA9C7',
            },
            'RestApiId': {
                'Ref': 'apiC8550315',
            },
            'AuthorizationType': 'NONE',
            'Integration': {
                'Type': 'MOCK',
            },
        }));
        test.done();
    },
    'if "anyMethod" is false, then an ANY method will not be defined'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        // WHEN
        const proxy = new apigw.ProxyResource(stack, 'proxy', {
            parent: api.root,
            anyMethod: false,
        });
        proxy.addMethod('GET');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Resource'));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', { 'HttpMethod': 'GET' }));
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::ApiGateway::Method', { 'HttpMethod': 'ANY' }));
        test.done();
    },
    'addProxy can be used on any resource to attach a proxy from that route'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api', {
            deploy: false,
            cloudWatchRole: false,
        });
        const v2 = api.root.addResource('v2');
        v2.addProxy();
        assert_1.expect(stack).toMatch({
            'Resources': {
                'apiC8550315': {
                    'Type': 'AWS::ApiGateway::RestApi',
                    'Properties': {
                        'Name': 'api',
                    },
                },
                'apiv25206B108': {
                    'Type': 'AWS::ApiGateway::Resource',
                    'Properties': {
                        'ParentId': {
                            'Fn::GetAtt': [
                                'apiC8550315',
                                'RootResourceId',
                            ],
                        },
                        'PathPart': 'v2',
                        'RestApiId': {
                            'Ref': 'apiC8550315',
                        },
                    },
                },
                'apiv2proxyAEA4DAC8': {
                    'Type': 'AWS::ApiGateway::Resource',
                    'Properties': {
                        'ParentId': {
                            'Ref': 'apiv25206B108',
                        },
                        'PathPart': '{proxy+}',
                        'RestApiId': {
                            'Ref': 'apiC8550315',
                        },
                    },
                },
                'apiv2proxyANY889F4CE1': {
                    'Type': 'AWS::ApiGateway::Method',
                    'Properties': {
                        'HttpMethod': 'ANY',
                        'ResourceId': {
                            'Ref': 'apiv2proxyAEA4DAC8',
                        },
                        'RestApiId': {
                            'Ref': 'apiC8550315',
                        },
                        'AuthorizationType': 'NONE',
                        'Integration': {
                            'Type': 'MOCK',
                        },
                    },
                },
            },
        });
        test.done();
    },
    'if proxy is added to root, proxy methods are automatically duplicated (with integration and options)'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        const proxy = api.root.addProxy({
            anyMethod: false,
        });
        const deleteInteg = new apigw.MockIntegration({
            requestParameters: {
                foo: 'bar',
            },
        });
        // WHEN
        proxy.addMethod('DELETE', deleteInteg, {
            operationName: 'DeleteMe',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'DELETE',
            ResourceId: { Ref: 'apiproxy4EA44110' },
            Integration: {
                RequestParameters: { foo: 'bar' },
                Type: 'MOCK',
            },
            OperationName: 'DeleteMe',
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'DELETE',
            ResourceId: { 'Fn::GetAtt': ['apiC8550315', 'RootResourceId'] },
            Integration: {
                RequestParameters: { foo: 'bar' },
                Type: 'MOCK',
            },
            OperationName: 'DeleteMe',
        }));
        test.done();
    },
    'if proxy is added to root, proxy methods are only added if they are not defined already on the root resource'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        api.root.addMethod('POST');
        const proxy = api.root.addProxy({ anyMethod: false });
        // WHEN
        proxy.addMethod('POST');
        // THEN
        test.done();
    },
    'url for a resource'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        // WHEN
        const aResource = api.root.addResource('a');
        const cResource = aResource.addResource('b').addResource('c');
        // THEN
        test.deepEqual(stack.resolve(aResource.url), {
            'Fn::Join': [
                '',
                [
                    'https://',
                    { Ref: 'apiC8550315' },
                    '.execute-api.',
                    { Ref: 'AWS::Region' },
                    '.',
                    { Ref: 'AWS::URLSuffix' },
                    '/',
                    { Ref: 'apiDeploymentStageprod896C8101' },
                    '/a',
                ],
            ],
        });
        test.deepEqual(stack.resolve(cResource.url), {
            'Fn::Join': [
                '',
                [
                    'https://',
                    { Ref: 'apiC8550315' },
                    '.execute-api.',
                    { Ref: 'AWS::Region' },
                    '.',
                    { Ref: 'AWS::URLSuffix' },
                    '/',
                    { Ref: 'apiDeploymentStageprod896C8101' },
                    '/a/b/c',
                ],
            ],
        });
        test.done();
    },
    'getResource': {
        'root resource': {
            'returns undefined if not found'(test) {
                // GIVEN
                const stack = new core_1.Stack();
                const api = new apigw.RestApi(stack, 'MyRestApi');
                // THEN
                test.deepEqual(api.root.getResource('boom'), undefined);
                test.done();
            },
            'returns the resource if found'(test) {
                // GIVEN
                const stack = new core_1.Stack();
                const api = new apigw.RestApi(stack, 'MyRestApi');
                // WHEN
                const r1 = api.root.addResource('hello');
                const r2 = api.root.addResource('world');
                // THEN
                test.deepEqual(api.root.getResource('hello'), r1);
                test.deepEqual(api.root.getResource('world'), r2);
                test.done();
            },
            'returns the resource even if it was created using "new"'(test) {
                // GIVEN
                const stack = new core_1.Stack();
                const api = new apigw.RestApi(stack, 'MyRestApi');
                // WHEN
                const r1 = new apigw.Resource(stack, 'child', {
                    parent: api.root,
                    pathPart: 'yello',
                });
                // THEN
                test.deepEqual(api.root.getResource('yello'), r1);
                test.done();
            },
        },
        'non-root': {
            'returns undefined if not found'(test) {
                // GIVEN
                const stack = new core_1.Stack();
                const api = new apigw.RestApi(stack, 'MyRestApi');
                const res = api.root.addResource('boom');
                // THEN
                test.deepEqual(res.getResource('child-of-boom'), undefined);
                test.done();
            },
            'returns the resource if found'(test) {
                // GIVEN
                const stack = new core_1.Stack();
                const api = new apigw.RestApi(stack, 'MyRestApi');
                const child = api.root.addResource('boom');
                // WHEN
                const r1 = child.addResource('hello');
                const r2 = child.addResource('world');
                // THEN
                test.deepEqual(child.getResource('hello'), r1);
                test.deepEqual(child.getResource('world'), r2);
                test.done();
            },
            'returns the resource even if created with "new"'(test) {
                // GIVEN
                const stack = new core_1.Stack();
                const api = new apigw.RestApi(stack, 'MyRestApi');
                const child = api.root.addResource('boom');
                // WHEN
                const r1 = child.addResource('hello');
                const r2 = new apigw.Resource(stack, 'world', {
                    parent: child,
                    pathPart: 'outside-world',
                });
                // THEN
                test.deepEqual(child.getResource('hello'), r1);
                test.deepEqual(child.getResource('outside-world'), r2);
                test.done();
            },
        },
        'resourceForPath': {
            'empty path or "/" (on root) returns this'(test) {
                // GIVEN
                const stack = new core_1.Stack();
                const api = new apigw.RestApi(stack, 'MyRestApi');
                // THEN
                test.deepEqual(api.root.resourceForPath(''), api.root);
                test.deepEqual(api.root.resourceForPath('/'), api.root);
                test.deepEqual(api.root.resourceForPath('///'), api.root);
                test.done();
            },
            'returns a resource for that path'(test) {
                // GIVEN
                const stack = new core_1.Stack();
                const api = new apigw.RestApi(stack, 'MyRestApi');
                // WHEN
                const resource = api.root.resourceForPath('/boom/trach');
                // THEN
                test.deepEqual(resource.path, '/boom/trach');
                test.done();
            },
            'resources not created if not needed'(test) {
                // GIVEN
                const stack = new core_1.Stack();
                const api = new apigw.RestApi(stack, 'MyRestApi');
                // WHEN
                const trach = api.root.resourceForPath('/boom/trach');
                const bam1 = api.root.resourceForPath('/boom/bam');
                // THEN
                const parent = api.root.getResource('boom');
                test.ok(parent);
                test.deepEqual(parent.path, '/boom');
                test.same(trach.parentResource, parent);
                test.deepEqual(trach.parentResource.path, '/boom');
                const bam2 = api.root.resourceForPath('/boom/bam');
                test.same(bam1, bam2);
                test.deepEqual(bam1.parentResource.path, '/boom');
                test.done();
            },
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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