#!/usr/bin/env python
#
# BSD 3-Clause License
#
# Copyright (c) 2016-21, University of Liverpool
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# * Redistributions of source code must retain the above copyright notice, this
#   list of conditions and the following disclaimer.
#
# * Redistributions in binary form must reproduce the above copyright notice,
#   this list of conditions and the following disclaimer in the documentation
#   and/or other materials provided with the distribution.
#
# * Neither the name of the copyright holder nor the names of its
#   contributors may be used to endorse or promote products derived from
#   this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
"""This script provides direct access to ConKit's conversion algorithms.

This script can convert either contact prediction files or sequence files.
In case of the latter, a file with a single or multiple sequences can be
converted.

!!! IMPORTANT
=============
Do not attempt to mix formats, i.e. convert from a contact file format
to a sequence file format.

"""

__author__ = "Felix Simkovic"
__date__ = "01 Oct 2016"
__version__ = "0.13.1"

import argparse

import conkit.command_line
import conkit.io

logger = None


def main():
    parser = argparse.ArgumentParser(description=__doc__, formatter_class=argparse.RawDescriptionHelpFormatter)
    parser.add_argument("infile")
    parser.add_argument("informat")
    parser.add_argument("outfile")
    parser.add_argument("outformat")
    args = parser.parse_args()

    global logger
    logger = conkit.command_line.setup_logging(level="info")

    if args.outformat == "rosetta":
        raise NotImplementedError("This conversion is not yet supported")

    msg = "Converting file{nline}{tab}{infile} of format {informat}{nline}"
    msg += "to file{nline}{tab}{outfile} of format {outformat}"
    msg = msg.format(
        infile=args.infile, informat=args.informat, outfile=args.outfile, outformat=args.outformat, nline="\n", tab="\t"
    )
    logger.info(msg)
    conkit.io.convert(args.infile, args.informat, args.outfile, args.outformat)

    return


if __name__ == "__main__":
    import sys
    import traceback

    try:
        main()
        sys.exit(0)
    except Exception as e:
        if not isinstance(e, SystemExit):
            msg = "".join(traceback.format_exception(*sys.exc_info()))
            logger.critical(msg)
        sys.exit(1)
