"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConfigMap = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const cdk8s = require("cdk8s");
const minimatch = require("minimatch");
const base_1 = require("./base");
const k8s = require("./imports/k8s");
const utils_1 = require("./utils");
/**
 * ConfigMap holds configuration data for pods to consume.
 *
 * @stability stable
 */
class ConfigMap extends base_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _b, _c;
        super(scope, id);
        this._binaryData = {};
        this._data = {};
        this.apiObject = new k8s.KubeConfigMap(this, 'Resource', {
            metadata: props.metadata,
            // we need lazy here because we filter empty
            data: cdk8s.Lazy.any({ produce: () => this.synthesizeData() }),
            binaryData: cdk8s.Lazy.any({ produce: () => this.synthesizeBinaryData() }),
        });
        for (const [k, v] of Object.entries((_b = props.data) !== null && _b !== void 0 ? _b : {})) {
            this.addData(k, v);
        }
        for (const [k, v] of Object.entries((_c = props.binaryData) !== null && _c !== void 0 ? _c : {})) {
            this.addBinaryData(k, v);
        }
    }
    /**
     * Represents a ConfigMap created elsewhere.
     *
     * @param name The name of the config map to import.
     * @stability stable
     */
    static fromConfigMapName(name) {
        return { name };
    }
    /**
     * Adds a data entry to the config map.
     *
     * @param key The key.
     * @param value The value.
     * @stability stable
     * @throws if there is either a `data` or `binaryData` entry with the same key
     */
    addData(key, value) {
        this.verifyKeyAvailable(key);
        this._data[key] = value;
    }
    /**
     * The data associated with this config map.
     *
     * Returns an copy. To add data records, use `addData()` or `addBinaryData()`.
     *
     * @stability stable
     */
    get data() {
        return { ...this._data };
    }
    /**
     * Adds a binary data entry to the config map.
     *
     * BinaryData can contain byte
     * sequences that are not in the UTF-8 range.
     *
     * @param key The key.
     * @param value The value.
     * @stability stable
     * @throws if there is either a `data` or `binaryData` entry with the same key
     */
    addBinaryData(key, value) {
        this.verifyKeyAvailable(key);
        this._binaryData[key] = value;
    }
    /**
     * The binary data associated with this config map.
     *
     * Returns a copy. To add data records, use `addBinaryData()` or `addData()`.
     *
     * @stability stable
     */
    get binaryData() {
        return { ...this._binaryData };
    }
    /**
     * Adds a file to the ConfigMap.
     *
     * @param localFile The path to the local file.
     * @param key The ConfigMap key (default to the file name).
     * @stability stable
     */
    addFile(localFile, key) {
        key = key !== null && key !== void 0 ? key : path.basename(localFile);
        const value = fs.readFileSync(localFile, 'utf-8');
        this.addData(key, value);
    }
    /**
     * Adds a directory to the ConfigMap.
     *
     * @param localDir A path to a local directory.
     * @param options Options.
     * @stability stable
     */
    addDirectory(localDir, options = {}) {
        var _b, _c;
        const exclude = (_b = options.exclude) !== null && _b !== void 0 ? _b : [];
        const shouldInclude = (file) => {
            for (const pattern of exclude) {
                if (minimatch(file, pattern)) {
                    return false;
                }
            }
            return true;
        };
        const keyPrefix = (_c = options.keyPrefix) !== null && _c !== void 0 ? _c : '';
        for (const file of fs.readdirSync(localDir)) {
            const filePath = path.join(localDir, file);
            if (fs.statSync(filePath).isDirectory()) {
                continue;
            }
            if (!shouldInclude(file)) {
                continue;
            }
            const relativeFilePath = keyPrefix + file;
            this.addFile(filePath, relativeFilePath);
        }
    }
    verifyKeyAvailable(key) {
        if (key in this._data || key in this._binaryData) {
            throw new Error(`unable to add a ConfigMap entry with key "${key}". It is already used`);
        }
    }
    synthesizeData() {
        return utils_1.undefinedIfEmpty(this._data);
    }
    synthesizeBinaryData() {
        return utils_1.undefinedIfEmpty(this._binaryData);
    }
}
exports.ConfigMap = ConfigMap;
_a = JSII_RTTI_SYMBOL_1;
ConfigMap[_a] = { fqn: "cdk8s-plus-17.ConfigMap", version: "1.0.0-beta.69" };
//# sourceMappingURL=data:application/json;base64,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