"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EndpointType = exports.ApiKeySourceType = exports.RestApi = exports.SpecRestApi = exports.RestApiBase = void 0;
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const api_key_1 = require("./api-key");
const apigateway_generated_1 = require("./apigateway.generated");
const deployment_1 = require("./deployment");
const domain_name_1 = require("./domain-name");
const gateway_response_1 = require("./gateway-response");
const model_1 = require("./model");
const requestvalidator_1 = require("./requestvalidator");
const resource_1 = require("./resource");
const stage_1 = require("./stage");
const usage_plan_1 = require("./usage-plan");
const RESTAPI_SYMBOL = Symbol.for('@aws-cdk/aws-apigateway.RestApiBase');
/**
 * Base implementation that are common to various implementations of IRestApi.
 */
class RestApiBase extends core_1.Resource {
    /**
     *
     */
    constructor(scope, id, props = {}) {
        var _a;
        super(scope, id);
        this.restApiName = (_a = props.restApiName) !== null && _a !== void 0 ? _a : id;
        Object.defineProperty(this, RESTAPI_SYMBOL, { value: true });
    }
    /**
     * Checks if the given object is an instance of RestApiBase.
     * @internal
     */
    static _isRestApiBase(x) {
        return x !== null && typeof (x) === 'object' && RESTAPI_SYMBOL in x;
    }
    /**
     * API Gateway deployment that represents the latest changes of the API.
     *
     * This resource will be automatically updated every time the REST API model changes.
     * This will be undefined if `deploy` is false.
     */
    get latestDeployment() {
        return this._latestDeployment;
    }
    /**
     * The first domain name mapped to this API, if defined through the `domainName` configuration prop, or added via `addDomainName`.
     */
    get domainName() {
        return this._domainName;
    }
    /**
     * Returns the URL for an HTTP path.
     *
     * Fails if `deploymentStage` is not set either by `deploy` or explicitly.
     */
    urlForPath(path = '/') {
        if (!this.deploymentStage) {
            throw new Error('Cannot determine deployment stage for API from "deploymentStage". Use "deploy" or explicitly set "deploymentStage"');
        }
        return this.deploymentStage.urlForPath(path);
    }
    /**
     * Defines an API Gateway domain name and maps it to this API.
     *
     * @param id The construct id.
     * @param options custom domain options.
     */
    addDomainName(id, options) {
        const domainName = new domain_name_1.DomainName(this, id, {
            ...options,
            mapping: this,
        });
        if (!this._domainName) {
            this._domainName = domainName;
        }
        return domainName;
    }
    /**
     * Adds a usage plan.
     */
    addUsagePlan(id, props = {}) {
        return new usage_plan_1.UsagePlan(this, id, props);
    }
    /**
     * Gets the "execute-api" ARN.
     */
    arnForExecuteApi(method = '*', path = '/*', stage = '*') {
        if (!path.startsWith('/')) {
            throw new Error(`"path" must begin with a "/": '${path}'`);
        }
        if (method.toUpperCase() === 'ANY') {
            method = '*';
        }
        return core_1.Stack.of(this).formatArn({
            service: 'execute-api',
            resource: this.restApiId,
            sep: '/',
            resourceName: `${stage}/${method}${path}`,
        });
    }
    /**
     * Adds a new gateway response.
     */
    addGatewayResponse(id, options) {
        return new gateway_response_1.GatewayResponse(this, id, {
            restApi: this,
            ...options,
        });
    }
    /**
     * Add an ApiKey.
     */
    addApiKey(id, options) {
        return new api_key_1.ApiKey(this, id, {
            resources: [this],
            ...options,
        });
    }
    /**
     * Returns the given named metric for this API.
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/ApiGateway',
            metricName,
            dimensions: { ApiName: this.restApiName },
            ...props,
        });
    }
    /**
     * Metric for the number of client-side errors captured in a given period.
     *
     * @default - sum over 5 minutes
     */
    metricClientError(props) {
        return this.metric('4XXError', { statistic: 'Sum', ...props });
    }
    /**
     * Metric for the number of server-side errors captured in a given period.
     *
     * @default - sum over 5 minutes
     */
    metricServerError(props) {
        return this.metric('5XXError', { statistic: 'Sum', ...props });
    }
    /**
     * Metric for the number of requests served from the API cache in a given period.
     *
     * @default - sum over 5 minutes
     */
    metricCacheHitCount(props) {
        return this.metric('CacheHitCount', { statistic: 'Sum', ...props });
    }
    /**
     * Metric for the number of requests served from the backend in a given period, when API caching is enabled.
     *
     * @default - sum over 5 minutes
     */
    metricCacheMissCount(props) {
        return this.metric('CacheMissCount', { statistic: 'Sum', ...props });
    }
    /**
     * Metric for the total number API requests in a given period.
     *
     * @default - SampleCount over 5 minutes
     */
    metricCount(props) {
        return this.metric('Count', { statistic: 'SampleCount', ...props });
    }
    /**
     * Metric for the time between when API Gateway relays a request to the backend and when it receives a response from the backend.
     *
     * @default - no statistic
     */
    metricIntegrationLatency(props) {
        return this.metric('IntegrationLatency', props);
    }
    /**
     * The time between when API Gateway receives a request from a client and when it returns a response to the client.
     *
     * The latency includes the integration latency and other API Gateway overhead.
     *
     * @default - no statistic
     */
    metricLatency(props) {
        return this.metric('Latency', props);
    }
    /**
     * Internal API used by `Method` to keep an inventory of methods at the API
     * level for validation purposes.
     *
     * @internal
     */
    _attachMethod(method) {
        ignore(method);
    }
    /**
     * Associates a Deployment resource with this REST API.
     *
     * @internal
     */
    _attachDeployment(deployment) {
        ignore(deployment);
    }
    /**
     *
     */
    configureCloudWatchRole(apiResource) {
        const role = new iam.Role(this, 'CloudWatchRole', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonAPIGatewayPushToCloudWatchLogs')],
        });
        const resource = new apigateway_generated_1.CfnAccount(this, 'Account', {
            cloudWatchRoleArn: role.roleArn,
        });
        resource.node.addDependency(apiResource);
    }
    /**
     *
     */
    configureDeployment(props) {
        const deploy = props.deploy === undefined ? true : props.deploy;
        if (deploy) {
            this._latestDeployment = new deployment_1.Deployment(this, 'Deployment', {
                description: 'Automatically created by the RestApi construct',
                api: this,
                retainDeployments: props.retainDeployments,
            });
            // encode the stage name into the construct id, so if we change the stage name, it will recreate a new stage.
            // stage name is part of the endpoint, so that makes sense.
            const stageName = (props.deployOptions && props.deployOptions.stageName) || 'prod';
            this.deploymentStage = new stage_1.Stage(this, `DeploymentStage.${stageName}`, {
                deployment: this._latestDeployment,
                ...props.deployOptions,
            });
            new core_1.CfnOutput(this, 'Endpoint', { exportName: props.endpointExportName, value: this.urlForPath() });
        }
        else {
            if (props.deployOptions) {
                throw new Error('Cannot set \'deployOptions\' if \'deploy\' is disabled');
            }
        }
    }
    /**
     * @internal
     */
    _configureEndpoints(props) {
        var _a, _b;
        if (props.endpointTypes && props.endpointConfiguration) {
            throw new Error('Only one of the RestApi props, endpointTypes or endpointConfiguration, is allowed');
        }
        if (props.endpointConfiguration) {
            return {
                types: props.endpointConfiguration.types,
                vpcEndpointIds: (_b = (_a = props.endpointConfiguration) === null || _a === void 0 ? void 0 : _a.vpcEndpoints) === null || _b === void 0 ? void 0 : _b.map(vpcEndpoint => vpcEndpoint.vpcEndpointId),
            };
        }
        if (props.endpointTypes) {
            return { types: props.endpointTypes };
        }
        return undefined;
    }
}
exports.RestApiBase = RestApiBase;
/**
 * (experimental) Represents a REST API in Amazon API Gateway, created with an OpenAPI specification.
 *
 * Some properties normally accessible on @see {@link RestApi} - such as the description -
 * must be declared in the specification. All Resources and Methods need to be defined as
 * part of the OpenAPI specification file, and cannot be added via the CDK.
 *
 * By default, the API will automatically be deployed and accessible from a
 * public endpoint.
 *
 * @experimental
 * @resource AWS::ApiGateway::RestApi
 */
class SpecRestApi extends RestApiBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        const apiDefConfig = props.apiDefinition.bind(this);
        const resource = new apigateway_generated_1.CfnRestApi(this, 'Resource', {
            name: this.restApiName,
            policy: props.policy,
            failOnWarnings: props.failOnWarnings,
            body: apiDefConfig.inlineDefinition ? apiDefConfig.inlineDefinition : undefined,
            bodyS3Location: apiDefConfig.inlineDefinition ? undefined : apiDefConfig.s3Location,
            endpointConfiguration: this._configureEndpoints(props),
            parameters: props.parameters,
        });
        this.node.defaultChild = resource;
        this.restApiId = resource.ref;
        this.restApiRootResourceId = resource.attrRootResourceId;
        this.root = new RootResource(this, {}, this.restApiRootResourceId);
        this.configureDeployment(props);
        if (props.domainName) {
            this.addDomainName('CustomDomain', props.domainName);
        }
        const cloudWatchRole = props.cloudWatchRole !== undefined ? props.cloudWatchRole : true;
        if (cloudWatchRole) {
            this.configureCloudWatchRole(resource);
        }
    }
}
exports.SpecRestApi = SpecRestApi;
/**
 * Represents a REST API in Amazon API Gateway.
 *
 * Use `addResource` and `addMethod` to configure the API model.
 *
 * By default, the API will automatically be deployed and accessible from a
 * public endpoint.
 */
class RestApi extends RestApiBase {
    /**
     *
     */
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        /**
         * The list of methods bound to this RestApi.
         */
        this.methods = new Array();
        /**
         * This list of deployments bound to this RestApi
         */
        this.deployments = new Array();
        const resource = new apigateway_generated_1.CfnRestApi(this, 'Resource', {
            name: this.restApiName,
            description: props.description,
            policy: props.policy,
            failOnWarnings: props.failOnWarnings,
            minimumCompressionSize: props.minimumCompressionSize,
            binaryMediaTypes: props.binaryMediaTypes,
            endpointConfiguration: this._configureEndpoints(props),
            apiKeySourceType: props.apiKeySourceType,
            cloneFrom: props.cloneFrom ? props.cloneFrom.restApiId : undefined,
            parameters: props.parameters,
        });
        this.node.defaultChild = resource;
        this.restApiId = resource.ref;
        const cloudWatchRole = props.cloudWatchRole !== undefined ? props.cloudWatchRole : true;
        if (cloudWatchRole) {
            this.configureCloudWatchRole(resource);
        }
        this.configureDeployment(props);
        if (props.domainName) {
            this.addDomainName('CustomDomain', props.domainName);
        }
        this.root = new RootResource(this, props, resource.attrRootResourceId);
        this.restApiRootResourceId = resource.attrRootResourceId;
    }
    /**
     * Import an existing RestApi.
     */
    static fromRestApiId(scope, id, restApiId) {
        class Import extends RestApiBase {
            constructor() {
                super(...arguments);
                this.restApiId = restApiId;
            }
            get root() {
                throw new Error('root is not configured when imported using `fromRestApiId()`. Use `fromRestApiAttributes()` API instead.');
            }
            get restApiRootResourceId() {
                throw new Error('restApiRootResourceId is not configured when imported using `fromRestApiId()`. Use `fromRestApiAttributes()` API instead.');
            }
        }
        return new Import(scope, id);
    }
    /**
     * (experimental) Import an existing RestApi that can be configured with additional Methods and Resources.
     *
     * @experimental
     */
    static fromRestApiAttributes(scope, id, attrs) {
        class Import extends RestApiBase {
            constructor() {
                super(...arguments);
                this.restApiId = attrs.restApiId;
                this.restApiRootResourceId = attrs.rootResourceId;
                this.root = new RootResource(this, {}, this.restApiRootResourceId);
            }
        }
        return new Import(scope, id);
    }
    /**
     * The deployed root URL of this REST API.
     */
    get url() {
        return this.urlForPath();
    }
    /**
     * Adds a new model.
     */
    addModel(id, props) {
        return new model_1.Model(this, id, {
            ...props,
            restApi: this,
        });
    }
    /**
     * Adds a new request validator.
     */
    addRequestValidator(id, props) {
        return new requestvalidator_1.RequestValidator(this, id, {
            ...props,
            restApi: this,
        });
    }
    /**
     * Internal API used by `Method` to keep an inventory of methods at the API
     * level for validation purposes.
     *
     * @internal
     */
    _attachMethod(method) {
        this.methods.push(method);
        // add this method as a dependency to all deployments defined for this api
        // when additional deployments are added, _attachDeployment is called and
        // this method will be added there.
        for (const dep of this.deployments) {
            dep._addMethodDependency(method);
        }
    }
    /**
     * Attaches a deployment to this REST API.
     *
     * @internal
     */
    _attachDeployment(deployment) {
        this.deployments.push(deployment);
        // add all methods that were already defined as dependencies of this
        // deployment when additional methods are added, _attachMethod is called and
        // it will be added as a dependency to this deployment.
        for (const method of this.methods) {
            deployment._addMethodDependency(method);
        }
    }
    /**
     * Performs validation of the REST API.
     */
    validate() {
        if (this.methods.length === 0) {
            return ["The REST API doesn't contain any methods"];
        }
        return [];
    }
}
exports.RestApi = RestApi;
/**
 *
 */
var ApiKeySourceType;
(function (ApiKeySourceType) {
    ApiKeySourceType["HEADER"] = "HEADER";
    ApiKeySourceType["AUTHORIZER"] = "AUTHORIZER";
})(ApiKeySourceType = exports.ApiKeySourceType || (exports.ApiKeySourceType = {}));
/**
 *
 */
var EndpointType;
(function (EndpointType) {
    EndpointType["EDGE"] = "EDGE";
    EndpointType["REGIONAL"] = "REGIONAL";
    EndpointType["PRIVATE"] = "PRIVATE";
})(EndpointType = exports.EndpointType || (exports.EndpointType = {}));
class RootResource extends resource_1.ResourceBase {
    constructor(api, props, resourceId) {
        super(api, 'Default');
        this.parentResource = undefined;
        this.defaultIntegration = props.defaultIntegration;
        this.defaultMethodOptions = props.defaultMethodOptions;
        this.defaultCorsPreflightOptions = props.defaultCorsPreflightOptions;
        this.api = api;
        this.resourceId = resourceId;
        this.path = '/';
        if (api instanceof RestApi) {
            this._restApi = api;
        }
        if (this.defaultCorsPreflightOptions) {
            this.addCorsPreflight(this.defaultCorsPreflightOptions);
        }
    }
    /**
     * Get the RestApi associated with this Resource.
     * @deprecated - Throws an error if this Resource is not associated with an instance of `RestApi`. Use `api` instead.
     */
    get restApi() {
        if (!this._restApi) {
            throw new Error('RestApi is not available on Resource not connected to an instance of RestApi. Use `api` instead');
        }
        return this._restApi;
    }
}
function ignore(_x) {
    return;
}
//# sourceMappingURL=data:application/json;base64,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