"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UsagePlan = exports.Period = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const cx_api_1 = require("@aws-cdk/cx-api");
const apigateway_generated_1 = require("./apigateway.generated");
const util_1 = require("./util");
/**
 * Time period for which quota settings apply.
 */
var Period;
(function (Period) {
    Period["DAY"] = "DAY";
    Period["WEEK"] = "WEEK";
    Period["MONTH"] = "MONTH";
})(Period = exports.Period || (exports.Period = {}));
class UsagePlanBase extends core_1.Resource {
    /**
     * Adds an ApiKey.
     *
     * @param apiKey the api key to associate with this usage plan
     * @param options options that control the behaviour of this method
     */
    addApiKey(apiKey, options) {
        let id;
        const prefix = 'UsagePlanKeyResource';
        if (core_1.FeatureFlags.of(this).isEnabled(cx_api_1.APIGATEWAY_USAGEPLANKEY_ORDERINSENSITIVE_ID)) {
            id = `${prefix}:${core_1.Names.nodeUniqueId(apiKey.node)}`;
        }
        else {
            // Postfixing apikey id only from the 2nd child, to keep physicalIds of UsagePlanKey for existing CDK apps unmodified.
            id = this.node.tryFindChild(prefix) ? `${prefix}:${core_1.Names.nodeUniqueId(apiKey.node)}` : prefix;
        }
        const resource = new apigateway_generated_1.CfnUsagePlanKey(this, id, {
            keyId: apiKey.keyId,
            keyType: "API_KEY" /* API_KEY */,
            usagePlanId: this.usagePlanId,
        });
        if (options === null || options === void 0 ? void 0 : options.overrideLogicalId) {
            resource.overrideLogicalId(options === null || options === void 0 ? void 0 : options.overrideLogicalId);
        }
    }
}
class UsagePlan extends UsagePlanBase {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.apiStages = new Array();
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_apigateway_UsagePlanProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        let resource;
        resource = new apigateway_generated_1.CfnUsagePlan(this, 'Resource', {
            apiStages: core_1.Lazy.any({ produce: () => this.renderApiStages(this.apiStages) }),
            description: props.description,
            quota: this.renderQuota(props),
            throttle: this.renderThrottle(props.throttle),
            usagePlanName: props.name,
        });
        this.apiStages.push(...(props.apiStages || []));
        this.usagePlanId = resource.ref;
        // Add ApiKey when
        if (props.apiKey) {
            this.addApiKey(props.apiKey);
        }
    }
    /**
     * Import an externally defined usage plan using its ARN.
     *
     * @param scope  the construct that will "own" the imported usage plan.
     * @param id     the id of the imported usage plan in the construct tree.
     * @param usagePlanId the id of an existing usage plan.
     */
    static fromUsagePlanId(scope, id, usagePlanId) {
        class Import extends UsagePlanBase {
            constructor() {
                super(scope, id);
                this.usagePlanId = usagePlanId;
            }
        }
        return new Import();
    }
    /**
     * Adds an apiStage.
     * @param apiStage
     */
    addApiStage(apiStage) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_apigateway_UsagePlanPerApiStage(apiStage);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addApiStage);
            }
            throw error;
        }
        this.apiStages.push(apiStage);
    }
    /**
     *
     * @param props
     */
    renderApiStages(apiStages) {
        if (apiStages && apiStages.length > 0) {
            const stages = [];
            apiStages.forEach((apiStage) => {
                stages.push(this.createStage(apiStage));
            });
            return stages;
        }
        return undefined;
    }
    createStage(apiStage) {
        const stage = apiStage.stage ? apiStage.stage.stageName.toString() : undefined;
        const apiId = apiStage.stage ? apiStage.stage.restApi.restApiId : undefined;
        const throttle = this.renderThrottlePerMethod(apiStage.throttle);
        return {
            apiId,
            stage,
            throttle,
        };
    }
    renderQuota(props) {
        if (props.quota === undefined) {
            return undefined;
        }
        else {
            const limit = props.quota ? props.quota.limit : undefined;
            util_1.validateInteger(limit, 'Throttle quota limit');
            const ret = {
                limit: limit ? limit : undefined,
                offset: props.quota ? props.quota.offset : undefined,
                period: props.quota ? props.quota.period : undefined,
            };
            return ret;
        }
    }
    renderThrottle(props) {
        let ret;
        if (props !== undefined) {
            const burstLimit = props.burstLimit;
            util_1.validateInteger(burstLimit, 'Throttle burst limit');
            const rateLimit = props.rateLimit;
            util_1.validateDouble(rateLimit, 'Throttle rate limit');
            ret = {
                burstLimit: burstLimit,
                rateLimit: rateLimit,
            };
        }
        return ret;
    }
    renderThrottlePerMethod(throttlePerMethod) {
        const ret = {};
        if (throttlePerMethod && throttlePerMethod.length > 0) {
            throttlePerMethod.forEach((value) => {
                const method = value.method;
                // this methodId is resource path and method for example /GET or /pets/GET
                const methodId = `${method.resource.path}/${method.httpMethod}`;
                ret[methodId] = this.renderThrottle(value.throttle);
            });
        }
        return ret;
    }
}
exports.UsagePlan = UsagePlan;
_a = JSII_RTTI_SYMBOL_1;
UsagePlan[_a] = { fqn: "@aws-cdk/aws-apigateway.UsagePlan", version: "1.158.0" };
//# sourceMappingURL=data:application/json;base64,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