"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseCluster = exports.DatabaseClusterBase = exports.LogType = exports.EngineVersion = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const ec2 = require("aws-cdk-lib/aws-ec2");
const iam = require("aws-cdk-lib/aws-iam");
const logs = require("aws-cdk-lib/aws-logs");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const endpoint_1 = require("./endpoint");
const aws_neptune_1 = require("aws-cdk-lib/aws-neptune");
const subnet_group_1 = require("./subnet-group");
/**
 * Possible Instances Types to use in Neptune cluster
 * used for defining `DatabaseClusterProps.engineVersion`.
 */
class EngineVersion {
    /**
     * Constructor for specifying a custom engine version
     * @param version the engine version of Neptune
     */
    constructor(version) {
        this.version = version;
    }
}
exports.EngineVersion = EngineVersion;
_a = JSII_RTTI_SYMBOL_1;
EngineVersion[_a] = { fqn: "@aws-cdk/aws-neptune-alpha.EngineVersion", version: "2.64.0-alpha.0" };
/**
 * Neptune engine version 1.0.1.0
 */
EngineVersion.V1_0_1_0 = new EngineVersion('1.0.1.0');
/**
 * Neptune engine version 1.0.1.1
 */
EngineVersion.V1_0_1_1 = new EngineVersion('1.0.1.1');
/**
 * Neptune engine version 1.0.1.2
 */
EngineVersion.V1_0_1_2 = new EngineVersion('1.0.1.2');
/**
 * Neptune engine version 1.0.2.1
 */
EngineVersion.V1_0_2_1 = new EngineVersion('1.0.2.1');
/**
 * Neptune engine version 1.0.2.2
 */
EngineVersion.V1_0_2_2 = new EngineVersion('1.0.2.2');
/**
 * Neptune engine version 1.0.3.0
 */
EngineVersion.V1_0_3_0 = new EngineVersion('1.0.3.0');
/**
 * Neptune engine version 1.0.4.0
 */
EngineVersion.V1_0_4_0 = new EngineVersion('1.0.4.0');
/**
 * Neptune engine version 1.0.4.1
 */
EngineVersion.V1_0_4_1 = new EngineVersion('1.0.4.1');
/**
 * Neptune engine version 1.0.5.0
 */
EngineVersion.V1_0_5_0 = new EngineVersion('1.0.5.0');
/**
 * Neptune engine version 1.1.0.0
 */
EngineVersion.V1_1_0_0 = new EngineVersion('1.1.0.0');
/**
 * Neptune engine version 1.1.1.0
 */
EngineVersion.V1_1_1_0 = new EngineVersion('1.1.1.0');
/**
 * Neptune engine version 1.2.0.0
 */
EngineVersion.V1_2_0_0 = new EngineVersion('1.2.0.0');
/**
 * Neptune log types that can be exported to CloudWatch logs
 *
 * @see https://docs.aws.amazon.com/neptune/latest/userguide/cloudwatch-logs.html
 */
class LogType {
    /**
     * Constructor for specifying a custom log type
     * @param value the log type
     */
    constructor(value) {
        this.value = value;
    }
}
exports.LogType = LogType;
_b = JSII_RTTI_SYMBOL_1;
LogType[_b] = { fqn: "@aws-cdk/aws-neptune-alpha.LogType", version: "2.64.0-alpha.0" };
/**
 * Audit logs
 *
 * @see https://docs.aws.amazon.com/neptune/latest/userguide/auditing.html
 */
LogType.AUDIT = new LogType('audit');
/**
 * A new or imported database cluster.
 */
class DatabaseClusterBase extends aws_cdk_lib_1.Resource {
    /**
     * Import an existing DatabaseCluster from properties
     */
    static fromDatabaseClusterAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_neptune_alpha_DatabaseClusterAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromDatabaseClusterAttributes);
            }
            throw error;
        }
        class Import extends DatabaseClusterBase {
            constructor() {
                super(...arguments);
                this.defaultPort = ec2.Port.tcp(attrs.port);
                this.connections = new ec2.Connections({
                    securityGroups: [attrs.securityGroup],
                    defaultPort: this.defaultPort,
                });
                this.clusterIdentifier = attrs.clusterIdentifier;
                this.clusterResourceIdentifier = attrs.clusterResourceIdentifier;
                this.clusterEndpoint = new endpoint_1.Endpoint(attrs.clusterEndpointAddress, attrs.port);
                this.clusterReadEndpoint = new endpoint_1.Endpoint(attrs.readerEndpointAddress, attrs.port);
                this.enableIamAuthentication = true;
            }
        }
        return new Import(scope, id);
    }
    grant(grantee, ...actions) {
        if (this.enableIamAuthentication === false) {
            throw new Error('Cannot grant permissions when IAM authentication is disabled');
        }
        this.enableIamAuthentication = true;
        return iam.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [
                [
                    'arn',
                    aws_cdk_lib_1.Aws.PARTITION,
                    'neptune-db',
                    aws_cdk_lib_1.Aws.REGION,
                    aws_cdk_lib_1.Aws.ACCOUNT_ID,
                    `${this.clusterResourceIdentifier}/*`,
                ].join(':'),
            ],
        });
    }
    grantConnect(grantee) {
        return this.grant(grantee, 'neptune-db:*');
    }
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/Neptune',
            dimensionsMap: {
                DBClusterIdentifier: this.clusterIdentifier,
            },
            metricName,
            ...props,
        });
    }
}
exports.DatabaseClusterBase = DatabaseClusterBase;
_c = JSII_RTTI_SYMBOL_1;
DatabaseClusterBase[_c] = { fqn: "@aws-cdk/aws-neptune-alpha.DatabaseClusterBase", version: "2.64.0-alpha.0" };
/**
 * Create a clustered database with a given number of instances.
 *
 * @resource AWS::Neptune::DBCluster
 */
class DatabaseCluster extends DatabaseClusterBase {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * Identifiers of the instance
         */
        this.instanceIdentifiers = [];
        /**
         * Endpoints which address each individual instance.
         */
        this.instanceEndpoints = [];
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_neptune_alpha_DatabaseClusterProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, DatabaseCluster);
            }
            throw error;
        }
        this.vpc = props.vpc;
        this.vpcSubnets = props.vpcSubnets ?? { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS };
        // Determine the subnet(s) to deploy the Neptune cluster to
        const { subnetIds, internetConnectivityEstablished } = this.vpc.selectSubnets(this.vpcSubnets);
        // Cannot test whether the subnets are in different AZs, but at least we can test the amount.
        if (subnetIds.length < 2) {
            throw new Error(`Cluster requires at least 2 subnets, got ${subnetIds.length}`);
        }
        this.subnetGroup = props.subnetGroup ?? new subnet_group_1.SubnetGroup(this, 'Subnets', {
            description: `Subnets for ${id} database`,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            removalPolicy: props.removalPolicy === aws_cdk_lib_1.RemovalPolicy.RETAIN ? props.removalPolicy : undefined,
        });
        const securityGroups = props.securityGroups ?? [
            new ec2.SecurityGroup(this, 'SecurityGroup', {
                description: 'Neptune security group',
                vpc: this.vpc,
            }),
        ];
        // Default to encrypted storage
        const storageEncrypted = props.storageEncrypted ?? true;
        if (props.kmsKey && !storageEncrypted) {
            throw new Error('KMS key supplied but storageEncrypted is false');
        }
        const deletionProtection = props.deletionProtection ?? (props.removalPolicy === aws_cdk_lib_1.RemovalPolicy.RETAIN ? true : undefined);
        this.enableIamAuthentication = props.iamAuthentication;
        // Create the Neptune cluster
        const cluster = new aws_neptune_1.CfnDBCluster(this, 'Resource', {
            // Basic
            engineVersion: props.engineVersion?.version,
            dbClusterIdentifier: props.dbClusterName,
            dbSubnetGroupName: this.subnetGroup.subnetGroupName,
            port: props.port,
            vpcSecurityGroupIds: securityGroups.map(sg => sg.securityGroupId),
            dbClusterParameterGroupName: props.clusterParameterGroup?.clusterParameterGroupName,
            deletionProtection: deletionProtection,
            associatedRoles: props.associatedRoles ? props.associatedRoles.map(role => ({ roleArn: role.roleArn })) : undefined,
            iamAuthEnabled: aws_cdk_lib_1.Lazy.any({ produce: () => this.enableIamAuthentication }),
            // Backup
            backupRetentionPeriod: props.backupRetention?.toDays(),
            preferredBackupWindow: props.preferredBackupWindow,
            preferredMaintenanceWindow: props.preferredMaintenanceWindow,
            // Encryption
            kmsKeyId: props.kmsKey?.keyArn,
            // CloudWatch Logs exports
            enableCloudwatchLogsExports: props.cloudwatchLogsExports?.map(logType => logType.value),
            storageEncrypted,
        });
        cluster.applyRemovalPolicy(props.removalPolicy, {
            applyToUpdateReplacePolicy: true,
        });
        this.clusterIdentifier = cluster.ref;
        this.clusterResourceIdentifier = cluster.attrClusterResourceId;
        const port = aws_cdk_lib_1.Token.asNumber(cluster.attrPort);
        this.clusterEndpoint = new endpoint_1.Endpoint(cluster.attrEndpoint, port);
        this.clusterReadEndpoint = new endpoint_1.Endpoint(cluster.attrReadEndpoint, port);
        // Log retention
        const retention = props.cloudwatchLogsRetention;
        if (retention) {
            props.cloudwatchLogsExports?.forEach(logType => {
                new logs.LogRetention(this, `${logType}LogRetention`, {
                    logGroupName: `/aws/neptune/${this.clusterIdentifier}/${logType.value}`,
                    role: props.cloudwatchLogsRetentionRole,
                    retention,
                });
            });
        }
        // Create the instances
        const instanceCount = props.instances ?? DatabaseCluster.DEFAULT_NUM_INSTANCES;
        if (instanceCount < 1) {
            throw new Error('At least one instance is required');
        }
        for (let i = 0; i < instanceCount; i++) {
            const instanceIndex = i + 1;
            const instanceIdentifier = props.instanceIdentifierBase != null ? `${props.instanceIdentifierBase}${instanceIndex}`
                : props.dbClusterName != null ? `${props.dbClusterName}instance${instanceIndex}` : undefined;
            const instance = new aws_neptune_1.CfnDBInstance(this, `Instance${instanceIndex}`, {
                // Link to cluster
                dbClusterIdentifier: cluster.ref,
                dbInstanceIdentifier: instanceIdentifier,
                // Instance properties
                dbInstanceClass: props.instanceType._instanceType,
                dbParameterGroupName: props.parameterGroup?.parameterGroupName,
                autoMinorVersionUpgrade: props.autoMinorVersionUpgrade === true,
            });
            // We must have a dependency on the NAT gateway provider here to create
            // things in the right order.
            instance.node.addDependency(internetConnectivityEstablished);
            instance.applyRemovalPolicy(props.removalPolicy, {
                applyToUpdateReplacePolicy: true,
            });
            this.instanceIdentifiers.push(instance.ref);
            this.instanceEndpoints.push(new endpoint_1.Endpoint(instance.attrEndpoint, port));
        }
        this.connections = new ec2.Connections({
            defaultPort: ec2.Port.tcp(port),
            securityGroups: securityGroups,
        });
    }
}
exports.DatabaseCluster = DatabaseCluster;
_d = JSII_RTTI_SYMBOL_1;
DatabaseCluster[_d] = { fqn: "@aws-cdk/aws-neptune-alpha.DatabaseCluster", version: "2.64.0-alpha.0" };
/**
 * The default number of instances in the Neptune cluster if none are
 * specified
 */
DatabaseCluster.DEFAULT_NUM_INSTANCES = 1;
//# sourceMappingURL=data:application/json;base64,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