"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StatefulSet = exports.PodManagementPolicy = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const base_1 = require("./base");
const k8s = require("./imports/k8s");
const pod_1 = require("./pod");
/**
 * Controls how pods are created during initial scale up, when replacing pods on nodes, or when scaling down.
 *
 * The default policy is `OrderedReady`, where pods are created in increasing order
 * (pod-0, then pod-1, etc) and the controller will wait until each pod is ready before
 * continuing. When scaling down, the pods are removed in the opposite order.
 *
 * The alternative policy is `Parallel` which will create pods in parallel to match the
 * desired scale without waiting, and on scale down will delete all pods at once.
 *
 * @stability stable
 */
var PodManagementPolicy;
(function (PodManagementPolicy) {
    PodManagementPolicy["ORDERED_READY"] = "OrderedReady";
    PodManagementPolicy["PARALLEL"] = "Parallel";
})(PodManagementPolicy = exports.PodManagementPolicy || (exports.PodManagementPolicy = {}));
/**
 * StatefulSet is the workload API object used to manage stateful applications.
 *
 * Manages the deployment and scaling of a set of Pods, and provides guarantees
 * about the ordering and uniqueness of these Pods.
 *
 * Like a Deployment, a StatefulSet manages Pods that are based on an identical
 * container spec. Unlike a Deployment, a StatefulSet maintains a sticky identity
 * for each of their Pods. These pods are created from the same spec, but are not
 * interchangeable: each has a persistent identifier that it maintains across any
 * rescheduling.
 *
 * If you want to use storage volumes to provide persistence for your workload, you
 * can use a StatefulSet as part of the solution. Although individual Pods in a StatefulSet
 * are susceptible to failure, the persistent Pod identifiers make it easier to match existing
 * volumes to the new Pods that replace any that have failed.
 *
 * Using StatefulSets
 * ------------------
 * StatefulSets are valuable for applications that require one or more of the following.
 *
 * - Stable, unique network identifiers.
 * - Stable, persistent storage.
 * - Ordered, graceful deployment and scaling.
 * - Ordered, automated rolling updates.
 *
 * @stability stable
 */
class StatefulSet extends base_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d;
        super(scope, id);
        this.apiObject = new k8s.KubeStatefulSet(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        this._service = props.service;
        this.apiObject.addDependency(this._service);
        this.replicas = (_b = props.replicas) !== null && _b !== void 0 ? _b : 1;
        this.podManagementPolicy = (_c = props.podManagementPolicy) !== null && _c !== void 0 ? _c : PodManagementPolicy.ORDERED_READY;
        this._podTemplate = new pod_1.PodTemplate(props);
        this._labelSelector = {};
        if ((_d = props.defaultSelector) !== null && _d !== void 0 ? _d : true) {
            const selector = 'cdk8s.statefulset';
            const matcher = cdk8s_1.Names.toLabelValue(this);
            this.podMetadata.addLabel(selector, matcher);
            this.selectByLabel(selector, matcher);
        }
        const selectors = Object.entries(this.labelSelector);
        for (const [k, v] of selectors) {
            this._service.addSelector(k, v);
        }
    }
    /**
     * Provides read/write access to the underlying pod metadata of the resource.
     *
     * @stability stable
     */
    get podMetadata() {
        return this._podTemplate.podMetadata;
    }
    /**
     * The labels this statefulset will match against in order to select pods.
     *
     * Returns a a copy. Use `selectByLabel()` to add labels.
     *
     * @stability stable
     */
    get labelSelector() {
        return { ...this._labelSelector };
    }
    /**
     * The containers belonging to the pod.
     *
     * Use `addContainer` to add containers.
     *
     * @stability stable
     */
    get containers() {
        return this._podTemplate.containers;
    }
    /**
     * The volumes associated with this pod.
     *
     * Use `addVolume` to add volumes.
     *
     * @stability stable
     */
    get volumes() {
        return this._podTemplate.volumes;
    }
    /**
     * Restart policy for all containers within the pod.
     *
     * @stability stable
     */
    get restartPolicy() {
        return this._podTemplate.restartPolicy;
    }
    /**
     * The service account used to run this pod.
     *
     * @stability stable
     */
    get serviceAccount() {
        return this._podTemplate.serviceAccount;
    }
    /**
     * Configure a label selector to this deployment.
     *
     * Pods that have the label will be selected by deployments configured with this spec.
     *
     * @param key - The label key.
     * @param value - The label value.
     * @stability stable
     */
    selectByLabel(key, value) {
        this._labelSelector[key] = value;
    }
    /**
     * Add a container to the pod.
     *
     * @stability stable
     */
    addContainer(container) {
        return this._podTemplate.addContainer(container);
    }
    /**
     * Add a volume to the pod.
     *
     * @stability stable
     */
    addVolume(volume) {
        return this._podTemplate.addVolume(volume);
    }
    /**
      * @internal
      */
    _toKube() {
        return {
            serviceName: this._service.name,
            replicas: this.replicas,
            template: this._podTemplate._toPodTemplateSpec(),
            selector: {
                matchLabels: this._labelSelector,
            },
            podManagementPolicy: this.podManagementPolicy,
        };
    }
}
exports.StatefulSet = StatefulSet;
_a = JSII_RTTI_SYMBOL_1;
StatefulSet[_a] = { fqn: "cdk8s-plus-17.StatefulSet", version: "1.0.0-beta.37" };
//# sourceMappingURL=data:application/json;base64,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