"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.IngressV1Beta1Backend = exports.IngressV1Beta1 = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const base_1 = require("./base");
const k8s = require("./imports/k8s");
/**
 * Ingress is a collection of rules that allow inbound connections to reach the endpoints defined by a backend.
 *
 * An Ingress can be configured to give services
 * externally-reachable urls, load balance traffic, terminate SSL, offer name
 * based virtual hosting etc.
 *
 * @stability stable
 */
class IngressV1Beta1 extends base_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _c;
        super(scope, id);
        this._rulesPerHost = {};
        this._tlsConfig = [];
        this.apiObject = new k8s.KubeIngressV1Beta1(this, 'Resource', {
            metadata: props.metadata,
            spec: {
                backend: cdk8s_1.Lazy.any({ produce: () => { var _c; return (_c = this._defaultBackend) === null || _c === void 0 ? void 0 : _c._toKube(); } }),
                rules: cdk8s_1.Lazy.any({ produce: () => this.synthRules() }),
                tls: cdk8s_1.Lazy.any({ produce: () => this.tlsConfig() }),
            },
        });
        if (props.defaultBackend) {
            this.addDefaultBackend(props.defaultBackend);
        }
        this.addRules(...(_c = props.rules) !== null && _c !== void 0 ? _c : []);
        if (props.tls) {
            this.addTls(props.tls);
        }
    }
    /**
     * (deprecated) Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    onValidate() {
        if (!this._defaultBackend && Object.keys(this._rulesPerHost).length === 0) {
            return ['ingress with no rules or default backend'];
        }
        return [];
    }
    /**
     * Defines the default backend for this ingress.
     *
     * A default backend capable of
     * servicing requests that don't match any rule.
     *
     * @param backend The backend to use for requests that do not match any rule.
     * @stability stable
     */
    addDefaultBackend(backend) {
        this.addRules({ backend });
    }
    /**
     * Specify a default backend for a specific host name.
     *
     * This backend will be used as a catch-all for requests
     * targeted to this host name (the `Host` header matches this value).
     *
     * @param host The host name to match.
     * @param backend The backend to route to.
     * @stability stable
     */
    addHostDefaultBackend(host, backend) {
        if (!host) {
            throw new Error('host must not be an empty string');
        }
        this.addRules({ host, backend });
    }
    /**
     * Adds an ingress rule applied to requests to a specific host and a specific HTTP path (the `Host` header matches this value).
     *
     * @param host The host name.
     * @param path The HTTP path.
     * @param backend The backend to route requests to.
     * @stability stable
     */
    addHostRule(host, path, backend) {
        if (!host) {
            throw new Error('host must not be an empty string');
        }
        this.addRules({ host, backend, path });
    }
    /**
     * Adds an ingress rule applied to requests sent to a specific HTTP path.
     *
     * @param path The HTTP path.
     * @param backend The backend to route requests to.
     * @stability stable
     */
    addRule(path, backend) {
        this.addRules({ backend, path });
    }
    /**
     * Adds rules to this ingress.
     *
     * @param rules The rules to add.
     * @stability stable
     */
    addRules(...rules) {
        var _c, _d;
        for (const rule of rules) {
            // default backend is not really a rule
            if (!rule.host && !rule.path) {
                if (this._defaultBackend) {
                    throw new Error('a default backend is already defined for this ingress');
                }
                this._defaultBackend = rule.backend;
                continue;
            }
            const host = (_c = rule.host) !== null && _c !== void 0 ? _c : '';
            const backend = rule.backend;
            const path = rule.path;
            if (path && !path.startsWith('/')) {
                throw new Error(`ingress paths must begin with a "/": ${path}`);
            }
            const routes = this._rulesPerHost[host] = (_d = this._rulesPerHost[host]) !== null && _d !== void 0 ? _d : [];
            // check if we already have a rule for this host/path
            if (routes.find(r => r.path === path)) {
                throw new Error(`there is already an ingress rule for ${host}${path}`);
            }
            routes.push({ backend: backend._toKube(), path });
        }
    }
    synthRules() {
        const rules = new Array();
        for (const [host, paths] of Object.entries(this._rulesPerHost)) {
            rules.push({
                host: host ? host : undefined,
                http: { paths: paths.sort(sortByPath) },
            });
        }
        return rules.length > 0 ? rules : undefined;
    }
    /**
     * @stability stable
     */
    addTls(tls) {
        this._tlsConfig.push(...tls);
    }
    tlsConfig() {
        var _c;
        if (this._tlsConfig.length == 0) {
            return undefined;
        }
        const tls = new Array();
        for (const entry of this._tlsConfig) {
            tls.push({
                hosts: entry.hosts,
                secretName: (_c = entry.secret) === null || _c === void 0 ? void 0 : _c.name,
            });
        }
        return tls;
    }
}
exports.IngressV1Beta1 = IngressV1Beta1;
_a = JSII_RTTI_SYMBOL_1;
IngressV1Beta1[_a] = { fqn: "cdk8s-plus-17.IngressV1Beta1", version: "1.0.0-beta.45" };
/**
 * The backend for an ingress path.
 *
 * @stability stable
 */
class IngressV1Beta1Backend {
    constructor(backend) {
        this.backend = backend;
    }
    /**
     * A Kubernetes `Service` to use as the backend for this path.
     *
     * @param service The service object.
     * @stability stable
     */
    static fromService(service, options = {}) {
        if (service.ports.length === 0) {
            throw new Error('service does not expose any ports');
        }
        let servicePort;
        if (service.ports.length === 1) {
            servicePort = service.ports[0].port;
        }
        else {
            if (options.port !== undefined) {
                const found = service.ports.find(p => p.port === options.port);
                if (found) {
                    servicePort = found.port;
                }
                else {
                    throw new Error(`service exposes ports ${service.ports.map(p => p.port).join(',')} but backend is defined to use port ${options.port}`);
                }
            }
            else {
                throw new Error(`unable to determine service port since service exposes multiple ports: ${service.ports.map(x => x.port).join(',')}`);
            }
        }
        if (options.port !== undefined && servicePort !== options.port) {
            throw new Error(`backend defines port ${options.port} but service exposes port ${servicePort}`);
        }
        return new IngressV1Beta1Backend({
            serviceName: service.name,
            servicePort,
        });
    }
    /**
     * @internal
     */
    _toKube() { return this.backend; }
}
exports.IngressV1Beta1Backend = IngressV1Beta1Backend;
_b = JSII_RTTI_SYMBOL_1;
IngressV1Beta1Backend[_b] = { fqn: "cdk8s-plus-17.IngressV1Beta1Backend", version: "1.0.0-beta.45" };
function sortByPath(lhs, rhs) {
    var _c, _d;
    const p1 = (_c = lhs.path) !== null && _c !== void 0 ? _c : '';
    const p2 = (_d = rhs.path) !== null && _d !== void 0 ? _d : '';
    return p1.localeCompare(p2);
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5ncmVzcy12MWJldGExLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vc3JjL2luZ3Jlc3MtdjFiZXRhMS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLGlDQUF3QztBQUV4QyxpQ0FBaUQ7QUFDakQscUNBQXFDOzs7Ozs7Ozs7O0FBa0JyQyxNQUFhLGNBQWUsU0FBUSxlQUFROzs7O0lBUzFDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsUUFBNkIsRUFBRTs7UUFDdkUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUxGLGtCQUFhLEdBQThDLEVBQUUsQ0FBQztRQUU5RCxlQUFVLEdBQXdCLEVBQUUsQ0FBQztRQUtwRCxJQUFJLENBQUMsU0FBUyxHQUFHLElBQUksR0FBRyxDQUFDLGtCQUFrQixDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDNUQsUUFBUSxFQUFFLEtBQUssQ0FBQyxRQUFRO1lBQ3hCLElBQUksRUFBRTtnQkFDSixPQUFPLEVBQUUsWUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsd0JBQUMsSUFBSSxDQUFDLGVBQWUsMENBQUUsT0FBTyxLQUFFLEVBQUUsQ0FBQztnQkFDckUsS0FBSyxFQUFFLFlBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxFQUFFLENBQUM7Z0JBQ3JELEdBQUcsRUFBRSxZQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsRUFBRSxDQUFDO2FBQ25EO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxLQUFLLENBQUMsY0FBYyxFQUFFO1lBQ3hCLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUFDLENBQUM7U0FDOUM7UUFFRCxJQUFJLENBQUMsUUFBUSxDQUFDLFNBQUcsS0FBSyxDQUFDLEtBQUssbUNBQUksRUFBRSxDQUFDLENBQUM7UUFFcEMsSUFBSSxLQUFLLENBQUMsR0FBRyxFQUFFO1lBQ2IsSUFBSSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUM7U0FDeEI7SUFDSCxDQUFDOzs7Ozs7Ozs7SUFFUyxVQUFVO1FBQ2xCLElBQUksQ0FBQyxJQUFJLENBQUMsZUFBZSxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDekUsT0FBTyxDQUFDLDBDQUEwQyxDQUFDLENBQUM7U0FDckQ7UUFDRCxPQUFPLEVBQUUsQ0FBQztJQUNaLENBQUM7Ozs7Ozs7Ozs7SUFHTSxpQkFBaUIsQ0FBQyxPQUE4QjtRQUNyRCxJQUFJLENBQUMsUUFBUSxDQUFDLEVBQUUsT0FBTyxFQUFFLENBQUMsQ0FBQztJQUM3QixDQUFDOzs7Ozs7Ozs7OztJQUdNLHFCQUFxQixDQUFDLElBQVksRUFBRSxPQUE4QjtRQUN2RSxJQUFJLENBQUMsSUFBSSxFQUFFO1lBQUUsTUFBTSxJQUFJLEtBQUssQ0FBQyxrQ0FBa0MsQ0FBQyxDQUFDO1NBQUU7UUFDbkUsSUFBSSxDQUFDLFFBQVEsQ0FBQyxFQUFFLElBQUksRUFBRSxPQUFPLEVBQUUsQ0FBQyxDQUFDO0lBQ25DLENBQUM7Ozs7Ozs7OztJQUdNLFdBQVcsQ0FBQyxJQUFZLEVBQUUsSUFBWSxFQUFFLE9BQThCO1FBQzNFLElBQUksQ0FBQyxJQUFJLEVBQUU7WUFBRSxNQUFNLElBQUksS0FBSyxDQUFDLGtDQUFrQyxDQUFDLENBQUM7U0FBRTtRQUNuRSxJQUFJLENBQUMsUUFBUSxDQUFDLEVBQUUsSUFBSSxFQUFFLE9BQU8sRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO0lBQ3pDLENBQUM7Ozs7Ozs7O0lBR00sT0FBTyxDQUFDLElBQVksRUFBRSxPQUE4QjtRQUN6RCxJQUFJLENBQUMsUUFBUSxDQUFDLEVBQUUsT0FBTyxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7SUFDbkMsQ0FBQzs7Ozs7OztJQUdNLFFBQVEsQ0FBQyxHQUFHLEtBQTJCOztRQUM1QyxLQUFLLE1BQU0sSUFBSSxJQUFJLEtBQUssRUFBRTtZQUV4Qix1Q0FBdUM7WUFDdkMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFO2dCQUM1QixJQUFJLElBQUksQ0FBQyxlQUFlLEVBQUU7b0JBQ3hCLE1BQU0sSUFBSSxLQUFLLENBQUMsdURBQXVELENBQUMsQ0FBQztpQkFDMUU7Z0JBQ0QsSUFBSSxDQUFDLGVBQWUsR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDO2dCQUNwQyxTQUFTO2FBQ1Y7WUFFRCxNQUFNLElBQUksU0FBRyxJQUFJLENBQUMsSUFBSSxtQ0FBSSxFQUFFLENBQUM7WUFDN0IsTUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQztZQUM3QixNQUFNLElBQUksR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDO1lBRXZCLElBQUksSUFBSSxJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsRUFBRTtnQkFDakMsTUFBTSxJQUFJLEtBQUssQ0FBQyx3Q0FBd0MsSUFBSSxFQUFFLENBQUMsQ0FBQzthQUNqRTtZQUVELE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLFNBQUcsSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsbUNBQUksRUFBRSxDQUFDO1lBRXpFLHFEQUFxRDtZQUNyRCxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsSUFBSSxLQUFLLElBQUksQ0FBQyxFQUFFO2dCQUNyQyxNQUFNLElBQUksS0FBSyxDQUFDLHdDQUF3QyxJQUFJLEdBQUcsSUFBSSxFQUFFLENBQUMsQ0FBQzthQUN4RTtZQUVELE1BQU0sQ0FBQyxJQUFJLENBQUMsRUFBRSxPQUFPLEVBQUUsT0FBTyxDQUFDLE9BQU8sRUFBRSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7U0FDbkQ7SUFDSCxDQUFDO0lBRU8sVUFBVTtRQUNoQixNQUFNLEtBQUssR0FBRyxJQUFJLEtBQUssRUFBbUIsQ0FBQztRQUUzQyxLQUFLLE1BQU0sQ0FBQyxJQUFJLEVBQUUsS0FBSyxDQUFDLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLEVBQUU7WUFDOUQsS0FBSyxDQUFDLElBQUksQ0FBQztnQkFDVCxJQUFJLEVBQUUsSUFBSSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLFNBQVM7Z0JBQzdCLElBQUksRUFBRSxFQUFFLEtBQUssRUFBRSxLQUFLLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxFQUFFO2FBQ3hDLENBQUMsQ0FBQztTQUNKO1FBRUQsT0FBTyxLQUFLLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUM7SUFDOUMsQ0FBQzs7OztJQUVNLE1BQU0sQ0FBQyxHQUF3QjtRQUNwQyxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxHQUFHLEdBQUcsQ0FBQyxDQUFDO0lBQy9CLENBQUM7SUFFTyxTQUFTOztRQUNmLElBQUksSUFBSSxDQUFDLFVBQVUsQ0FBQyxNQUFNLElBQUksQ0FBQyxFQUFFO1lBQy9CLE9BQU8sU0FBUyxDQUFDO1NBQ2xCO1FBRUQsTUFBTSxHQUFHLEdBQUcsSUFBSSxLQUFLLEVBQWtCLENBQUM7UUFDeEMsS0FBSyxNQUFNLEtBQUssSUFBSSxJQUFJLENBQUMsVUFBVSxFQUFFO1lBQ25DLEdBQUcsQ0FBQyxJQUFJLENBQUM7Z0JBQ1AsS0FBSyxFQUFFLEtBQUssQ0FBQyxLQUFLO2dCQUNsQixVQUFVLFFBQUUsS0FBSyxDQUFDLE1BQU0sMENBQUUsSUFBSTthQUMvQixDQUFDLENBQUM7U0FDSjtRQUVELE9BQU8sR0FBRyxDQUFDO0lBQ2IsQ0FBQzs7QUE1SEgsd0NBNkhDOzs7Ozs7OztBQVNELE1BQWEscUJBQXFCO0lBaUNoQyxZQUFxQyxPQUEyQjtRQUEzQixZQUFPLEdBQVAsT0FBTyxDQUFvQjtJQUVoRSxDQUFDOzs7Ozs7O0lBakNNLE1BQU0sQ0FBQyxXQUFXLENBQUMsT0FBZ0IsRUFBRSxVQUE4QyxFQUFFO1FBQzFGLElBQUksT0FBTyxDQUFDLEtBQUssQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQzlCLE1BQU0sSUFBSSxLQUFLLENBQUMsbUNBQW1DLENBQUMsQ0FBQztTQUN0RDtRQUVELElBQUksV0FBVyxDQUFDO1FBQ2hCLElBQUksT0FBTyxDQUFDLEtBQUssQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQzlCLFdBQVcsR0FBRyxPQUFPLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQztTQUNyQzthQUFNO1lBQ0wsSUFBSSxPQUFPLENBQUMsSUFBSSxLQUFLLFNBQVMsRUFBRTtnQkFDOUIsTUFBTSxLQUFLLEdBQUcsT0FBTyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsSUFBSSxLQUFLLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQztnQkFDL0QsSUFBSSxLQUFLLEVBQUU7b0JBQ1QsV0FBVyxHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUM7aUJBQzFCO3FCQUFNO29CQUNMLE1BQU0sSUFBSSxLQUFLLENBQUMseUJBQXlCLE9BQU8sQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsdUNBQXVDLE9BQU8sQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDO2lCQUN6STthQUNGO2lCQUFNO2dCQUNMLE1BQU0sSUFBSSxLQUFLLENBQUMsMEVBQTBFLE9BQU8sQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLENBQUM7YUFDdkk7U0FDRjtRQUVELElBQUksT0FBTyxDQUFDLElBQUksS0FBSyxTQUFTLElBQUksV0FBVyxLQUFLLE9BQU8sQ0FBQyxJQUFJLEVBQUU7WUFDOUQsTUFBTSxJQUFJLEtBQUssQ0FBQyx3QkFBd0IsT0FBTyxDQUFDLElBQUksNkJBQTZCLFdBQVcsRUFBRSxDQUFDLENBQUM7U0FDakc7UUFFRCxPQUFPLElBQUkscUJBQXFCLENBQUM7WUFDL0IsV0FBVyxFQUFFLE9BQU8sQ0FBQyxJQUFJO1lBQ3pCLFdBQVc7U0FDWixDQUFDLENBQUM7SUFDTCxDQUFDO0lBTUQ7O09BRUc7SUFDSSxPQUFPLEtBQUssT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQzs7QUF4QzNDLHNEQXlDQzs7O0FBd0JELFNBQVMsVUFBVSxDQUFDLEdBQXdCLEVBQUUsR0FBd0I7O0lBQ3BFLE1BQU0sRUFBRSxTQUFHLEdBQUcsQ0FBQyxJQUFJLG1DQUFJLEVBQUUsQ0FBQztJQUMxQixNQUFNLEVBQUUsU0FBRyxHQUFHLENBQUMsSUFBSSxtQ0FBSSxFQUFFLENBQUM7SUFDMUIsT0FBTyxFQUFFLENBQUMsYUFBYSxDQUFDLEVBQUUsQ0FBQyxDQUFDO0FBQzlCLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBBcGlPYmplY3QsIExhenkgfSBmcm9tICdjZGs4cyc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IFJlc291cmNlLCBSZXNvdXJjZVByb3BzIH0gZnJvbSAnLi9iYXNlJztcbmltcG9ydCAqIGFzIGs4cyBmcm9tICcuL2ltcG9ydHMvazhzJztcbmltcG9ydCB7IElTZWNyZXQgfSBmcm9tICcuL3NlY3JldCc7XG5pbXBvcnQgeyBTZXJ2aWNlIH0gZnJvbSAnLi9zZXJ2aWNlJztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIEluZ3Jlc3NWMUJldGExUHJvcHMgZXh0ZW5kcyBSZXNvdXJjZVByb3BzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGRlZmF1bHRCYWNrZW5kPzogSW5ncmVzc1YxQmV0YTFCYWNrZW5kO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHJ1bGVzPzogSW5ncmVzc1YxQmV0YTFSdWxlW107XG5cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdGxzPzogSW5ncmVzc1YxQmV0YTFUbHNbXTtcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgSW5ncmVzc1YxQmV0YTEgZXh0ZW5kcyBSZXNvdXJjZSB7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwcm90ZWN0ZWQgcmVhZG9ubHkgYXBpT2JqZWN0OiBBcGlPYmplY3Q7XG5cbiAgcHJpdmF0ZSByZWFkb25seSBfcnVsZXNQZXJIb3N0OiB7IFtob3N0OiBzdHJpbmddOiBrOHMuSHR0cEluZ3Jlc3NQYXRoW10gfSA9IHt9O1xuICBwcml2YXRlIF9kZWZhdWx0QmFja2VuZD86IEluZ3Jlc3NWMUJldGExQmFja2VuZDtcbiAgcHJpdmF0ZSByZWFkb25seSBfdGxzQ29uZmlnOiBJbmdyZXNzVjFCZXRhMVRsc1tdID0gW107XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IEluZ3Jlc3NWMUJldGExUHJvcHMgPSB7fSkge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICB0aGlzLmFwaU9iamVjdCA9IG5ldyBrOHMuS3ViZUluZ3Jlc3NWMUJldGExKHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIG1ldGFkYXRhOiBwcm9wcy5tZXRhZGF0YSxcbiAgICAgIHNwZWM6IHtcbiAgICAgICAgYmFja2VuZDogTGF6eS5hbnkoeyBwcm9kdWNlOiAoKSA9PiB0aGlzLl9kZWZhdWx0QmFja2VuZD8uX3RvS3ViZSgpIH0pLFxuICAgICAgICBydWxlczogTGF6eS5hbnkoeyBwcm9kdWNlOiAoKSA9PiB0aGlzLnN5bnRoUnVsZXMoKSB9KSxcbiAgICAgICAgdGxzOiBMYXp5LmFueSh7IHByb2R1Y2U6ICgpID0+IHRoaXMudGxzQ29uZmlnKCkgfSksXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgaWYgKHByb3BzLmRlZmF1bHRCYWNrZW5kKSB7XG4gICAgICB0aGlzLmFkZERlZmF1bHRCYWNrZW5kKHByb3BzLmRlZmF1bHRCYWNrZW5kKTtcbiAgICB9XG5cbiAgICB0aGlzLmFkZFJ1bGVzKC4uLnByb3BzLnJ1bGVzID8/IFtdKTtcblxuICAgIGlmIChwcm9wcy50bHMpIHtcbiAgICAgIHRoaXMuYWRkVGxzKHByb3BzLnRscyk7XG4gICAgfVxuICB9XG5cbiAgcHJvdGVjdGVkIG9uVmFsaWRhdGUoKSB7XG4gICAgaWYgKCF0aGlzLl9kZWZhdWx0QmFja2VuZCAmJiBPYmplY3Qua2V5cyh0aGlzLl9ydWxlc1Blckhvc3QpLmxlbmd0aCA9PT0gMCkge1xuICAgICAgcmV0dXJuIFsnaW5ncmVzcyB3aXRoIG5vIHJ1bGVzIG9yIGRlZmF1bHQgYmFja2VuZCddO1xuICAgIH1cbiAgICByZXR1cm4gW107XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGFkZERlZmF1bHRCYWNrZW5kKGJhY2tlbmQ6IEluZ3Jlc3NWMUJldGExQmFja2VuZCkge1xuICAgIHRoaXMuYWRkUnVsZXMoeyBiYWNrZW5kIH0pO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGFkZEhvc3REZWZhdWx0QmFja2VuZChob3N0OiBzdHJpbmcsIGJhY2tlbmQ6IEluZ3Jlc3NWMUJldGExQmFja2VuZCkge1xuICAgIGlmICghaG9zdCkgeyB0aHJvdyBuZXcgRXJyb3IoJ2hvc3QgbXVzdCBub3QgYmUgYW4gZW1wdHkgc3RyaW5nJyk7IH1cbiAgICB0aGlzLmFkZFJ1bGVzKHsgaG9zdCwgYmFja2VuZCB9KTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYWRkSG9zdFJ1bGUoaG9zdDogc3RyaW5nLCBwYXRoOiBzdHJpbmcsIGJhY2tlbmQ6IEluZ3Jlc3NWMUJldGExQmFja2VuZCkge1xuICAgIGlmICghaG9zdCkgeyB0aHJvdyBuZXcgRXJyb3IoJ2hvc3QgbXVzdCBub3QgYmUgYW4gZW1wdHkgc3RyaW5nJyk7IH1cbiAgICB0aGlzLmFkZFJ1bGVzKHsgaG9zdCwgYmFja2VuZCwgcGF0aCB9KTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYWRkUnVsZShwYXRoOiBzdHJpbmcsIGJhY2tlbmQ6IEluZ3Jlc3NWMUJldGExQmFja2VuZCkge1xuICAgIHRoaXMuYWRkUnVsZXMoeyBiYWNrZW5kLCBwYXRoIH0pO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGFkZFJ1bGVzKC4uLnJ1bGVzOiBJbmdyZXNzVjFCZXRhMVJ1bGVbXSkge1xuICAgIGZvciAoY29uc3QgcnVsZSBvZiBydWxlcykge1xuXG4gICAgICAvLyBkZWZhdWx0IGJhY2tlbmQgaXMgbm90IHJlYWxseSBhIHJ1bGVcbiAgICAgIGlmICghcnVsZS5ob3N0ICYmICFydWxlLnBhdGgpIHtcbiAgICAgICAgaWYgKHRoaXMuX2RlZmF1bHRCYWNrZW5kKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdhIGRlZmF1bHQgYmFja2VuZCBpcyBhbHJlYWR5IGRlZmluZWQgZm9yIHRoaXMgaW5ncmVzcycpO1xuICAgICAgICB9XG4gICAgICAgIHRoaXMuX2RlZmF1bHRCYWNrZW5kID0gcnVsZS5iYWNrZW5kO1xuICAgICAgICBjb250aW51ZTtcbiAgICAgIH1cblxuICAgICAgY29uc3QgaG9zdCA9IHJ1bGUuaG9zdCA/PyAnJztcbiAgICAgIGNvbnN0IGJhY2tlbmQgPSBydWxlLmJhY2tlbmQ7XG4gICAgICBjb25zdCBwYXRoID0gcnVsZS5wYXRoO1xuXG4gICAgICBpZiAocGF0aCAmJiAhcGF0aC5zdGFydHNXaXRoKCcvJykpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBpbmdyZXNzIHBhdGhzIG11c3QgYmVnaW4gd2l0aCBhIFwiL1wiOiAke3BhdGh9YCk7XG4gICAgICB9XG5cbiAgICAgIGNvbnN0IHJvdXRlcyA9IHRoaXMuX3J1bGVzUGVySG9zdFtob3N0XSA9IHRoaXMuX3J1bGVzUGVySG9zdFtob3N0XSA/PyBbXTtcblxuICAgICAgLy8gY2hlY2sgaWYgd2UgYWxyZWFkeSBoYXZlIGEgcnVsZSBmb3IgdGhpcyBob3N0L3BhdGhcbiAgICAgIGlmIChyb3V0ZXMuZmluZChyID0+IHIucGF0aCA9PT0gcGF0aCkpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGB0aGVyZSBpcyBhbHJlYWR5IGFuIGluZ3Jlc3MgcnVsZSBmb3IgJHtob3N0fSR7cGF0aH1gKTtcbiAgICAgIH1cblxuICAgICAgcm91dGVzLnB1c2goeyBiYWNrZW5kOiBiYWNrZW5kLl90b0t1YmUoKSwgcGF0aCB9KTtcbiAgICB9XG4gIH1cblxuICBwcml2YXRlIHN5bnRoUnVsZXMoKTogdW5kZWZpbmVkIHwgazhzLkluZ3Jlc3NSdWxlW10ge1xuICAgIGNvbnN0IHJ1bGVzID0gbmV3IEFycmF5PGs4cy5JbmdyZXNzUnVsZT4oKTtcblxuICAgIGZvciAoY29uc3QgW2hvc3QsIHBhdGhzXSBvZiBPYmplY3QuZW50cmllcyh0aGlzLl9ydWxlc1Blckhvc3QpKSB7XG4gICAgICBydWxlcy5wdXNoKHtcbiAgICAgICAgaG9zdDogaG9zdCA/IGhvc3QgOiB1bmRlZmluZWQsXG4gICAgICAgIGh0dHA6IHsgcGF0aHM6IHBhdGhzLnNvcnQoc29ydEJ5UGF0aCkgfSxcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIHJldHVybiBydWxlcy5sZW5ndGggPiAwID8gcnVsZXMgOiB1bmRlZmluZWQ7XG4gIH1cblxuICBwdWJsaWMgYWRkVGxzKHRsczogSW5ncmVzc1YxQmV0YTFUbHNbXSkge1xuICAgIHRoaXMuX3Rsc0NvbmZpZy5wdXNoKC4uLnRscyk7XG4gIH1cblxuICBwcml2YXRlIHRsc0NvbmZpZygpOiB1bmRlZmluZWQgfCBrOHMuSW5ncmVzc1Rsc1tdIHtcbiAgICBpZiAodGhpcy5fdGxzQ29uZmlnLmxlbmd0aCA9PSAwKSB7XG4gICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgIH1cblxuICAgIGNvbnN0IHRscyA9IG5ldyBBcnJheTxrOHMuSW5ncmVzc1Rscz4oKTtcbiAgICBmb3IgKGNvbnN0IGVudHJ5IG9mIHRoaXMuX3Rsc0NvbmZpZykge1xuICAgICAgdGxzLnB1c2goe1xuICAgICAgICBob3N0czogZW50cnkuaG9zdHMsXG4gICAgICAgIHNlY3JldE5hbWU6IGVudHJ5LnNlY3JldD8ubmFtZSxcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIHJldHVybiB0bHM7XG4gIH1cbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBTZXJ2aWNlSW5ncmVzc1YxQmV0YUJhY2tlbmRPcHRpb25zIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHBvcnQ/OiBudW1iZXI7XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBJbmdyZXNzVjFCZXRhMUJhY2tlbmQge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBmcm9tU2VydmljZShzZXJ2aWNlOiBTZXJ2aWNlLCBvcHRpb25zOiBTZXJ2aWNlSW5ncmVzc1YxQmV0YUJhY2tlbmRPcHRpb25zID0ge30pIHtcbiAgICBpZiAoc2VydmljZS5wb3J0cy5sZW5ndGggPT09IDApIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignc2VydmljZSBkb2VzIG5vdCBleHBvc2UgYW55IHBvcnRzJyk7XG4gICAgfVxuXG4gICAgbGV0IHNlcnZpY2VQb3J0O1xuICAgIGlmIChzZXJ2aWNlLnBvcnRzLmxlbmd0aCA9PT0gMSkge1xuICAgICAgc2VydmljZVBvcnQgPSBzZXJ2aWNlLnBvcnRzWzBdLnBvcnQ7XG4gICAgfSBlbHNlIHtcbiAgICAgIGlmIChvcHRpb25zLnBvcnQgIT09IHVuZGVmaW5lZCkge1xuICAgICAgICBjb25zdCBmb3VuZCA9IHNlcnZpY2UucG9ydHMuZmluZChwID0+IHAucG9ydCA9PT0gb3B0aW9ucy5wb3J0KTtcbiAgICAgICAgaWYgKGZvdW5kKSB7XG4gICAgICAgICAgc2VydmljZVBvcnQgPSBmb3VuZC5wb3J0O1xuICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgc2VydmljZSBleHBvc2VzIHBvcnRzICR7c2VydmljZS5wb3J0cy5tYXAocCA9PiBwLnBvcnQpLmpvaW4oJywnKX0gYnV0IGJhY2tlbmQgaXMgZGVmaW5lZCB0byB1c2UgcG9ydCAke29wdGlvbnMucG9ydH1gKTtcbiAgICAgICAgfVxuICAgICAgfSBlbHNlIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGB1bmFibGUgdG8gZGV0ZXJtaW5lIHNlcnZpY2UgcG9ydCBzaW5jZSBzZXJ2aWNlIGV4cG9zZXMgbXVsdGlwbGUgcG9ydHM6ICR7c2VydmljZS5wb3J0cy5tYXAoeCA9PiB4LnBvcnQpLmpvaW4oJywnKX1gKTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICBpZiAob3B0aW9ucy5wb3J0ICE9PSB1bmRlZmluZWQgJiYgc2VydmljZVBvcnQgIT09IG9wdGlvbnMucG9ydCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBiYWNrZW5kIGRlZmluZXMgcG9ydCAke29wdGlvbnMucG9ydH0gYnV0IHNlcnZpY2UgZXhwb3NlcyBwb3J0ICR7c2VydmljZVBvcnR9YCk7XG4gICAgfVxuXG4gICAgcmV0dXJuIG5ldyBJbmdyZXNzVjFCZXRhMUJhY2tlbmQoe1xuICAgICAgc2VydmljZU5hbWU6IHNlcnZpY2UubmFtZSxcbiAgICAgIHNlcnZpY2VQb3J0LFxuICAgIH0pO1xuICB9XG5cbiAgcHJpdmF0ZSBjb25zdHJ1Y3Rvcihwcml2YXRlIHJlYWRvbmx5IGJhY2tlbmQ6IGs4cy5JbmdyZXNzQmFja2VuZCkge1xuXG4gIH1cblxuICAvKipcbiAgICogQGludGVybmFsXG4gICAqL1xuICBwdWJsaWMgX3RvS3ViZSgpIHsgcmV0dXJuIHRoaXMuYmFja2VuZDsgfVxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIEluZ3Jlc3NWMUJldGExUnVsZSB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGJhY2tlbmQ6IEluZ3Jlc3NWMUJldGExQmFja2VuZDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBob3N0Pzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHBhdGg/OiBzdHJpbmc7XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIEluZ3Jlc3NWMUJldGExVGxzIHtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgaG9zdHM/OiBzdHJpbmdbXTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHNlY3JldD86IElTZWNyZXQ7XG59XG5cbmZ1bmN0aW9uIHNvcnRCeVBhdGgobGhzOiBrOHMuSHR0cEluZ3Jlc3NQYXRoLCByaHM6IGs4cy5IdHRwSW5ncmVzc1BhdGgpIHtcbiAgY29uc3QgcDEgPSBsaHMucGF0aCA/PyAnJztcbiAgY29uc3QgcDIgPSByaHMucGF0aCA/PyAnJztcbiAgcmV0dXJuIHAxLmxvY2FsZUNvbXBhcmUocDIpO1xufVxuIl19