"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssociationManager = void 0;
const servicecatalog_generated_1 = require("../servicecatalog.generated");
const util_1 = require("./util");
const validation_1 = require("./validation");
class AssociationManager {
    static associateProductWithPortfolio(portfolio, product, options) {
        validation_1.InputValidator.validateLength(this.prettyPrintAssociation(portfolio, product), 'description', 0, 2000, options === null || options === void 0 ? void 0 : options.description);
        const associationKey = util_1.hashValues(portfolio.node.addr, product.node.addr, product.stack.node.addr);
        const constructId = `PortfolioProductAssociation${associationKey}`;
        const existingAssociation = portfolio.node.tryFindChild(constructId);
        const cfnAssociation = existingAssociation
            ? existingAssociation
            : new servicecatalog_generated_1.CfnPortfolioProductAssociation(portfolio, constructId, {
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
            });
        return {
            associationKey: associationKey,
            cfnPortfolioProductAssociation: cfnAssociation,
        };
    }
    static constrainTagUpdates(portfolio, product, options) {
        const association = this.associateProductWithPortfolio(portfolio, product, options);
        const constructId = `ResourceUpdateConstraint${association.associationKey}`;
        if (!portfolio.node.tryFindChild(constructId)) {
            const constraint = new servicecatalog_generated_1.CfnResourceUpdateConstraint(portfolio, constructId, {
                acceptLanguage: options.messageLanguage,
                description: options.description,
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
                tagUpdateOnProvisionedProduct: options.allow === false ? 'NOT_ALLOWED' : 'ALLOWED',
            });
            // Add dependsOn to force proper order in deployment.
            constraint.addDependsOn(association.cfnPortfolioProductAssociation);
        }
        else {
            throw new Error(`Cannot have multiple tag update constraints for association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
    }
    static notifyOnStackEvents(portfolio, product, topic, options) {
        const association = this.associateProductWithPortfolio(portfolio, product, options);
        const constructId = `LaunchNotificationConstraint${util_1.hashValues(topic.node.addr, topic.stack.node.addr, association.associationKey)}`;
        if (!portfolio.node.tryFindChild(constructId)) {
            const constraint = new servicecatalog_generated_1.CfnLaunchNotificationConstraint(portfolio, constructId, {
                acceptLanguage: options.messageLanguage,
                description: options.description,
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
                notificationArns: [topic.topicArn],
            });
            // Add dependsOn to force proper order in deployment.
            constraint.addDependsOn(association.cfnPortfolioProductAssociation);
        }
        else {
            throw new Error(`Topic ${topic.node.path} is already subscribed to association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
    }
    static constrainCloudFormationParameters(portfolio, product, options) {
        const association = this.associateProductWithPortfolio(portfolio, product, options);
        const constructId = `LaunchTemplateConstraint${util_1.hashValues(association.associationKey, options.rule.ruleName)}`;
        if (!portfolio.node.tryFindChild(constructId)) {
            const constraint = new servicecatalog_generated_1.CfnLaunchTemplateConstraint(portfolio, constructId, {
                acceptLanguage: options.messageLanguage,
                description: options.description,
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
                rules: this.formatTemplateRule(portfolio.stack, options.rule),
            });
            // Add dependsOn to force proper order in deployment.
            constraint.addDependsOn(association.cfnPortfolioProductAssociation);
        }
        else {
            throw new Error(`Provisioning rule ${options.rule.ruleName} already configured on association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
    }
    static setLaunchRole(portfolio, product, launchRole, options) {
        this.setLaunchRoleConstraint(portfolio, product, options, {
            roleArn: launchRole.roleArn,
        });
    }
    static setLocalLaunchRoleName(portfolio, product, launchRoleName, options) {
        this.setLaunchRoleConstraint(portfolio, product, options, {
            localRoleName: launchRoleName,
        });
    }
    static deployWithStackSets(portfolio, product, options) {
        var _a;
        const association = this.associateProductWithPortfolio(portfolio, product, options);
        // Check if a launch role has already been set.
        if (portfolio.node.tryFindChild(this.launchRoleConstraintLogicalId(association.associationKey))) {
            throw new Error(`Cannot configure StackSet deployment when a launch role is already defined for association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
        const constructId = this.stackSetConstraintLogicalId(association.associationKey);
        if (!portfolio.node.tryFindChild(constructId)) {
            const constraint = new servicecatalog_generated_1.CfnStackSetConstraint(portfolio, constructId, {
                acceptLanguage: options.messageLanguage,
                description: (_a = options.description) !== null && _a !== void 0 ? _a : '',
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
                accountList: options.accounts,
                regionList: options.regions,
                adminRole: options.adminRole.roleArn,
                executionRole: options.executionRoleName,
                stackInstanceControl: options.allowStackSetInstanceOperations ? 'ALLOWED' : 'NOT_ALLOWED',
            });
            // Add dependsOn to force proper order in deployment.
            constraint.addDependsOn(association.cfnPortfolioProductAssociation);
        }
        else {
            throw new Error(`Cannot configure multiple StackSet deployment constraints for association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
    }
    static associateTagOptions(resource, resourceId, tagOptions) {
        for (const cfnTagOption of tagOptions._cfnTagOptions) {
            const tagAssocationConstructId = `TagOptionAssociation${util_1.hashValues(cfnTagOption.key, cfnTagOption.value, resource.node.addr)}`;
            if (!resource.node.tryFindChild(tagAssocationConstructId)) {
                new servicecatalog_generated_1.CfnTagOptionAssociation(resource, tagAssocationConstructId, {
                    resourceId: resourceId,
                    tagOptionId: cfnTagOption.ref,
                });
            }
        }
    }
    static setLaunchRoleConstraint(portfolio, product, options, roleOptions) {
        const association = this.associateProductWithPortfolio(portfolio, product, options);
        // Check if a stackset deployment constraint has already been configured.
        if (portfolio.node.tryFindChild(this.stackSetConstraintLogicalId(association.associationKey))) {
            throw new Error(`Cannot set launch role when a StackSet rule is already defined for association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
        const constructId = this.launchRoleConstraintLogicalId(association.associationKey);
        if (!portfolio.node.tryFindChild(constructId)) {
            const constraint = new servicecatalog_generated_1.CfnLaunchRoleConstraint(portfolio, constructId, {
                acceptLanguage: options.messageLanguage,
                description: options.description,
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
                roleArn: roleOptions.roleArn,
                localRoleName: roleOptions.localRoleName,
            });
            // Add dependsOn to force proper order in deployment.
            constraint.addDependsOn(association.cfnPortfolioProductAssociation);
        }
        else {
            throw new Error(`Cannot set multiple launch roles for association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
    }
    static stackSetConstraintLogicalId(associationKey) {
        return `StackSetConstraint${associationKey}`;
    }
    static launchRoleConstraintLogicalId(associationKey) {
        return `LaunchRoleConstraint${associationKey}`;
    }
    static prettyPrintAssociation(portfolio, product) {
        return `- Portfolio: ${portfolio.node.path} | Product: ${product.node.path}`;
    }
    static formatTemplateRule(stack, rule) {
        return JSON.stringify({
            [rule.ruleName]: {
                Assertions: this.formatAssertions(stack, rule.assertions),
                RuleCondition: rule.condition ? stack.resolve(rule.condition) : undefined,
            },
        });
    }
    static formatAssertions(stack, assertions) {
        return assertions.reduce((formattedAssertions, assertion) => {
            formattedAssertions.push({
                Assert: stack.resolve(assertion.assert),
                AssertDescription: assertion.description,
            });
            return formattedAssertions;
        }, new Array());
    }
    ;
}
exports.AssociationManager = AssociationManager;
//# sourceMappingURL=data:application/json;base64,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