"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseCluster = exports.ServerlessLaravel = exports.ServerlessApi = void 0;
const cdk = require("@aws-cdk/core");
const apigateway = require("@aws-cdk/aws-apigatewayv2");
const lambda = require("@aws-cdk/aws-lambda");
const rds = require("@aws-cdk/aws-rds");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
/**
 * Use `ServerlessApi` to create the serverless API resource
 */
class ServerlessApi extends cdk.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k;
        super(scope, id);
        const DEFAULT_LAMBDA_ASSET_PATH = path.join(__dirname, '../composer/laravel58-bref');
        const DEFAULT_DB_MASTER_USER = 'admin';
        this.vpc = props.vpc;
        this.handler = (_a = props.handler) !== null && _a !== void 0 ? _a : new lambda.Function(this, 'handler', {
            runtime: lambda.Runtime.PROVIDED,
            handler: 'public/index.php',
            layers: [
                lambda.LayerVersion.fromLayerVersionArn(this, 'BrefPHPLayer', props.brefLayerVersion),
            ],
            code: lambda.Code.fromAsset((_b = props === null || props === void 0 ? void 0 : props.lambdaCodePath) !== null && _b !== void 0 ? _b : DEFAULT_LAMBDA_ASSET_PATH),
            environment: {
                APP_STORAGE: '/tmp',
                DB_WRITER: (_d = (_c = props.databaseConfig) === null || _c === void 0 ? void 0 : _c.writerEndpoint) !== null && _d !== void 0 ? _d : '',
                DB_READER: (_h = (_f = (_e = props.databaseConfig) === null || _e === void 0 ? void 0 : _e.readerEndpoint) !== null && _f !== void 0 ? _f : (_g = props.databaseConfig) === null || _g === void 0 ? void 0 : _g.writerEndpoint) !== null && _h !== void 0 ? _h : '',
                DB_USER: (_k = (_j = props.databaseConfig) === null || _j === void 0 ? void 0 : _j.masterUserName) !== null && _k !== void 0 ? _k : DEFAULT_DB_MASTER_USER,
            },
            timeout: cdk.Duration.seconds(120),
            vpc: props.vpc,
        });
        // allow lambda execution role to connect to RDS proxy
        if (props.rdsProxy) {
            this.handler.addToRolePolicy(new iam.PolicyStatement({
                actions: ['rds-db:connect'],
                resources: [props.rdsProxy.dbProxyArn],
            }));
        }
        const endpoint = new apigateway.HttpApi(this, 'apiservice', {
            defaultIntegration: new apigateway.LambdaProxyIntegration({
                handler: this.handler,
            }),
        });
        new cdk.CfnOutput(this, 'EndpointURL', { value: endpoint.url });
    }
}
exports.ServerlessApi = ServerlessApi;
/**
 * Use `ServerlessLaravel` to create the serverless Laravel resource
 */
class ServerlessLaravel extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        new ServerlessApi(this, id, {
            lambdaCodePath: props.laravelPath,
            brefLayerVersion: props.brefLayerVersion,
            handler: props.handler,
            vpc: props.vpc,
            databaseConfig: props.databaseConfig,
            rdsProxy: props.rdsProxy,
        });
    }
}
exports.ServerlessLaravel = ServerlessLaravel;
class DatabaseCluster extends cdk.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id);
        this.masterUser = (_a = props.masterUserName) !== null && _a !== void 0 ? _a : 'admin';
        // generate and store password for masterUser in the secrets manager
        const masterUserSecret = new secretsmanager.Secret(this, 'DbMasterSecret', {
            secretName: `${cdk.Stack.of(this).stackName}-DbMasterSecret`,
            generateSecretString: {
                secretStringTemplate: JSON.stringify({
                    username: this.masterUser,
                }),
                passwordLength: 12,
                excludePunctuation: true,
                includeSpace: false,
                generateStringKey: 'password',
            },
        });
        this.masterPassword = masterUserSecret;
        const dbConnectionGroup = new aws_ec2_1.SecurityGroup(this, 'DB Secuirty Group', {
            vpc: props.vpc,
        });
        dbConnectionGroup.connections.allowInternally(aws_ec2_1.Port.tcp(3306));
        const dbCluster = new rds.DatabaseCluster(this, 'DBCluster', {
            engine: rds.DatabaseClusterEngine.auroraMysql({
                version: rds.AuroraMysqlEngineVersion.VER_2_08_1,
            }),
            instanceProps: {
                vpc: props.vpc,
                instanceType: (_b = props.instanceType) !== null && _b !== void 0 ? _b : new aws_ec2_1.InstanceType('t3.medium'),
                securityGroups: [dbConnectionGroup],
            },
            masterUser: {
                username: masterUserSecret.secretValueFromJson('username').toString(),
                password: masterUserSecret.secretValueFromJson('password'),
            },
            instances: props.instanceCapacity,
            removalPolicy: core_1.RemovalPolicy.DESTROY,
        });
        // Workaround for bug where TargetGroupName is not set but required
        let cfnDbInstance = dbCluster.node.children.find((child) => {
            return child instanceof rds.CfnDBInstance;
        });
        // enable the RDS proxy by default
        if (props.rdsProxy !== false) {
            // create iam role for RDS proxy
            const rdsProxyRole = new iam.Role(this, 'RdsProxyRole', {
                assumedBy: new iam.ServicePrincipal('rds.amazonaws.com'),
            });
            // see: https://aws.amazon.com/tw/blogs/compute/using-amazon-rds-proxy-with-aws-lambda/
            rdsProxyRole.addToPolicy(new iam.PolicyStatement({
                actions: [
                    'secretsmanager:GetResourcePolicy',
                    'secretsmanager:GetSecretValue',
                    'secretsmanager:DescribeSecret',
                    'secretsmanager:ListSecretVersionIds',
                ],
                resources: [masterUserSecret.secretArn],
            }));
            const proxyOptions = {
                vpc: props.vpc,
                secrets: [masterUserSecret],
                iamAuth: true,
                dbProxyName: `${cdk.Stack.of(this).stackName}-RDSProxy`,
                securityGroups: [dbConnectionGroup],
                role: rdsProxyRole,
            };
            // create the RDS proxy
            this.rdsProxy = dbCluster.addProxy('RDSProxy', proxyOptions);
            // ensure DB instance is ready before creating the proxy
            (_c = this.rdsProxy) === null || _c === void 0 ? void 0 : _c.node.addDependency(cfnDbInstance);
        }
    }
}
exports.DatabaseCluster = DatabaseCluster;
//# sourceMappingURL=data:application/json;base64,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