# -*- coding: utf-8 -*-

# PLEASE DO NOT EDIT THIS FILE, IT IS GENERATED AND WILL BE OVERWRITTEN:
# https://github.com/ccxt/ccxt/blob/master/CONTRIBUTING.md#how-to-contribute-code

from ccxt.async_support.base.exchange import Exchange
import math
from ccxt.base.errors import ExchangeError
from ccxt.base.errors import AuthenticationError
from ccxt.base.errors import PermissionDenied
from ccxt.base.errors import BadSymbol
from ccxt.base.errors import InsufficientFunds
from ccxt.base.errors import InvalidOrder
from ccxt.base.errors import OrderNotFound
from ccxt.base.errors import ExchangeNotAvailable
from ccxt.base.errors import RequestTimeout
from ccxt.base.decimal_to_precision import TRUNCATE
from ccxt.base.decimal_to_precision import DECIMAL_PLACES
from ccxt.base.decimal_to_precision import TICK_SIZE
from ccxt.base.precise import Precise


class hitbtc(Exchange):

    def describe(self):
        return self.deep_extend(super(hitbtc, self).describe(), {
            'id': 'hitbtc',
            'name': 'HitBTC',
            'countries': ['HK'],
            'rateLimit': 1500,
            'version': '2',
            'pro': True,
            'has': {
                'cancelOrder': True,
                'CORS': False,
                'createDepositAddress': True,
                'createOrder': True,
                'editOrder': True,
                'fetchBalance': True,
                'fetchClosedOrders': True,
                'fetchCurrencies': True,
                'fetchDepositAddress': True,
                'fetchDeposits': False,
                'fetchMarkets': True,
                'fetchMyTrades': True,
                'fetchOHLCV': True,
                'fetchOpenOrder': True,
                'fetchOpenOrders': True,
                'fetchOrder': True,
                'fetchOrderBook': True,
                'fetchOrders': False,
                'fetchOrderTrades': True,
                'fetchTicker': True,
                'fetchTickers': True,
                'fetchTrades': True,
                'fetchTradingFee': True,
                'fetchTransactions': True,
                'fetchWithdrawals': False,
                'withdraw': True,
                'transfer': True,
            },
            'timeframes': {
                '1m': 'M1',
                '3m': 'M3',
                '5m': 'M5',
                '15m': 'M15',
                '30m': 'M30',  # default
                '1h': 'H1',
                '4h': 'H4',
                '1d': 'D1',
                '1w': 'D7',
                '1M': '1M',
            },
            'urls': {
                'logo': 'https://user-images.githubusercontent.com/1294454/27766555-8eaec20e-5edc-11e7-9c5b-6dc69fc42f5e.jpg',
                'test': {
                    'public': 'https://api.demo.hitbtc.com',
                    'private': 'https://api.demo.hitbtc.com',
                },
                'api': {
                    'public': 'https://api.hitbtc.com',
                    'private': 'https://api.hitbtc.com',
                },
                'www': 'https://hitbtc.com',
                'referral': 'https://hitbtc.com/?ref_id=5a5d39a65d466',
                'doc': [
                    'https://api.hitbtc.com',
                    'https://github.com/hitbtc-com/hitbtc-api/blob/master/APIv2.md',
                ],
                'fees': [
                    'https://hitbtc.com/fees-and-limits',
                    'https://support.hitbtc.com/hc/en-us/articles/115005148605-Fees-and-limits',
                ],
            },
            'api': {
                'public': {
                    'get': [
                        'currency',  # Available Currencies
                        'currency/{currency}',  # Get currency info
                        'symbol',  # Available Currency Symbols
                        'symbol/{symbol}',  # Get symbol info
                        'ticker',  # Ticker list for all symbols
                        'ticker/{symbol}',  # Ticker for symbol
                        'trades',
                        'trades/{symbol}',  # Trades
                        'orderbook',
                        'orderbook/{symbol}',  # Orderbook
                        'candles',
                        'candles/{symbol}',  # Candles
                    ],
                },
                'private': {
                    'get': [
                        'trading/balance',  # Get trading balance
                        'order',  # List your current open orders
                        'order/{clientOrderId}',  # Get a single order by clientOrderId
                        'trading/fee/all',  # Get trading fee rate
                        'trading/fee/{symbol}',  # Get trading fee rate
                        'margin/account',
                        'margin/account/{symbol}',
                        'margin/position',
                        'margin/position/{symbol}',
                        'margin/order',
                        'margin/order/{clientOrderId}',
                        'history/order',  # Get historical orders
                        'history/trades',  # Get historical trades
                        'history/order/{orderId}/trades',  # Get historical trades by specified order
                        'account/balance',  # Get main acccount balance
                        'account/crypto/address/{currency}',  # Get current address
                        'account/crypto/addresses/{currency}',  # Get last 10 deposit addresses for currency
                        'account/crypto/used-addresses/{currency}',  # Get last 10 unique addresses used for withdraw by currency
                        'account/crypto/estimate-withdraw',
                        'account/crypto/is-mine/{address}',
                        'account/transactions',  # Get account transactions
                        'account/transactions/{id}',  # Get account transaction by id
                        'sub-acc',
                        'sub-acc/acl',
                        'sub-acc/balance/{subAccountUserID}',
                        'sub-acc/deposit-address/{subAccountUserId}/{currency}',
                    ],
                    'post': [
                        'order',  # Create new order
                        'margin/order',
                        'account/crypto/address/{currency}',  # Create new crypto deposit address
                        'account/crypto/withdraw',  # Withdraw crypto
                        'account/crypto/transfer-convert',
                        'account/transfer',  # Transfer amount to trading account or to main account
                        'account/transfer/internal',
                        'sub-acc/freeze',
                        'sub-acc/activate',
                        'sub-acc/transfer',
                    ],
                    'put': [
                        'order/{clientOrderId}',  # Create new order
                        'margin/account/{symbol}',
                        'margin/order/{clientOrderId}',
                        'account/crypto/withdraw/{id}',  # Commit crypto withdrawal
                        'sub-acc/acl/{subAccountUserId}',
                    ],
                    'delete': [
                        'order',  # Cancel all open orders
                        'order/{clientOrderId}',  # Cancel order
                        'margin/account',
                        'margin/account/{symbol}',
                        'margin/position',
                        'margin/position/{symbol}',
                        'margin/order',
                        'margin/order/{clientOrderId}',
                        'account/crypto/withdraw/{id}',  # Rollback crypto withdrawal
                    ],
                    # outdated?
                    'patch': [
                        'order/{clientOrderId}',  # Cancel Replace order
                    ],
                },
            },
            'precisionMode': TICK_SIZE,
            'fees': {
                'trading': {
                    'tierBased': False,
                    'percentage': True,
                    'maker': 0.1 / 100,
                    'taker': 0.2 / 100,
                },
            },
            'options': {
                'networks': {
                    'ETH': 'T20',
                    'ERC20': 'T20',
                    'TRX': 'TTRX',
                    'TRC20': 'TTRX',
                    'OMNI': '',
                },
                'defaultTimeInForce': 'FOK',
                'accountsByType': {
                    'bank': 'bank',
                    'exchange': 'exchange',
                    'main': 'bank',  # alias of the above
                    'funding': 'bank',
                    'spot': 'exchange',
                    'trade': 'exchange',
                    'trading': 'exchange',
                },
                'fetchBalanceMethod': {
                    'account': 'account',
                    'bank': 'account',
                    'main': 'account',
                    'funding': 'account',
                    'exchange': 'trading',
                    'spot': 'trading',
                    'trade': 'trading',
                    'trading': 'trading',
                },
            },
            'commonCurrencies': {
                'AUTO': 'Cube',
                'BCC': 'BCC',  # initial symbol for Bitcoin Cash, now inactive
                'BDP': 'BidiPass',
                'BET': 'DAO.Casino',
                'BIT': 'BitRewards',
                'BOX': 'BOX Token',
                'CPT': 'Cryptaur',  # conflict with CPT = Contents Protocol https://github.com/ccxt/ccxt/issues/4920 and https://github.com/ccxt/ccxt/issues/6081
                'GET': 'Themis',
                'HSR': 'HC',
                'IQ': 'IQ.Cash',
                'LNC': 'LinkerCoin',
                'PLA': 'PlayChip',
                'PNT': 'Penta',
                'SBTC': 'Super Bitcoin',
                'STX': 'Stox',
                'TV': 'Tokenville',
                'USD': 'USDT',
                'XPNT': 'PNT',
            },
            'exceptions': {
                '504': RequestTimeout,  # {"error":{"code":504,"message":"Gateway Timeout"}}
                '1002': AuthenticationError,  # {"error":{"code":1002,"message":"Authorization failed","description":""}}
                '1003': PermissionDenied,  # "Action is forbidden for self API key"
                '2010': InvalidOrder,  # "Quantity not a valid number"
                '2001': BadSymbol,  # "Symbol not found"
                '2011': InvalidOrder,  # "Quantity too low"
                '2020': InvalidOrder,  # "Price not a valid number"
                '20002': OrderNotFound,  # canceling non-existent order
                '20001': InsufficientFunds,  # {"error":{"code":20001,"message":"Insufficient funds","description":"Check that the funds are sufficient, given commissions"}}
            },
        })

    def fee_to_precision(self, symbol, fee):
        return self.decimal_to_precision(fee, TRUNCATE, 8, DECIMAL_PLACES)

    async def fetch_markets(self, params={}):
        response = await self.publicGetSymbol(params)
        #
        #     [
        #         {
        #             "id":"BCNBTC",
        #             "baseCurrency":"BCN",
        #             "quoteCurrency":"BTC",
        #             "quantityIncrement":"100",
        #             "tickSize":"0.00000000001",
        #             "takeLiquidityRate":"0.002",
        #             "provideLiquidityRate":"0.001",
        #             "feeCurrency":"BTC"
        #         }
        #     ]
        #
        result = []
        for i in range(0, len(response)):
            market = response[i]
            id = self.safe_string(market, 'id')
            baseId = self.safe_string(market, 'baseCurrency')
            quoteId = self.safe_string(market, 'quoteCurrency')
            base = self.safe_currency_code(baseId)
            quote = self.safe_currency_code(quoteId)
            symbol = base + '/' + quote
            # bequant fix
            if id.find('_') >= 0:
                symbol = id
            lotString = self.safe_string(market, 'quantityIncrement')
            stepString = self.safe_string(market, 'tickSize')
            lot = self.parse_number(lotString)
            step = self.parse_number(stepString)
            precision = {
                'price': step,
                'amount': lot,
            }
            taker = self.safe_number(market, 'takeLiquidityRate')
            maker = self.safe_number(market, 'provideLiquidityRate')
            feeCurrencyId = self.safe_string(market, 'feeCurrency')
            feeCurrencyCode = self.safe_currency_code(feeCurrencyId)
            result.append(self.extend(self.fees['trading'], {
                'info': market,
                'id': id,
                'symbol': symbol,
                'base': base,
                'quote': quote,
                'baseId': baseId,
                'quoteId': quoteId,
                'active': True,
                'taker': taker,
                'maker': maker,
                'precision': precision,
                'feeCurrency': feeCurrencyCode,
                'limits': {
                    'amount': {
                        'min': lot,
                        'max': None,
                    },
                    'price': {
                        'min': step,
                        'max': None,
                    },
                    'cost': {
                        'min': self.parse_number(Precise.string_mul(lotString, stepString)),
                        'max': None,
                    },
                },
            }))
        return result

    async def transfer(self, code, amount, fromAccount, toAccount, params={}):
        # account can be "exchange" or "bank", with aliases "main" or "trading" respectively
        await self.load_markets()
        currency = self.currency(code)
        requestAmount = self.currency_to_precision(code, amount)
        request = {
            'currency': currency['id'],
            'amount': requestAmount,
        }
        type = self.safe_string(params, 'type')
        if type is None:
            accountsByType = self.safe_value(self.options, 'accountsByType', {})
            fromId = self.safe_string(accountsByType, fromAccount)
            toId = self.safe_string(accountsByType, toAccount)
            keys = list(accountsByType.keys())
            if fromId is None:
                raise ExchangeError(self.id + ' fromAccount must be one of ' + ', '.join(keys) + ' instead of ' + fromId)
            if toId is None:
                raise ExchangeError(self.id + ' toAccount must be one of ' + ', '.join(keys) + ' instead of ' + toId)
            if fromId == toId:
                raise ExchangeError(self.id + ' from and to cannot be the same account')
            type = fromId + 'To' + self.capitalize(toId)
        request['type'] = type
        response = await self.privatePostAccountTransfer(self.extend(request, params))
        # {id: '2db6ebab-fb26-4537-9ef8-1a689472d236'}
        id = self.safe_string(response, 'id')
        return {
            'info': response,
            'id': id,
            'timestamp': None,
            'datetime': None,
            'amount': requestAmount,
            'currency': code,
            'fromAccount': fromAccount,
            'toAccount': toAccount,
            'status': None,
        }

    async def fetch_currencies(self, params={}):
        response = await self.publicGetCurrency(params)
        #
        #     [
        #         {
        #             "id":"XPNT",
        #             "fullName":"pToken",
        #             "crypto":true,
        #             "payinEnabled":true,
        #             "payinPaymentId":false,
        #             "payinConfirmations":9,
        #             "payoutEnabled":true,
        #             "payoutIsPaymentId":false,
        #             "transferEnabled":true,
        #             "delisted":false,
        #             "payoutFee":"26.510000000000",
        #             "precisionPayout":18,
        #             "precisionTransfer":8
        #         }
        #     ]
        #
        result = {}
        for i in range(0, len(response)):
            currency = response[i]
            id = self.safe_string(currency, 'id')
            # todo: will need to rethink the fees
            # to add support for multiple withdrawal/deposit methods and
            # differentiated fees for each particular method
            decimals = self.safe_integer(currency, 'precisionTransfer', 8)
            precision = 1 / math.pow(10, decimals)
            code = self.safe_currency_code(id)
            payin = self.safe_value(currency, 'payinEnabled')
            payout = self.safe_value(currency, 'payoutEnabled')
            transfer = self.safe_value(currency, 'transferEnabled')
            active = payin and payout and transfer
            if 'disabled' in currency:
                if currency['disabled']:
                    active = False
            type = 'fiat'
            if ('crypto' in currency) and currency['crypto']:
                type = 'crypto'
            name = self.safe_string(currency, 'fullName')
            result[code] = {
                'id': id,
                'code': code,
                'type': type,
                'payin': payin,
                'payout': payout,
                'transfer': transfer,
                'info': currency,
                'name': name,
                'active': active,
                'fee': self.safe_number(currency, 'payoutFee'),  # todo: redesign
                'precision': precision,
                'limits': {
                    'amount': {
                        'min': 1 / math.pow(10, decimals),
                        'max': math.pow(10, decimals),
                    },
                    'withdraw': {
                        'min': None,
                        'max': math.pow(10, precision),
                    },
                },
            }
        return result

    async def fetch_trading_fee(self, symbol, params={}):
        await self.load_markets()
        market = self.market(symbol)
        request = self.extend({
            'symbol': market['id'],
        }, self.omit(params, 'symbol'))
        response = await self.privateGetTradingFeeSymbol(request)
        #
        #     {
        #         takeLiquidityRate: '0.001',
        #         provideLiquidityRate: '-0.0001'
        #     }
        #
        return {
            'info': response,
            'maker': self.safe_number(response, 'provideLiquidityRate'),
            'taker': self.safe_number(response, 'takeLiquidityRate'),
        }

    async def fetch_balance(self, params={}):
        await self.load_markets()
        type = self.safe_string(params, 'type', 'trading')
        fetchBalanceAccounts = self.safe_value(self.options, 'fetchBalanceMethod', {})
        typeId = self.safe_string(fetchBalanceAccounts, type)
        if typeId is None:
            raise ExchangeError(self.id + ' fetchBalance account type must be either main or trading')
        method = 'privateGet' + self.capitalize(typeId) + 'Balance'
        query = self.omit(params, 'type')
        response = await getattr(self, method)(query)
        #
        #     [
        #         {"currency":"SPI","available":"0","reserved":"0"},
        #         {"currency":"GRPH","available":"0","reserved":"0"},
        #         {"currency":"DGTX","available":"0","reserved":"0"},
        #     ]
        #
        result = {
            'info': response,
            'timestamp': None,
            'datetime': None,
        }
        for i in range(0, len(response)):
            balance = response[i]
            currencyId = self.safe_string(balance, 'currency')
            code = self.safe_currency_code(currencyId)
            account = self.account()
            account['free'] = self.safe_string(balance, 'available')
            account['used'] = self.safe_string(balance, 'reserved')
            result[code] = account
        return self.parse_balance(result)

    def parse_ohlcv(self, ohlcv, market=None):
        #
        #     {
        #         "timestamp":"2015-08-20T19:01:00.000Z",
        #         "open":"0.006",
        #         "close":"0.006",
        #         "min":"0.006",
        #         "max":"0.006",
        #         "volume":"0.003",
        #         "volumeQuote":"0.000018"
        #     }
        #
        return [
            self.parse8601(self.safe_string(ohlcv, 'timestamp')),
            self.safe_number(ohlcv, 'open'),
            self.safe_number(ohlcv, 'max'),
            self.safe_number(ohlcv, 'min'),
            self.safe_number(ohlcv, 'close'),
            self.safe_number(ohlcv, 'volume'),
        ]

    async def fetch_ohlcv(self, symbol, timeframe='1m', since=None, limit=None, params={}):
        await self.load_markets()
        market = self.market(symbol)
        request = {
            'symbol': market['id'],
            'period': self.timeframes[timeframe],
        }
        if since is not None:
            request['from'] = self.iso8601(since)
        if limit is not None:
            request['limit'] = limit
        response = await self.publicGetCandlesSymbol(self.extend(request, params))
        #
        #     [
        #         {"timestamp":"2015-08-20T19:01:00.000Z","open":"0.006","close":"0.006","min":"0.006","max":"0.006","volume":"0.003","volumeQuote":"0.000018"},
        #         {"timestamp":"2015-08-20T19:03:00.000Z","open":"0.006","close":"0.006","min":"0.006","max":"0.006","volume":"0.013","volumeQuote":"0.000078"},
        #         {"timestamp":"2015-08-20T19:06:00.000Z","open":"0.0055","close":"0.005","min":"0.005","max":"0.0055","volume":"0.003","volumeQuote":"0.0000155"},
        #     ]
        #
        return self.parse_ohlcvs(response, market, timeframe, since, limit)

    async def fetch_order_book(self, symbol, limit=None, params={}):
        await self.load_markets()
        request = {
            'symbol': self.market_id(symbol),
        }
        if limit is not None:
            request['limit'] = limit  # default = 100, 0 = unlimited
        response = await self.publicGetOrderbookSymbol(self.extend(request, params))
        return self.parse_order_book(response, symbol, None, 'bid', 'ask', 'price', 'size')

    def parse_ticker(self, ticker, market=None):
        timestamp = self.parse8601(ticker['timestamp'])
        symbol = self.safe_symbol(None, market)
        baseVolume = self.safe_number(ticker, 'volume')
        quoteVolume = self.safe_number(ticker, 'volumeQuote')
        open = self.safe_number(ticker, 'open')
        last = self.safe_number(ticker, 'last')
        vwap = self.vwap(baseVolume, quoteVolume)
        return self.safe_ticker({
            'symbol': symbol,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'high': self.safe_number(ticker, 'high'),
            'low': self.safe_number(ticker, 'low'),
            'bid': self.safe_number(ticker, 'bid'),
            'bidVolume': None,
            'ask': self.safe_number(ticker, 'ask'),
            'askVolume': None,
            'vwap': vwap,
            'open': open,
            'close': last,
            'last': last,
            'previousClose': None,
            'change': None,
            'percentage': None,
            'average': None,
            'baseVolume': baseVolume,
            'quoteVolume': quoteVolume,
            'info': ticker,
        }, market)

    async def fetch_tickers(self, symbols=None, params={}):
        await self.load_markets()
        response = await self.publicGetTicker(params)
        result = {}
        for i in range(0, len(response)):
            ticker = response[i]
            marketId = self.safe_string(ticker, 'symbol')
            market = self.safe_market(marketId)
            symbol = market['symbol']
            result[symbol] = self.parse_ticker(ticker, market)
        return self.filter_by_array(result, 'symbol', symbols)

    async def fetch_ticker(self, symbol, params={}):
        await self.load_markets()
        market = self.market(symbol)
        request = {
            'symbol': market['id'],
        }
        response = await self.publicGetTickerSymbol(self.extend(request, params))
        if 'message' in response:
            raise ExchangeError(self.id + ' ' + response['message'])
        return self.parse_ticker(response, market)

    def parse_trade(self, trade, market=None):
        # createMarketOrder
        #
        #  {      fee: "0.0004644",
        #           id:  386394956,
        #        price: "0.4644",
        #     quantity: "1",
        #    timestamp: "2018-10-25T16:41:44.780Z"}
        #
        # fetchTrades
        #
        # {id: 974786185,
        #   price: '0.032462',
        #   quantity: '0.3673',
        #   side: 'buy',
        #   timestamp: '2020-10-16T12:57:39.846Z'}
        #
        # fetchMyTrades
        #
        # {id: 277210397,
        #   clientOrderId: '6e102f3e7f3f4e04aeeb1cdc95592f1a',
        #   orderId: 28102855393,
        #   symbol: 'ETHBTC',
        #   side: 'sell',
        #   quantity: '0.002',
        #   price: '0.073365',
        #   fee: '0.000000147',
        #   timestamp: '2018-04-28T18:39:55.345Z'}
        timestamp = self.parse8601(trade['timestamp'])
        marketId = self.safe_string(trade, 'symbol')
        market = self.safe_market(marketId, market)
        symbol = market['symbol']
        fee = None
        feeCost = self.safe_number(trade, 'fee')
        if feeCost is not None:
            feeCurrencyCode = market['feeCurrency'] if market else None
            fee = {
                'cost': feeCost,
                'currency': feeCurrencyCode,
            }
        # we use clientOrderId as the order id with self exchange intentionally
        # because most of their endpoints will require clientOrderId
        # explained here: https://github.com/ccxt/ccxt/issues/5674
        orderId = self.safe_string(trade, 'clientOrderId')
        priceString = self.safe_string(trade, 'price')
        amountString = self.safe_string(trade, 'quantity')
        price = self.parse_number(priceString)
        amount = self.parse_number(amountString)
        cost = self.parse_number(Precise.string_mul(priceString, amountString))
        side = self.safe_string(trade, 'side')
        id = self.safe_string(trade, 'id')
        return {
            'info': trade,
            'id': id,
            'order': orderId,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'symbol': symbol,
            'type': None,
            'side': side,
            'takerOrMaker': None,
            'price': price,
            'amount': amount,
            'cost': cost,
            'fee': fee,
        }

    async def fetch_transactions(self, code=None, since=None, limit=None, params={}):
        await self.load_markets()
        currency = None
        request = {}
        if code is not None:
            currency = self.currency(code)
            request['asset'] = currency['id']
        if since is not None:
            request['startTime'] = since
        response = await self.privateGetAccountTransactions(self.extend(request, params))
        return self.parse_transactions(response, currency, since, limit)

    def parse_transaction(self, transaction, currency=None):
        #
        #     {
        #         id: 'd53ee9df-89bf-4d09-886e-849f8be64647',
        #         index: 1044718371,
        #         type: 'payout',
        #         status: 'success',
        #         currency: 'ETH',
        #         amount: '4.522683200000000000000000',
        #         createdAt: '2018-06-07T00:43:32.426Z',
        #         updatedAt: '2018-06-07T00:45:36.447Z',
        #         hash: '0x973e5683dfdf80a1fb1e0b96e19085b6489221d2ddf864daa46903c5ec283a0f',
        #         address: '0xC5a59b21948C1d230c8C54f05590000Eb3e1252c',
        #         fee: '0.00958',
        #     },
        #     {
        #         id: 'e6c63331-467e-4922-9edc-019e75d20ba3',
        #         index: 1044714672,
        #         type: 'exchangeToBank',
        #         status: 'success',
        #         currency: 'ETH',
        #         amount: '4.532263200000000000',
        #         createdAt: '2018-06-07T00:42:39.543Z',
        #         updatedAt: '2018-06-07T00:42:39.683Z',
        #     },
        #     {
        #         id: '3b052faa-bf97-4636-a95c-3b5260015a10',
        #         index: 1009280164,
        #         type: 'bankToExchange',
        #         status: 'success',
        #         currency: 'CAS',
        #         amount: '104797.875800000000000000',
        #         createdAt: '2018-05-19T02:34:36.750Z',
        #         updatedAt: '2018-05-19T02:34:36.857Z',
        #     },
        #     {
        #         id: 'd525249f-7498-4c81-ba7b-b6ae2037dc08',
        #         index: 1009279948,
        #         type: 'payin',
        #         status: 'success',
        #         currency: 'CAS',
        #         amount: '104797.875800000000000000',
        #         createdAt: '2018-05-19T02:30:16.698Z',
        #         updatedAt: '2018-05-19T02:34:28.159Z',
        #         hash: '0xa6530e1231de409cf1f282196ed66533b103eac1df2aa4a7739d56b02c5f0388',
        #         address: '0xd53ed559a6d963af7cb3f3fcd0e7ca499054db8b',
        #     }
        #
        #     {
        #         "id": "4f351f4f-a8ee-4984-a468-189ed590ddbd",
        #         "index": 3112719565,
        #         "type": "withdraw",
        #         "status": "success",
        #         "currency": "BCHOLD",
        #         "amount": "0.02423133",
        #         "createdAt": "2019-07-16T16:52:04.494Z",
        #         "updatedAt": "2019-07-16T16:54:07.753Z"
        #     }
        id = self.safe_string(transaction, 'id')
        timestamp = self.parse8601(self.safe_string(transaction, 'createdAt'))
        updated = self.parse8601(self.safe_string(transaction, 'updatedAt'))
        currencyId = self.safe_string(transaction, 'currency')
        code = self.safe_currency_code(currencyId, currency)
        status = self.parse_transaction_status(self.safe_string(transaction, 'status'))
        amount = self.safe_number(transaction, 'amount')
        address = self.safe_string(transaction, 'address')
        txid = self.safe_string(transaction, 'hash')
        fee = None
        feeCost = self.safe_number(transaction, 'fee')
        if feeCost is not None:
            fee = {
                'cost': feeCost,
                'currency': code,
            }
        type = self.parse_transaction_type(self.safe_string(transaction, 'type'))
        return {
            'info': transaction,
            'id': id,
            'txid': txid,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'address': address,
            'tag': None,
            'type': type,
            'amount': amount,
            'currency': code,
            'status': status,
            'updated': updated,
            'fee': fee,
        }

    def parse_transaction_status(self, status):
        statuses = {
            'pending': 'pending',
            'failed': 'failed',
            'success': 'ok',
        }
        return self.safe_string(statuses, status, status)

    def parse_transaction_type(self, type):
        types = {
            'payin': 'deposit',
            'payout': 'withdrawal',
            'withdraw': 'withdrawal',
        }
        return self.safe_string(types, type, type)

    async def fetch_trades(self, symbol, since=None, limit=None, params={}):
        await self.load_markets()
        market = self.market(symbol)
        request = {
            'symbol': market['id'],
        }
        if limit is not None:
            request['limit'] = limit
        if since is not None:
            request['sort'] = 'ASC'
            request['from'] = self.iso8601(since)
        response = await self.publicGetTradesSymbol(self.extend(request, params))
        return self.parse_trades(response, market, since, limit)

    async def create_order(self, symbol, type, side, amount, price=None, params={}):
        await self.load_markets()
        market = self.market(symbol)
        # we use clientOrderId as the order id with self exchange intentionally
        # because most of their endpoints will require clientOrderId
        # explained here: https://github.com/ccxt/ccxt/issues/5674
        # their max accepted length is 32 characters
        uuid = self.uuid()
        parts = uuid.split('-')
        clientOrderId = ''.join(parts)
        clientOrderId = clientOrderId[0:32]
        amount = float(amount)
        request = {
            'clientOrderId': clientOrderId,
            'symbol': market['id'],
            'side': side,
            'quantity': self.amount_to_precision(symbol, amount),
            'type': type,
        }
        if type == 'limit':
            request['price'] = self.price_to_precision(symbol, price)
        else:
            request['timeInForce'] = self.options['defaultTimeInForce']
        response = await self.privatePostOrder(self.extend(request, params))
        order = self.parse_order(response)
        if order['status'] == 'rejected':
            raise InvalidOrder(self.id + ' order was rejected by the exchange ' + self.json(order))
        return order

    async def edit_order(self, id, symbol, type, side, amount=None, price=None, params={}):
        await self.load_markets()
        # we use clientOrderId as the order id with self exchange intentionally
        # because most of their endpoints will require clientOrderId
        # explained here: https://github.com/ccxt/ccxt/issues/5674
        # their max accepted length is 32 characters
        uuid = self.uuid()
        parts = uuid.split('-')
        requestClientId = ''.join(parts)
        requestClientId = requestClientId[0:32]
        request = {
            'clientOrderId': id,
            'requestClientId': requestClientId,
        }
        if amount is not None:
            request['quantity'] = self.amount_to_precision(symbol, amount)
        if price is not None:
            request['price'] = self.price_to_precision(symbol, price)
        response = await self.privatePatchOrderClientOrderId(self.extend(request, params))
        return self.parse_order(response)

    async def cancel_order(self, id, symbol=None, params={}):
        await self.load_markets()
        # we use clientOrderId as the order id with self exchange intentionally
        # because most of their endpoints will require clientOrderId
        # explained here: https://github.com/ccxt/ccxt/issues/5674
        request = {
            'clientOrderId': id,
        }
        response = await self.privateDeleteOrderClientOrderId(self.extend(request, params))
        return self.parse_order(response)

    def parse_order_status(self, status):
        statuses = {
            'new': 'open',
            'suspended': 'open',
            'partiallyFilled': 'open',
            'filled': 'closed',
            'canceled': 'canceled',
            'expired': 'failed',
        }
        return self.safe_string(statuses, status, status)

    def parse_order(self, order, market=None):
        #
        # createMarketOrder
        #
        #     {
        #         clientOrderId: "fe36aa5e190149bf9985fb673bbb2ea0",
        #         createdAt: "2018-10-25T16:41:44.780Z",
        #         cumQuantity: "1",
        #         id: "66799540063",
        #         quantity: "1",
        #         side: "sell",
        #         status: "filled",
        #         symbol: "XRPUSDT",
        #         timeInForce: "FOK",
        #         tradesReport: [
        #             {
        #                 fee: "0.0004644",
        #                 id:  386394956,
        #                 price: "0.4644",
        #                 quantity: "1",
        #                 timestamp: "2018-10-25T16:41:44.780Z"
        #             }
        #         ],
        #         type: "market",
        #         updatedAt: "2018-10-25T16:41:44.780Z"
        #     }
        #
        #     {
        #         "id": 119499457455,
        #         "clientOrderId": "87baab109d58401b9202fa0749cb8288",
        #         "symbol": "ETHUSD",
        #         "side": "buy",
        #         "status": "filled",
        #         "type": "market",
        #         "timeInForce": "FOK",
        #         "quantity": "0.0007",
        #         "price": "181.487",
        #         "avgPrice": "164.989",
        #         "cumQuantity": "0.0007",
        #         "createdAt": "2019-04-17T13:27:38.062Z",
        #         "updatedAt": "2019-04-17T13:27:38.062Z"
        #     }
        #
        created = self.parse8601(self.safe_string(order, 'createdAt'))
        updated = self.parse8601(self.safe_string(order, 'updatedAt'))
        marketId = self.safe_string(order, 'symbol')
        market = self.safe_market(marketId, market)
        symbol = market['symbol']
        amount = self.safe_number(order, 'quantity')
        filled = self.safe_number(order, 'cumQuantity')
        status = self.parse_order_status(self.safe_string(order, 'status'))
        # we use clientOrderId as the order id with self exchange intentionally
        # because most of their endpoints will require clientOrderId
        # explained here: https://github.com/ccxt/ccxt/issues/5674
        id = self.safe_string(order, 'clientOrderId')
        clientOrderId = id
        price = self.safe_number(order, 'price')
        type = self.safe_string(order, 'type')
        side = self.safe_string(order, 'side')
        trades = self.safe_value(order, 'tradesReport')
        fee = None
        average = self.safe_number(order, 'avgPrice')
        if trades is not None:
            trades = self.parse_trades(trades, market)
        timeInForce = self.safe_string(order, 'timeInForce')
        return self.safe_order({
            'id': id,
            'clientOrderId': clientOrderId,  # https://github.com/ccxt/ccxt/issues/5674
            'timestamp': created,
            'datetime': self.iso8601(created),
            'lastTradeTimestamp': updated,
            'status': status,
            'symbol': symbol,
            'type': type,
            'timeInForce': timeInForce,
            'side': side,
            'price': price,
            'stopPrice': None,
            'average': average,
            'amount': amount,
            'cost': None,
            'filled': filled,
            'remaining': None,
            'fee': fee,
            'trades': trades,
            'info': order,
        })

    async def fetch_order(self, id, symbol=None, params={}):
        await self.load_markets()
        # we use clientOrderId as the order id with self exchange intentionally
        # because most of their endpoints will require clientOrderId
        # explained here: https://github.com/ccxt/ccxt/issues/5674
        request = {
            'clientOrderId': id,
        }
        response = await self.privateGetHistoryOrder(self.extend(request, params))
        numOrders = len(response)
        if numOrders > 0:
            return self.parse_order(response[0])
        raise OrderNotFound(self.id + ' order ' + id + ' not found')

    async def fetch_open_order(self, id, symbol=None, params={}):
        await self.load_markets()
        # we use clientOrderId as the order id with self exchange intentionally
        # because most of their endpoints will require clientOrderId
        # explained here: https://github.com/ccxt/ccxt/issues/5674
        request = {
            'clientOrderId': id,
        }
        response = await self.privateGetOrderClientOrderId(self.extend(request, params))
        return self.parse_order(response)

    async def fetch_open_orders(self, symbol=None, since=None, limit=None, params={}):
        await self.load_markets()
        market = None
        request = {}
        if symbol is not None:
            market = self.market(symbol)
            request['symbol'] = market['id']
        response = await self.privateGetOrder(self.extend(request, params))
        return self.parse_orders(response, market, since, limit)

    async def fetch_closed_orders(self, symbol=None, since=None, limit=None, params={}):
        await self.load_markets()
        market = None
        request = {}
        if symbol is not None:
            market = self.market(symbol)
            request['symbol'] = market['id']
        if limit is not None:
            request['limit'] = limit
        if since is not None:
            request['from'] = self.iso8601(since)
        response = await self.privateGetHistoryOrder(self.extend(request, params))
        parsedOrders = self.parse_orders(response, market)
        orders = []
        for i in range(0, len(parsedOrders)):
            order = parsedOrders[i]
            status = order['status']
            if (status == 'closed') or (status == 'canceled'):
                orders.append(order)
        return self.filter_by_since_limit(orders, since, limit)

    async def fetch_my_trades(self, symbol=None, since=None, limit=None, params={}):
        await self.load_markets()
        request = {
            # 'symbol': 'BTC/USD',  # optional
            # 'sort':   'DESC',  # or 'ASC'
            # 'by':     'timestamp',  # or 'id' String timestamp by default, or id
            # 'from':   'Datetime or Number',  # ISO 8601
            # 'till':   'Datetime or Number',
            # 'limit':  100,
            # 'offset': 0,
        }
        market = None
        if symbol is not None:
            market = self.market(symbol)
            request['symbol'] = market['id']
        if since is not None:
            request['from'] = self.iso8601(since)
        if limit is not None:
            request['limit'] = limit
        response = await self.privateGetHistoryTrades(self.extend(request, params))
        #
        #     [
        #         {
        #         "id": 9535486,
        #         "clientOrderId": "f8dbaab336d44d5ba3ff578098a68454",
        #         "orderId": 816088377,
        #         "symbol": "ETHBTC",
        #         "side": "sell",
        #         "quantity": "0.061",
        #         "price": "0.045487",
        #         "fee": "0.000002775",
        #         "timestamp": "2017-05-17T12:32:57.848Z"
        #         },
        #         {
        #         "id": 9535437,
        #         "clientOrderId": "27b9bfc068b44194b1f453c7af511ed6",
        #         "orderId": 816088021,
        #         "symbol": "ETHBTC",
        #         "side": "buy",
        #         "quantity": "0.038",
        #         "price": "0.046000",
        #         "fee": "-0.000000174",
        #         "timestamp": "2017-05-17T12:30:57.848Z"
        #         }
        #     ]
        #
        return self.parse_trades(response, market, since, limit)

    async def fetch_order_trades(self, id, symbol=None, since=None, limit=None, params={}):
        # The id needed here is the exchange's id, and not the clientOrderID,
        # which is the id that is stored in the unified order id
        # To get the exchange's id you need to grab it from order['info']['id']
        await self.load_markets()
        market = None
        if symbol is not None:
            market = self.market(symbol)
        request = {
            'orderId': id,
        }
        response = await self.privateGetHistoryOrderOrderIdTrades(self.extend(request, params))
        numOrders = len(response)
        if numOrders > 0:
            return self.parse_trades(response, market, since, limit)
        raise OrderNotFound(self.id + ' order ' + id + ' not found, ' + self.id + '.fetchOrderTrades() requires an exchange-specific order id, you need to grab it from order["info"]["id"]')

    async def create_deposit_address(self, code, params={}):
        await self.load_markets()
        currency = self.currency(code)
        request = {
            'currency': currency['id'],
        }
        response = await self.privatePostAccountCryptoAddressCurrency(self.extend(request, params))
        address = self.safe_string(response, 'address')
        self.check_address(address)
        tag = self.safe_string(response, 'paymentId')
        return {
            'currency': currency,
            'address': address,
            'tag': tag,
            'info': response,
        }

    async def fetch_deposit_address(self, code, params={}):
        await self.load_markets()
        currency = self.currency(code)
        request = {
            'currency': currency['id'],
        }
        network = self.safe_string(params, 'network')
        if network is not None:
            params = self.omit(params, 'network')
            networks = self.safe_value(self.options, 'networks')
            endpart = self.safe_string(networks, network, network)
            request['currency'] += endpart
        response = await self.privateGetAccountCryptoAddressCurrency(self.extend(request, params))
        address = self.safe_string(response, 'address')
        self.check_address(address)
        tag = self.safe_string(response, 'paymentId')
        return {
            'currency': currency['code'],
            'address': address,
            'tag': tag,
            'info': response,
        }

    async def convert_currency_network(self, code, amount, fromNetwork, toNetwork, params):
        await self.load_markets()
        currency = self.currency(code)
        networks = self.safe_value(self.options, 'networks', {})
        fromNetwork = self.safe_string(networks, fromNetwork, fromNetwork)  # handle ETH>ERC20 alias
        toNetwork = self.safe_string(networks, toNetwork, toNetwork)  # handle ETH>ERC20 alias
        if fromNetwork == toNetwork:
            raise ExchangeError(self.id + ' fromNetwork cannot be the same as toNetwork')
        request = {
            'fromCurrency': currency['id'] + fromNetwork,
            'toCurrency': currency['id'] + toNetwork,
            'amount': float(self.currency_to_precision(code, amount)),
        }
        response = await self.privatePostAccountCryptoTransferConvert(self.extend(request, params))
        return {
            'info': response,
        }

    async def withdraw(self, code, amount, address, tag=None, params={}):
        tag, params = self.handle_withdraw_tag_and_params(tag, params)
        await self.load_markets()
        self.check_address(address)
        currency = self.currency(code)
        request = {
            'currency': currency['id'],
            'amount': float(amount),
            'address': address,
        }
        if tag:
            request['paymentId'] = tag
        networks = self.safe_value(self.options, 'networks', {})
        network = self.safe_string_upper(params, 'network')  # self line allows the user to specify either ERC20 or ETH
        network = self.safe_string(networks, network, network)  # handle ERC20>ETH alias
        if network is not None:
            request['currency'] += network  # when network the currency need to be changed to currency + network
            params = self.omit(params, 'network')
        response = await self.privatePostAccountCryptoWithdraw(self.extend(request, params))
        return {
            'info': response,
            'id': response['id'],
        }

    def nonce(self):
        return self.milliseconds()

    def sign(self, path, api='public', method='GET', params={}, headers=None, body=None):
        url = '/api/' + self.version + '/'
        query = self.omit(params, self.extract_params(path))
        if api == 'public':
            url += api + '/' + self.implode_params(path, params)
            if query:
                url += '?' + self.urlencode(query)
        else:
            self.check_required_credentials()
            url += self.implode_params(path, params)
            if method == 'GET':
                if query:
                    url += '?' + self.urlencode(query)
            elif query:
                body = self.json(query)
            payload = self.encode(self.apiKey + ':' + self.secret)
            auth = self.string_to_base64(payload)
            headers = {
                'Authorization': 'Basic ' + self.decode(auth),
                'Content-Type': 'application/json',
            }
        url = self.urls['api'][api] + url
        return {'url': url, 'method': method, 'body': body, 'headers': headers}

    def handle_errors(self, code, reason, url, method, headers, body, response, requestHeaders, requestBody):
        if response is None:
            return
        if code >= 400:
            feedback = self.id + ' ' + body
            # {"code":504,"message":"Gateway Timeout","description":""}
            if (code == 503) or (code == 504):
                raise ExchangeNotAvailable(feedback)
            # fallback to default error handler on rate limit errors
            # {"code":429,"message":"Too many requests","description":"Too many requests"}
            if code == 429:
                return
            # {"error":{"code":20002,"message":"Order not found","description":""}}
            if body[0] == '{':
                if 'error' in response:
                    errorCode = self.safe_string(response['error'], 'code')
                    self.throw_exactly_matched_exception(self.exceptions, errorCode, feedback)
                    message = self.safe_string(response['error'], 'message')
                    if message == 'Duplicate clientOrderId':
                        raise InvalidOrder(feedback)
            raise ExchangeError(feedback)
