"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("./apigateway.generated");
const base_path_mapping_1 = require("./base-path-mapping");
const restapi_1 = require("./restapi");
/**
 * The minimum version of the SSL protocol that you want API Gateway to use for HTTPS connections.
 */
var SecurityPolicy;
(function (SecurityPolicy) {
    /** Cipher suite TLS 1.0 */
    SecurityPolicy["TLS_1_0"] = "TLS_1_0";
    /** Cipher suite TLS 1.2 */
    SecurityPolicy["TLS_1_2"] = "TLS_1_2";
})(SecurityPolicy = exports.SecurityPolicy || (exports.SecurityPolicy = {}));
class DomainName extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        const endpointType = props.endpointType || restapi_1.EndpointType.REGIONAL;
        const edge = endpointType === restapi_1.EndpointType.EDGE;
        const resource = new apigateway_generated_1.CfnDomainName(this, 'Resource', {
            domainName: props.domainName,
            certificateArn: edge ? props.certificate.certificateArn : undefined,
            regionalCertificateArn: edge ? undefined : props.certificate.certificateArn,
            endpointConfiguration: { types: [endpointType] },
            securityPolicy: props.securityPolicy,
        });
        this.domainName = resource.ref;
        this.domainNameAliasDomainName = edge
            ? resource.attrDistributionDomainName
            : resource.attrRegionalDomainName;
        this.domainNameAliasHostedZoneId = edge
            ? resource.attrDistributionHostedZoneId
            : resource.attrRegionalHostedZoneId;
        if (props.mapping) {
            this.addBasePathMapping(props.mapping);
        }
    }
    /**
     * Imports an existing domain name.
     */
    static fromDomainNameAttributes(scope, id, attrs) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.domainName = attrs.domainName;
                this.domainNameAliasDomainName = attrs.domainNameAliasTarget;
                this.domainNameAliasHostedZoneId = attrs.domainNameAliasHostedZoneId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Maps this domain to an API endpoint.
     * @param targetApi That target API endpoint, requests will be mapped to the deployment stage.
     * @param options Options for mapping to base path with or without a stage
     */
    addBasePathMapping(targetApi, options = {}) {
        const basePath = options.basePath || '/';
        const id = `Map:${basePath}=>${targetApi.node.uniqueId}`;
        return new base_path_mapping_1.BasePathMapping(this, id, {
            domainName: this,
            restApi: targetApi,
            ...options,
        });
    }
}
exports.DomainName = DomainName;
//# sourceMappingURL=data:application/json;base64,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