"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../../lib");
module.exports = {
    'default token authorizer'(test) {
        const stack = new core_1.Stack();
        const func = new lambda.Function(stack, 'myfunction', {
            handler: 'handler',
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const auth = new lib_1.TokenAuthorizer(stack, 'myauthorizer', {
            handler: func,
        });
        const restApi = new lib_1.RestApi(stack, 'myrestapi');
        restApi.root.addMethod('ANY', undefined, {
            authorizer: auth,
            authorizationType: lib_1.AuthorizationType.CUSTOM,
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Authorizer', {
            Type: 'TOKEN',
            RestApiId: stack.resolve(restApi.restApiId),
            IdentitySource: 'method.request.header.Authorization',
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Permission', {
            Action: 'lambda:InvokeFunction',
            Principal: 'apigateway.amazonaws.com',
        }));
        test.ok(auth.authorizerArn.endsWith(`/authorizers/${auth.authorizerId}`), 'Malformed authorizer ARN');
        test.done();
    },
    'default request authorizer'(test) {
        const stack = new core_1.Stack();
        const func = new lambda.Function(stack, 'myfunction', {
            handler: 'handler',
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_12_X,
        });
        const auth = new lib_1.RequestAuthorizer(stack, 'myauthorizer', {
            handler: func,
            resultsCacheTtl: core_1.Duration.seconds(0),
            identitySources: [],
        });
        const restApi = new lib_1.RestApi(stack, 'myrestapi');
        restApi.root.addMethod('ANY', undefined, {
            authorizer: auth,
            authorizationType: lib_1.AuthorizationType.CUSTOM,
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Authorizer', {
            Type: 'REQUEST',
            RestApiId: stack.resolve(restApi.restApiId),
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Permission', {
            Action: 'lambda:InvokeFunction',
            Principal: 'apigateway.amazonaws.com',
        }));
        test.ok(auth.authorizerArn.endsWith(`/authorizers/${auth.authorizerId}`), 'Malformed authorizer ARN');
        test.done();
    },
    'invalid request authorizer config'(test) {
        const stack = new core_1.Stack();
        const func = new lambda.Function(stack, 'myfunction', {
            handler: 'handler',
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_12_X,
        });
        test.throws(() => new lib_1.RequestAuthorizer(stack, 'myauthorizer', {
            handler: func,
            resultsCacheTtl: core_1.Duration.seconds(1),
            identitySources: [],
        }), Error, 'At least one Identity Source is required for a REQUEST-based Lambda authorizer if caching is enabled.');
        test.done();
    },
    'token authorizer with all parameters specified'(test) {
        const stack = new core_1.Stack();
        const func = new lambda.Function(stack, 'myfunction', {
            handler: 'handler',
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const auth = new lib_1.TokenAuthorizer(stack, 'myauthorizer', {
            handler: func,
            identitySource: 'method.request.header.whoami',
            validationRegex: 'a-hacker',
            authorizerName: 'myauthorizer',
            resultsCacheTtl: core_1.Duration.minutes(1),
        });
        const restApi = new lib_1.RestApi(stack, 'myrestapi');
        restApi.root.addMethod('ANY', undefined, {
            authorizer: auth,
            authorizationType: lib_1.AuthorizationType.CUSTOM,
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Authorizer', {
            Type: 'TOKEN',
            RestApiId: stack.resolve(restApi.restApiId),
            IdentitySource: 'method.request.header.whoami',
            IdentityValidationExpression: 'a-hacker',
            Name: 'myauthorizer',
            AuthorizerResultTtlInSeconds: 60,
        }));
        test.done();
    },
    'request authorizer with all parameters specified'(test) {
        const stack = new core_1.Stack();
        const func = new lambda.Function(stack, 'myfunction', {
            handler: 'handler',
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_12_X,
        });
        const auth = new lib_1.RequestAuthorizer(stack, 'myauthorizer', {
            handler: func,
            identitySources: [lib_1.IdentitySource.header('whoami')],
            authorizerName: 'myauthorizer',
            resultsCacheTtl: core_1.Duration.minutes(1),
        });
        const restApi = new lib_1.RestApi(stack, 'myrestapi');
        restApi.root.addMethod('ANY', undefined, {
            authorizer: auth,
            authorizationType: lib_1.AuthorizationType.CUSTOM,
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Authorizer', {
            Type: 'REQUEST',
            RestApiId: stack.resolve(restApi.restApiId),
            IdentitySource: 'method.request.header.whoami',
            Name: 'myauthorizer',
            AuthorizerResultTtlInSeconds: 60,
        }));
        test.done();
    },
    'token authorizer with assume role'(test) {
        const stack = new core_1.Stack();
        const func = new lambda.Function(stack, 'myfunction', {
            handler: 'handler',
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const role = new iam.Role(stack, 'authorizerassumerole', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
            roleName: 'authorizerassumerole',
        });
        const auth = new lib_1.TokenAuthorizer(stack, 'myauthorizer', {
            handler: func,
            assumeRole: role,
        });
        const restApi = new lib_1.RestApi(stack, 'myrestapi');
        restApi.root.addMethod('ANY', undefined, {
            authorizer: auth,
            authorizationType: lib_1.AuthorizationType.CUSTOM,
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Authorizer', {
            Type: 'TOKEN',
            RestApiId: stack.resolve(restApi.restApiId),
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role'));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            Roles: [
                stack.resolve(role.roleName),
            ],
            PolicyDocument: {
                Statement: [
                    {
                        Resource: stack.resolve(func.functionArn),
                        Action: 'lambda:InvokeFunction',
                        Effect: 'Allow',
                    },
                ],
            },
        }, assert_1.ResourcePart.Properties, true));
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::Lambda::Permission'));
        test.done();
    },
    'request authorizer with assume role'(test) {
        const stack = new core_1.Stack();
        const func = new lambda.Function(stack, 'myfunction', {
            handler: 'handler',
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_12_X,
        });
        const role = new iam.Role(stack, 'authorizerassumerole', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
            roleName: 'authorizerassumerole',
        });
        const auth = new lib_1.RequestAuthorizer(stack, 'myauthorizer', {
            handler: func,
            assumeRole: role,
            resultsCacheTtl: core_1.Duration.seconds(0),
            identitySources: [],
        });
        const restApi = new lib_1.RestApi(stack, 'myrestapi');
        restApi.root.addMethod('ANY', undefined, {
            authorizer: auth,
            authorizationType: lib_1.AuthorizationType.CUSTOM,
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Authorizer', {
            Type: 'REQUEST',
            RestApiId: stack.resolve(restApi.restApiId),
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role'));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            Roles: [
                stack.resolve(role.roleName),
            ],
            PolicyDocument: {
                Statement: [
                    {
                        Resource: stack.resolve(func.functionArn),
                        Action: 'lambda:InvokeFunction',
                        Effect: 'Allow',
                    },
                ],
            },
        }, assert_1.ResourcePart.Properties, true));
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::Lambda::Permission'));
        test.done();
    },
    'token authorizer throws when not attached to a rest api'(test) {
        const stack = new core_1.Stack();
        const func = new lambda.Function(stack, 'myfunction', {
            handler: 'handler',
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_12_X,
        });
        const auth = new lib_1.TokenAuthorizer(stack, 'myauthorizer', {
            handler: func,
        });
        test.throws(() => stack.resolve(auth.authorizerArn), /must be attached to a RestApi/);
        test.done();
    },
    'request authorizer throws when not attached to a rest api'(test) {
        const stack = new core_1.Stack();
        const func = new lambda.Function(stack, 'myfunction', {
            handler: 'handler',
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_12_X,
        });
        const auth = new lib_1.RequestAuthorizer(stack, 'myauthorizer', {
            handler: func,
            identitySources: [lib_1.IdentitySource.header('myheader')],
        });
        test.throws(() => stack.resolve(auth.authorizerArn), /must be attached to a RestApi/);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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