"use strict";
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const apigw = require("../lib");
module.exports = {
    'minimal setup'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'my-api');
        api.root.addMethod('GET'); // must have at least one method
        // THEN
        assert_1.expect(stack).toMatch({
            Resources: {
                myapi4C7BF186: {
                    Type: 'AWS::ApiGateway::RestApi',
                    Properties: {
                        Name: 'my-api',
                    },
                },
                myapiGETF990CE3C: {
                    Type: 'AWS::ApiGateway::Method',
                    Properties: {
                        HttpMethod: 'GET',
                        ResourceId: { 'Fn::GetAtt': ['myapi4C7BF186', 'RootResourceId'] },
                        RestApiId: { Ref: 'myapi4C7BF186' },
                        AuthorizationType: 'NONE',
                        Integration: {
                            Type: 'MOCK',
                        },
                    },
                },
                myapiDeployment92F2CB4972a890db5063ec679071ba7eefc76f2a: {
                    Type: 'AWS::ApiGateway::Deployment',
                    Properties: {
                        RestApiId: { Ref: 'myapi4C7BF186' },
                        Description: 'Automatically created by the RestApi construct',
                    },
                    DependsOn: ['myapiGETF990CE3C'],
                },
                myapiDeploymentStageprod298F01AF: {
                    Type: 'AWS::ApiGateway::Stage',
                    Properties: {
                        RestApiId: { Ref: 'myapi4C7BF186' },
                        DeploymentId: { Ref: 'myapiDeployment92F2CB4972a890db5063ec679071ba7eefc76f2a' },
                        StageName: 'prod',
                    },
                },
                myapiCloudWatchRole095452E5: {
                    Type: 'AWS::IAM::Role',
                    Properties: {
                        AssumeRolePolicyDocument: {
                            Statement: [
                                {
                                    Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'apigateway.amazonaws.com' },
                                },
                            ],
                            Version: '2012-10-17',
                        },
                        ManagedPolicyArns: [
                            { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::aws:policy/service-role/AmazonAPIGatewayPushToCloudWatchLogs']] },
                        ],
                    },
                },
                myapiAccountEC421A0A: {
                    Type: 'AWS::ApiGateway::Account',
                    Properties: {
                        CloudWatchRoleArn: { 'Fn::GetAtt': ['myapiCloudWatchRole095452E5', 'Arn'] },
                    },
                    DependsOn: ['myapi4C7BF186'],
                },
            },
            Outputs: {
                myapiEndpoint3628AFE3: {
                    Value: {
                        'Fn::Join': ['', [
                                'https://',
                                { Ref: 'myapi4C7BF186' },
                                '.execute-api.',
                                { Ref: 'AWS::Region' },
                                '.',
                                { Ref: 'AWS::URLSuffix' },
                                '/',
                                { Ref: 'myapiDeploymentStageprod298F01AF' },
                                '/',
                            ]],
                    },
                },
            },
        });
        test.done();
    },
    'defaultChild is set correctly'(test) {
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'my-api');
        test.ok(api.node.defaultChild instanceof apigw.CfnRestApi);
        test.done();
    },
    '"name" is defaulted to resource physical name'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'restapi', {
            deploy: false,
            cloudWatchRole: false,
        });
        api.root.addMethod('GET');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::RestApi', {
            Name: 'restapi',
        }));
        test.done();
    },
    'fails in synthesis if there are no methods'(test) {
        // GIVEN
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'my-stack');
        const api = new apigw.RestApi(stack, 'API');
        // WHEN
        api.root.addResource('foo');
        api.root.addResource('bar').addResource('goo');
        // THEN
        test.throws(() => app.synth(), /The REST API doesn't contain any methods/);
        test.done();
    },
    '"addResource" can be used on "IRestApiResource" to form a tree'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'restapi', {
            deploy: false,
            cloudWatchRole: false,
            restApiName: 'my-rest-api',
        });
        api.root.addMethod('GET');
        // WHEN
        const foo = api.root.addResource('foo');
        api.root.addResource('bar');
        foo.addResource('{hello}');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Resource', {
            PathPart: 'foo',
            ParentId: { 'Fn::GetAtt': ['restapiC5611D27', 'RootResourceId'] },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Resource', {
            PathPart: 'bar',
            ParentId: { 'Fn::GetAtt': ['restapiC5611D27', 'RootResourceId'] },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Resource', {
            PathPart: '{hello}',
            ParentId: { Ref: 'restapifooF697E056' },
        }));
        test.done();
    },
    '"addResource" allows configuration of proxy paths'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'restapi', {
            deploy: false,
            cloudWatchRole: false,
            restApiName: 'my-rest-api',
        });
        // WHEN
        const proxy = api.root.addResource('{proxy+}');
        proxy.addMethod('ANY');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Resource', {
            PathPart: '{proxy+}',
            ParentId: { 'Fn::GetAtt': ['restapiC5611D27', 'RootResourceId'] },
        }));
        test.done();
    },
    '"addMethod" can be used to add methods to resources'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'restapi', { deploy: false, cloudWatchRole: false });
        const r1 = api.root.addResource('r1');
        // WHEN
        api.root.addMethod('GET');
        r1.addMethod('POST');
        // THEN
        assert_1.expect(stack).toMatch({
            Resources: {
                restapiC5611D27: {
                    Type: 'AWS::ApiGateway::RestApi',
                    Properties: {
                        Name: 'restapi',
                    },
                },
                restapir1CF2997EA: {
                    Type: 'AWS::ApiGateway::Resource',
                    Properties: {
                        ParentId: {
                            'Fn::GetAtt': [
                                'restapiC5611D27',
                                'RootResourceId',
                            ],
                        },
                        PathPart: 'r1',
                        RestApiId: {
                            Ref: 'restapiC5611D27',
                        },
                    },
                },
                restapir1POST766920C4: {
                    Type: 'AWS::ApiGateway::Method',
                    Properties: {
                        HttpMethod: 'POST',
                        ResourceId: {
                            Ref: 'restapir1CF2997EA',
                        },
                        RestApiId: {
                            Ref: 'restapiC5611D27',
                        },
                        AuthorizationType: 'NONE',
                        Integration: {
                            Type: 'MOCK',
                        },
                    },
                },
                restapiGET6FC1785A: {
                    Type: 'AWS::ApiGateway::Method',
                    Properties: {
                        HttpMethod: 'GET',
                        ResourceId: {
                            'Fn::GetAtt': [
                                'restapiC5611D27',
                                'RootResourceId',
                            ],
                        },
                        RestApiId: {
                            Ref: 'restapiC5611D27',
                        },
                        AuthorizationType: 'NONE',
                        Integration: {
                            Type: 'MOCK',
                        },
                    },
                },
            },
        });
        test.done();
    },
    'resourcePath returns the full path of the resource within the API'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'restapi');
        // WHEN
        const r1 = api.root.addResource('r1');
        const r11 = r1.addResource('r1_1');
        const r12 = r1.addResource('r1_2');
        const r121 = r12.addResource('r1_2_1');
        const r2 = api.root.addResource('r2');
        // THEN
        test.deepEqual(api.root.path, '/');
        test.deepEqual(r1.path, '/r1');
        test.deepEqual(r11.path, '/r1/r1_1');
        test.deepEqual(r12.path, '/r1/r1_2');
        test.deepEqual(r121.path, '/r1/r1_2/r1_2_1');
        test.deepEqual(r2.path, '/r2');
        test.done();
    },
    'resource path part validation'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'restapi');
        // THEN
        test.throws(() => api.root.addResource('foo/'));
        api.root.addResource('boom-bam');
        test.throws(() => api.root.addResource('illegal()'));
        api.root.addResource('{foo}');
        test.throws(() => api.root.addResource('foo{bar}'));
        test.done();
    },
    'fails if "deployOptions" is set with "deploy" disabled'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // THEN
        test.throws(() => new apigw.RestApi(stack, 'myapi', {
            deploy: false,
            deployOptions: { cachingEnabled: true },
        }), /Cannot set 'deployOptions' if 'deploy' is disabled/);
        test.done();
    },
    'CloudWatch role is created for API Gateway'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'myapi');
        api.root.addMethod('GET');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role'));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Account'));
        test.done();
    },
    'fromRestApiId'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const imported = apigw.RestApi.fromRestApiId(stack, 'imported-api', 'api-rxt4498f');
        // THEN
        test.deepEqual(stack.resolve(imported.restApiId), 'api-rxt4498f');
        test.done();
    },
    '"url" and "urlForPath" return the URL endpoints of the deployed API'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        api.root.addMethod('GET');
        // THEN
        test.deepEqual(stack.resolve(api.url), { 'Fn::Join': ['',
                ['https://',
                    { Ref: 'apiC8550315' },
                    '.execute-api.',
                    { Ref: 'AWS::Region' },
                    '.',
                    { Ref: 'AWS::URLSuffix' },
                    '/',
                    { Ref: 'apiDeploymentStageprod896C8101' },
                    '/']] });
        test.deepEqual(stack.resolve(api.urlForPath('/foo/bar')), { 'Fn::Join': ['',
                ['https://',
                    { Ref: 'apiC8550315' },
                    '.execute-api.',
                    { Ref: 'AWS::Region' },
                    '.',
                    { Ref: 'AWS::URLSuffix' },
                    '/',
                    { Ref: 'apiDeploymentStageprod896C8101' },
                    '/foo/bar']] });
        test.done();
    },
    '"urlForPath" would not work if there is no deployment'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api', { deploy: false });
        api.root.addMethod('GET');
        // THEN
        test.throws(() => api.url, /Cannot determine deployment stage for API from "deploymentStage". Use "deploy" or explicitly set "deploymentStage"/);
        test.throws(() => api.urlForPath('/foo'), /Cannot determine deployment stage for API from "deploymentStage". Use "deploy" or explicitly set "deploymentStage"/);
        test.done();
    },
    '"urlForPath" requires that path will begin with "/"'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        api.root.addMethod('GET');
        // THEN
        test.throws(() => api.urlForPath('foo'), /Path must begin with \"\/\": foo/);
        test.done();
    },
    '"executeApiArn" returns the execute-api ARN for a resource/method'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        api.root.addMethod('GET');
        // WHEN
        const arn = api.arnForExecuteApi('method', '/path', 'stage');
        // THEN
        test.deepEqual(stack.resolve(arn), { 'Fn::Join': ['',
                ['arn:',
                    { Ref: 'AWS::Partition' },
                    ':execute-api:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':',
                    { Ref: 'apiC8550315' },
                    '/stage/method/path']] });
        test.done();
    },
    '"executeApiArn" path must begin with "/"'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        api.root.addMethod('GET');
        // THEN
        test.throws(() => api.arnForExecuteApi('method', 'hey-path', 'stage'), /"path" must begin with a "\/": 'hey-path'/);
        test.done();
    },
    '"executeApiArn" will convert ANY to "*"'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        const method = api.root.addMethod('ANY');
        // THEN
        test.deepEqual(stack.resolve(method.methodArn), { 'Fn::Join': ['',
                ['arn:',
                    { Ref: 'AWS::Partition' },
                    ':execute-api:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':',
                    { Ref: 'apiC8550315' },
                    '/',
                    { Ref: 'apiDeploymentStageprod896C8101' },
                    '/*/']] });
        test.done();
    },
    '"endpointTypes" can be used to specify endpoint configuration for the api'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'api', {
            endpointTypes: [apigw.EndpointType.EDGE, apigw.EndpointType.PRIVATE],
        });
        api.root.addMethod('GET');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::RestApi', {
            EndpointConfiguration: {
                Types: [
                    'EDGE',
                    'PRIVATE',
                ],
            },
        }));
        test.done();
    },
    '"endpointConfiguration" can be used to specify endpoint types for the api'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'api', {
            endpointConfiguration: {
                types: [apigw.EndpointType.EDGE, apigw.EndpointType.PRIVATE],
            },
        });
        api.root.addMethod('GET');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::RestApi', {
            EndpointConfiguration: {
                Types: ['EDGE', 'PRIVATE'],
            },
        }));
        test.done();
    },
    '"endpointConfiguration" can be used to specify vpc endpoints on the API'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'api', {
            endpointConfiguration: {
                types: [apigw.EndpointType.EDGE, apigw.EndpointType.PRIVATE],
                vpcEndpoints: [
                    aws_ec2_1.GatewayVpcEndpoint.fromGatewayVpcEndpointId(stack, 'ImportedEndpoint', 'vpcEndpoint'),
                    aws_ec2_1.GatewayVpcEndpoint.fromGatewayVpcEndpointId(stack, 'ImportedEndpoint2', 'vpcEndpoint2'),
                ],
            },
        });
        api.root.addMethod('GET');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::RestApi', {
            EndpointConfiguration: {
                Types: [
                    'EDGE',
                    'PRIVATE',
                ],
                VpcEndpointIds: [
                    'vpcEndpoint',
                    'vpcEndpoint2',
                ],
            },
        }));
        test.done();
    },
    '"endpointTypes" and "endpointConfiguration" can NOT both be used to specify endpoint configuration for the api'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // THEN
        test.throws(() => new apigw.RestApi(stack, 'api', {
            endpointConfiguration: {
                types: [apigw.EndpointType.PRIVATE],
                vpcEndpoints: [aws_ec2_1.GatewayVpcEndpoint.fromGatewayVpcEndpointId(stack, 'ImportedEndpoint', 'vpcEndpoint')],
            },
            endpointTypes: [apigw.EndpointType.PRIVATE],
        }), /Only one of the RestApi props, endpointTypes or endpointConfiguration, is allowed/);
        test.done();
    },
    '"cloneFrom" can be used to clone an existing API'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cloneFrom = apigw.RestApi.fromRestApiId(stack, 'RestApi', 'foobar');
        // WHEN
        const api = new apigw.RestApi(stack, 'api', {
            cloneFrom,
        });
        api.root.addMethod('GET');
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::RestApi', {
            CloneFrom: 'foobar',
            Name: 'api',
        }));
        test.done();
    },
    'allow taking a dependency on the rest api (includes deployment and stage)'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'myapi');
        api.root.addMethod('GET');
        const resource = new core_1.CfnResource(stack, 'DependsOnRestApi', { type: 'My::Resource' });
        // WHEN
        resource.node.addDependency(api);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('My::Resource', {
            DependsOn: [
                'myapiAccountC3A4750C',
                'myapiCloudWatchRoleEB425128',
                'myapiGET9B7CD29E',
                'myapiDeploymentB7EF8EB7b8edc043bcd33e0d85a3c85151f47e98',
                'myapiDeploymentStageprod329F21FF',
                'myapi162F20B8',
            ],
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'defaultIntegration and defaultMethodOptions can be used at any level'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const rootInteg = new apigw.AwsIntegration({
            service: 's3',
            action: 'GetObject',
        });
        // WHEN
        const api = new apigw.RestApi(stack, 'myapi', {
            defaultIntegration: rootInteg,
            defaultMethodOptions: {
                authorizer: { authorizerId: 'AUTHID' },
                authorizationType: apigw.AuthorizationType.IAM,
            },
        });
        // CASE #1: should inherit integration and options from root resource
        api.root.addMethod('GET');
        const child = api.root.addResource('child');
        // CASE #2: should inherit integration from root and method options, but
        // "authorizationType" will be overridden to "None" instead of "IAM"
        child.addMethod('POST', undefined, {
            authorizationType: apigw.AuthorizationType.COGNITO,
        });
        const child2 = api.root.addResource('child2', {
            defaultIntegration: new apigw.MockIntegration(),
            defaultMethodOptions: {
                authorizer: { authorizerId: 'AUTHID2' },
            },
        });
        // CASE #3: integartion and authorizer ID are inherited from child2
        child2.addMethod('DELETE');
        // CASE #4: same as case #3, but integration is customized
        child2.addMethod('PUT', new apigw.AwsIntegration({ action: 'foo', service: 'bar' }));
        // THEN
        // CASE #1
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ApiGateway::Method', {
            HttpMethod: 'GET',
            ResourceId: { 'Fn::GetAtt': ['myapi162F20B8', 'RootResourceId'] },
            Integration: { Type: 'AWS' },
            AuthorizerId: 'AUTHID',
            AuthorizationType: 'AWS_IAM',
        }));
        // CASE #2
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ApiGateway::Method', {
            HttpMethod: 'POST',
            ResourceId: { Ref: 'myapichildA0A65412' },
            Integration: { Type: 'AWS' },
            AuthorizerId: 'AUTHID',
            AuthorizationType: 'COGNITO_USER_POOLS',
        }));
        // CASE #3
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ApiGateway::Method', {
            HttpMethod: 'DELETE',
            Integration: { Type: 'MOCK' },
            AuthorizerId: 'AUTHID2',
            AuthorizationType: 'AWS_IAM',
        }));
        // CASE #4
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ApiGateway::Method', {
            HttpMethod: 'PUT',
            Integration: { Type: 'AWS' },
            AuthorizerId: 'AUTHID2',
            AuthorizationType: 'AWS_IAM',
        }));
        test.done();
    },
    'addModel is supported'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'myapi');
        api.root.addMethod('OPTIONS');
        // WHEN
        api.addModel('model', {
            schema: {
                schema: apigw.JsonSchemaVersion.DRAFT4,
                title: 'test',
                type: apigw.JsonSchemaType.OBJECT,
                properties: { message: { type: apigw.JsonSchemaType.STRING } },
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Model', {
            RestApiId: { Ref: stack.getLogicalId(api.node.findChild('Resource')) },
            Schema: {
                $schema: 'http://json-schema.org/draft-04/schema#',
                title: 'test',
                type: 'object',
                properties: { message: { type: 'string' } },
            },
        }));
        test.done();
    },
    'addRequestValidator is supported'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'myapi');
        api.root.addMethod('OPTIONS');
        // WHEN
        api.addRequestValidator('params-validator', {
            requestValidatorName: 'Parameters',
            validateRequestBody: false,
            validateRequestParameters: true,
        });
        api.addRequestValidator('body-validator', {
            requestValidatorName: 'Body',
            validateRequestBody: true,
            validateRequestParameters: false,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::RequestValidator', {
            RestApiId: { Ref: stack.getLogicalId(api.node.findChild('Resource')) },
            Name: 'Parameters',
            ValidateRequestBody: false,
            ValidateRequestParameters: true,
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::RequestValidator', {
            RestApiId: { Ref: stack.getLogicalId(api.node.findChild('Resource')) },
            Name: 'Body',
            ValidateRequestBody: true,
            ValidateRequestParameters: false,
        }));
        test.done();
    },
    'creates output with given "exportName"'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'myapi', { endpointExportName: 'my-given-export-name' });
        api.root.addMethod('GET');
        // THEN
        test.deepEqual(assert_1.SynthUtils.toCloudFormation(stack).Outputs, {
            myapiEndpoint8EB17201: {
                Value: {
                    'Fn::Join': [
                        '',
                        [
                            'https://',
                            { Ref: 'myapi162F20B8' },
                            '.execute-api.',
                            { Ref: 'AWS::Region' },
                            '.',
                            { Ref: 'AWS::URLSuffix' },
                            '/',
                            { Ref: 'myapiDeploymentStageprod329F21FF' },
                            '/',
                        ],
                    ],
                },
                Export: { Name: 'my-given-export-name' },
            },
        });
        test.done();
    },
    'creates output when "exportName" is not specified'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'myapi');
        api.root.addMethod('GET');
        // THEN
        test.deepEqual(assert_1.SynthUtils.toCloudFormation(stack).Outputs, {
            myapiEndpoint8EB17201: {
                Value: {
                    'Fn::Join': [
                        '',
                        [
                            'https://',
                            { Ref: 'myapi162F20B8' },
                            '.execute-api.',
                            { Ref: 'AWS::Region' },
                            '.',
                            { Ref: 'AWS::URLSuffix' },
                            '/',
                            { Ref: 'myapiDeploymentStageprod329F21FF' },
                            '/',
                        ],
                    ],
                },
            },
        });
        test.done();
    },
    'gateway response resource is created'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'restapi', {
            deploy: false,
            cloudWatchRole: false,
        });
        api.root.addMethod('GET');
        api.addGatewayResponse('test-response', {
            type: apigw.ResponseType.ACCESS_DENIED,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::GatewayResponse', {
            ResponseType: 'ACCESS_DENIED',
            RestApiId: stack.resolve(api.restApiId),
            StatusCode: assert_1.ABSENT,
            ResponseParameters: assert_1.ABSENT,
            ResponseTemplates: assert_1.ABSENT,
        }));
        test.done();
    },
    'gateway response resource is created with parameters'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'restapi', {
            deploy: false,
            cloudWatchRole: false,
        });
        api.root.addMethod('GET');
        api.addGatewayResponse('test-response', {
            type: apigw.ResponseType.AUTHORIZER_FAILURE,
            statusCode: '500',
            responseHeaders: {
                'Access-Control-Allow-Origin': 'test.com',
                'test-key': 'test-value',
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::GatewayResponse', {
            ResponseType: 'AUTHORIZER_FAILURE',
            RestApiId: stack.resolve(api.restApiId),
            StatusCode: '500',
            ResponseParameters: {
                'gatewayresponse.header.Access-Control-Allow-Origin': 'test.com',
                'gatewayresponse.header.test-key': 'test-value',
            },
            ResponseTemplates: assert_1.ABSENT,
        }));
        test.done();
    },
    'gateway response resource is created with templates'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'restapi', {
            deploy: false,
            cloudWatchRole: false,
        });
        api.root.addMethod('GET');
        api.addGatewayResponse('test-response', {
            type: apigw.ResponseType.AUTHORIZER_FAILURE,
            statusCode: '500',
            templates: {
                'application/json': '{ "message": $context.error.messageString, "statusCode": "488" }',
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::GatewayResponse', {
            ResponseType: 'AUTHORIZER_FAILURE',
            RestApiId: stack.resolve(api.restApiId),
            StatusCode: '500',
            ResponseParameters: assert_1.ABSENT,
            ResponseTemplates: {
                'application/json': '{ "message": $context.error.messageString, "statusCode": "488" }',
            },
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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