"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
/**
 * The type of invocation to use for a Lambda Action.
 */
var LambdaInvocationType;
(function (LambdaInvocationType) {
    /**
     * The function will be invoked asynchronously.
     */
    LambdaInvocationType["EVENT"] = "Event";
    /**
     * The function will be invoked sychronously. Use RequestResponse only when
     * you want to make a mail flow decision, such as whether to stop the receipt
     * rule or the receipt rule set.
     */
    LambdaInvocationType["REQUEST_RESPONSE"] = "RequestResponse";
})(LambdaInvocationType = exports.LambdaInvocationType || (exports.LambdaInvocationType = {}));
/**
 * Calls an AWS Lambda function, and optionally, publishes a notification to
 * Amazon SNS.
 */
class Lambda {
    constructor(props) {
        this.props = props;
    }
    bind(rule) {
        // Allow SES to invoke Lambda function
        // See https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-permissions.html#receiving-email-permissions-lambda
        const permissionId = 'AllowSes';
        if (!this.props.function.permissionsNode.tryFindChild(permissionId)) {
            this.props.function.addPermission(permissionId, {
                action: 'lambda:InvokeFunction',
                principal: new iam.ServicePrincipal('ses.amazonaws.com'),
                sourceAccount: cdk.Aws.ACCOUNT_ID,
            });
        }
        // Ensure permission is deployed before rule
        const permission = this.props.function.permissionsNode.tryFindChild(permissionId);
        if (permission) { // The Lambda could be imported
            rule.node.addDependency(permission);
        }
        else {
            // tslint:disable-next-line:max-line-length
            rule.node.addWarning('This rule is using a Lambda action with an imported function. Ensure permission is given to SES to invoke that function.');
        }
        return {
            lambdaAction: {
                functionArn: this.props.function.functionArn,
                invocationType: this.props.invocationType,
                topicArn: this.props.topic ? this.props.topic.topicArn : undefined,
            },
        };
    }
}
exports.Lambda = Lambda;
//# sourceMappingURL=data:application/json;base64,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