"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
/**
 * Saves the received message to an Amazon S3 bucket and, optionally, publishes
 * a notification to Amazon SNS.
 */
class S3 {
    constructor(props) {
        this.props = props;
    }
    bind(rule) {
        // Allow SES to write to S3 bucket
        // See https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-permissions.html#receiving-email-permissions-s3
        const keyPattern = this.props.objectKeyPrefix || '';
        const s3Statement = new iam.PolicyStatement({
            actions: ['s3:PutObject'],
            principals: [new iam.ServicePrincipal('ses.amazonaws.com')],
            resources: [this.props.bucket.arnForObjects(`${keyPattern}*`)],
            conditions: {
                StringEquals: {
                    'aws:Referer': cdk.Aws.ACCOUNT_ID,
                },
            },
        });
        this.props.bucket.addToResourcePolicy(s3Statement);
        const policy = this.props.bucket.node.tryFindChild('Policy');
        if (policy) { // The bucket could be imported
            rule.node.addDependency(policy);
        }
        else {
            rule.node.addWarning('This rule is using a S3 action with an imported bucket. Ensure permission is given to SES to write to that bucket.');
        }
        // Allow SES to use KMS master key
        // See https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-permissions.html#receiving-email-permissions-kms
        if (this.props.kmsKey && !/alias\/aws\/ses$/.test(this.props.kmsKey.keyArn)) {
            const kmsStatement = new iam.PolicyStatement({
                actions: ['km:Encrypt', 'kms:GenerateDataKey'],
                principals: [new iam.ServicePrincipal('ses.amazonaws.com')],
                resources: ['*'],
                conditions: {
                    Null: {
                        'kms:EncryptionContext:aws:ses:rule-name': 'false',
                        'kms:EncryptionContext:aws:ses:message-id': 'false',
                    },
                    StringEquals: {
                        'kms:EncryptionContext:aws:ses:source-account': cdk.Aws.ACCOUNT_ID,
                    },
                },
            });
            this.props.kmsKey.addToResourcePolicy(kmsStatement);
        }
        return {
            s3Action: {
                bucketName: this.props.bucket.bucketName,
                kmsKeyArn: this.props.kmsKey ? this.props.kmsKey.keyArn : undefined,
                objectKeyPrefix: this.props.objectKeyPrefix,
                topicArn: this.props.topic ? this.props.topic.topicArn : undefined,
            },
        };
    }
}
exports.S3 = S3;
//# sourceMappingURL=data:application/json;base64,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