# -*- coding: utf-8 -*-

# PLEASE DO NOT EDIT THIS FILE, IT IS GENERATED AND WILL BE OVERWRITTEN:
# https://github.com/ccxt/ccxt/blob/master/CONTRIBUTING.md#how-to-contribute-code

from ccxt.base.exchange import Exchange
from ccxt.base.errors import ExchangeError
from ccxt.base.errors import PermissionDenied
from ccxt.base.errors import AccountSuspended
from ccxt.base.errors import ArgumentsRequired
from ccxt.base.errors import BadRequest
from ccxt.base.errors import BadSymbol
from ccxt.base.errors import InsufficientFunds
from ccxt.base.errors import InvalidAddress
from ccxt.base.errors import InvalidOrder
from ccxt.base.errors import OrderNotFound
from ccxt.base.errors import RateLimitExceeded
from ccxt.base.errors import ExchangeNotAvailable
from ccxt.base.errors import OnMaintenance
from ccxt.base.errors import AuthenticationError
from ccxt.base.decimal_to_precision import ROUND
from ccxt.base.decimal_to_precision import TRUNCATE
from ccxt.base.decimal_to_precision import DECIMAL_PLACES
from ccxt.base.decimal_to_precision import SIGNIFICANT_DIGITS
from ccxt.base.precise import Precise


class bitvavo(Exchange):

    def describe(self):
        return self.deep_extend(super(bitvavo, self).describe(), {
            'id': 'bitvavo',
            'name': 'Bitvavo',
            'countries': ['NL'],  # Netherlands
            'rateLimit': 60,  # 1000 requests per minute
            'version': 'v2',
            'certified': True,
            'pro': True,
            'has': {
                'CORS': None,
                'spot': True,
                'margin': False,
                'swap': False,
                'future': False,
                'option': False,
                'addMargin': False,
                'cancelAllOrders': True,
                'cancelOrder': True,
                'createOrder': True,
                'createReduceOnlyOrder': False,
                'createStopLimitOrder': True,
                'createStopMarketOrder': True,
                'createStopOrder': True,
                'editOrder': True,
                'fetchBalance': True,
                'fetchBorrowRate': False,
                'fetchBorrowRateHistories': False,
                'fetchBorrowRateHistory': False,
                'fetchBorrowRates': False,
                'fetchBorrowRatesPerSymbol': False,
                'fetchCurrencies': True,
                'fetchDepositAddress': True,
                'fetchDeposits': True,
                'fetchFundingHistory': False,
                'fetchFundingRate': False,
                'fetchFundingRateHistory': False,
                'fetchFundingRates': False,
                'fetchIndexOHLCV': False,
                'fetchLeverage': False,
                'fetchLeverageTiers': False,
                'fetchMarginMode': False,
                'fetchMarkets': True,
                'fetchMarkOHLCV': False,
                'fetchMyTrades': True,
                'fetchOHLCV': True,
                'fetchOpenInterestHistory': False,
                'fetchOpenOrders': True,
                'fetchOrder': True,
                'fetchOrderBook': True,
                'fetchOrders': True,
                'fetchPosition': False,
                'fetchPositionMode': False,
                'fetchPositions': False,
                'fetchPositionsRisk': False,
                'fetchPremiumIndexOHLCV': False,
                'fetchTicker': True,
                'fetchTickers': True,
                'fetchTime': True,
                'fetchTrades': True,
                'fetchTradingFee': False,
                'fetchTradingFees': True,
                'fetchTransfer': False,
                'fetchTransfers': False,
                'fetchWithdrawals': True,
                'reduceMargin': False,
                'setLeverage': False,
                'setMarginMode': False,
                'setPositionMode': False,
                'transfer': False,
                'withdraw': True,
            },
            'timeframes': {
                '1m': '1m',
                '5m': '5m',
                '15m': '15m',
                '30m': '30m',
                '1h': '1h',
                '2h': '2h',
                '4h': '4h',
                '6h': '6h',
                '8h': '8h',
                '12h': '12h',
                '1d': '1d',
            },
            'urls': {
                'logo': 'https://user-images.githubusercontent.com/1294454/169202626-bd130fc5-fcf9-41bb-8d97-6093225c73cd.jpg',
                'api': {
                    'public': 'https://api.bitvavo.com',
                    'private': 'https://api.bitvavo.com',
                },
                'www': 'https://bitvavo.com/',
                'doc': 'https://docs.bitvavo.com/',
                'fees': 'https://bitvavo.com/en/fees',
                'referral': 'https://bitvavo.com/?a=24F34952F7',
            },
            'api': {
                'public': {
                    'get': {
                        'time': 1,
                        'markets': 1,
                        'assets': 1,
                        '{market}/book': 1,
                        '{market}/trades': 5,
                        '{market}/candles': 1,
                        'ticker/price': 1,
                        'ticker/book': 1,
                        'ticker/24h': {'cost': 1, 'noMarket': 25},
                    },
                },
                'private': {
                    'get': {
                        'account': 1,
                        'order': 1,
                        'orders': 5,
                        'ordersOpen': {'cost': 1, 'noMarket': 25},
                        'trades': 5,
                        'balance': 5,
                        'deposit': 1,
                        'depositHistory': 5,
                        'withdrawalHistory': 5,
                    },
                    'post': {
                        'order': 1,
                        'withdrawal': 1,
                    },
                    'put': {
                        'order': 1,
                    },
                    'delete': {
                        'order': 1,
                        'orders': 1,
                    },
                },
            },
            'fees': {
                'trading': {
                    'tierBased': True,
                    'percentage': True,
                    'taker': self.parse_number('0.0025'),
                    'maker': self.parse_number('0.002'),
                    'tiers': {
                        'taker': [
                            [self.parse_number('0'), self.parse_number('0.0025')],
                            [self.parse_number('100000'), self.parse_number('0.0020')],
                            [self.parse_number('250000'), self.parse_number('0.0016')],
                            [self.parse_number('500000'), self.parse_number('0.0012')],
                            [self.parse_number('1000000'), self.parse_number('0.0010')],
                            [self.parse_number('2500000'), self.parse_number('0.0008')],
                            [self.parse_number('5000000'), self.parse_number('0.0006')],
                            [self.parse_number('10000000'), self.parse_number('0.0005')],
                            [self.parse_number('25000000'), self.parse_number('0.0004')],
                        ],
                        'maker': [
                            [self.parse_number('0'), self.parse_number('0.0015')],
                            [self.parse_number('100000'), self.parse_number('0.0010')],
                            [self.parse_number('250000'), self.parse_number('0.0008')],
                            [self.parse_number('500000'), self.parse_number('0.0006')],
                            [self.parse_number('1000000'), self.parse_number('0.0005')],
                            [self.parse_number('2500000'), self.parse_number('0.0004')],
                            [self.parse_number('5000000'), self.parse_number('0.0004')],
                            [self.parse_number('10000000'), self.parse_number('0.0003')],
                            [self.parse_number('25000000'), self.parse_number('0.0003')],
                        ],
                    },
                },
            },
            'requiredCredentials': {
                'apiKey': True,
                'secret': True,
            },
            'exceptions': {
                'exact': {
                    '101': ExchangeError,  # Unknown error. Operation may or may not have succeeded.
                    '102': BadRequest,  # Invalid JSON.
                    '103': RateLimitExceeded,  # You have been rate limited. Please observe the Bitvavo-Ratelimit-AllowAt header to see when you can send requests again. Failure to respect self limit will result in an IP ban. The default value is 1000 weighted requests per minute. Please contact support if you wish to increase self limit.
                    '104': RateLimitExceeded,  # You have been rate limited by the number of new orders. The default value is 100 new orders per second or 100.000 new orders per day. Please update existing orders instead of cancelling and creating orders. Please contact support if you wish to increase self limit.
                    '105': PermissionDenied,  # Your IP or API key has been banned for not respecting the rate limit. The ban expires at ${expiryInMs}.
                    '107': ExchangeNotAvailable,  # The matching engine is overloaded. Please wait 500ms and resubmit your order.
                    '108': ExchangeNotAvailable,  # The matching engine could not process your order in time. Please consider increasing the access window or resubmit your order.
                    '109': ExchangeNotAvailable,  # The matching engine did not respond in time. Operation may or may not have succeeded.
                    '110': BadRequest,  # Invalid endpoint. Please check url and HTTP method.
                    '200': BadRequest,  # ${param} url parameter is not supported. Please note that parameters are case-sensitive and use body parameters for PUT and POST requests.
                    '201': BadRequest,  # ${param} body parameter is not supported. Please note that parameters are case-sensitive and use url parameters for GET and DELETE requests.
                    '202': BadRequest,  # ${param} order parameter is not supported. Please note that certain parameters are only allowed for market or limit orders.
                    '203': BadSymbol,  # {"errorCode":203,"error":"symbol parameter is required."}
                    '204': BadRequest,  # ${param} parameter is not supported.
                    '205': BadRequest,  # ${param} parameter is invalid.
                    '206': BadRequest,  # Use either ${paramA} or ${paramB}. The usage of both parameters at the same time is not supported.
                    '210': InvalidOrder,  # Amount exceeds the maximum allowed amount(1000000000).
                    '211': InvalidOrder,  # Price exceeds the maximum allowed amount(100000000000).
                    '212': InvalidOrder,  # Amount is below the minimum allowed amount for self asset.
                    '213': InvalidOrder,  # Price is below the minimum allowed amount(0.000000000000001).
                    '214': InvalidOrder,  # Price is too detailed
                    '215': InvalidOrder,  # Price is too detailed. A maximum of 15 digits behind the decimal point are allowed.
                    '216': InsufficientFunds,  # {"errorCode":216,"error":"You do not have sufficient balance to complete self operation."}
                    '217': InvalidOrder,  # {"errorCode":217,"error":"Minimum order size in quote currency is 5 EUR or 0.001 BTC."}
                    '230': ExchangeError,  # The order is rejected by the matching engine.
                    '231': ExchangeError,  # The order is rejected by the matching engine. TimeInForce must be GTC when markets are paused.
                    '232': BadRequest,  # You must change at least one of amount, amountRemaining, price, timeInForce, selfTradePrevention or postOnly.
                    '233': InvalidOrder,  # {"errorCode":233,"error":"Order must be active(status new or partiallyFilled) to allow updating/cancelling."}
                    '234': InvalidOrder,  # Market orders cannot be updated.
                    '235': ExchangeError,  # You can only have 100 open orders on each book.
                    '236': BadRequest,  # You can only update amount or amountRemaining, not both.
                    '240': OrderNotFound,  # {"errorCode":240,"error":"No order found. Please be aware that simultaneously updating the same order may return self error."}
                    '300': AuthenticationError,  # Authentication is required for self endpoint.
                    '301': AuthenticationError,  # {"errorCode":301,"error":"API Key must be of length 64."}
                    '302': AuthenticationError,  # Timestamp is invalid. This must be a timestamp in ms. See Bitvavo-Access-Timestamp header or timestamp parameter for websocket.
                    '303': AuthenticationError,  # Window must be between 100 and 60000 ms.
                    '304': AuthenticationError,  # Request was not received within acceptable window(default 30s, or custom with Bitvavo-Access-Window header) of Bitvavo-Access-Timestamp header(or timestamp parameter for websocket).
                    # '304': AuthenticationError,  # Authentication is required for self endpoint.
                    '305': AuthenticationError,  # {"errorCode":305,"error":"No active API key found."}
                    '306': AuthenticationError,  # No active API key found. Please ensure that you have confirmed the API key by e-mail.
                    '307': PermissionDenied,  # This key does not allow access from self IP.
                    '308': AuthenticationError,  # {"errorCode":308,"error":"The signature length is invalid(HMAC-SHA256 should return a 64 length hexadecimal string)."}
                    '309': AuthenticationError,  # {"errorCode":309,"error":"The signature is invalid."}
                    '310': PermissionDenied,  # This key does not allow trading actions.
                    '311': PermissionDenied,  # This key does not allow showing account information.
                    '312': PermissionDenied,  # This key does not allow withdrawal of funds.
                    '315': BadRequest,  # Websocket connections may not be used in a browser. Please use REST requests for self.
                    '317': AccountSuspended,  # This account is locked. Please contact support.
                    '400': ExchangeError,  # Unknown error. Please contact support with a copy of your request.
                    '401': ExchangeError,  # Deposits for self asset are not available at self time.
                    '402': PermissionDenied,  # You need to verify your identitiy before you can deposit and withdraw digital assets.
                    '403': PermissionDenied,  # You need to verify your phone number before you can deposit and withdraw digital assets.
                    '404': OnMaintenance,  # Could not complete self operation, because our node cannot be reached. Possibly under maintenance.
                    '405': ExchangeError,  # You cannot withdraw digital assets during a cooldown period. This is the result of newly added bank accounts.
                    '406': BadRequest,  # {"errorCode":406,"error":"Your withdrawal is too small."}
                    '407': ExchangeError,  # Internal transfer is not possible.
                    '408': InsufficientFunds,  # {"errorCode":408,"error":"You do not have sufficient balance to complete self operation."}
                    '409': InvalidAddress,  # {"errorCode":409,"error":"This is not a verified bank account."}
                    '410': ExchangeError,  # Withdrawals for self asset are not available at self time.
                    '411': BadRequest,  # You can not transfer assets to yourself.
                    '412': InvalidAddress,  # {"errorCode":412,"error":"eth_address_invalid."}
                    '413': InvalidAddress,  # This address violates the whitelist.
                    '414': ExchangeError,  # You cannot withdraw assets within 2 minutes of logging in.
                },
                'broad': {
                    'start parameter is invalid': BadRequest,  # {"errorCode":205,"error":"start parameter is invalid."}
                    'symbol parameter is invalid': BadSymbol,  # {"errorCode":205,"error":"symbol parameter is invalid."}
                    'amount parameter is invalid': InvalidOrder,  # {"errorCode":205,"error":"amount parameter is invalid."}
                    'orderId parameter is invalid': InvalidOrder,  # {"errorCode":205,"error":"orderId parameter is invalid."}
                },
            },
            'options': {
                'BITVAVO-ACCESS-WINDOW': 10000,  # default 10 sec
                'fetchCurrencies': {
                    'expires': 1000,  # 1 second
                },
            },
            'precisionMode': SIGNIFICANT_DIGITS,
            'commonCurrencies': {
                'MIOTA': 'IOTA',  # https://github.com/ccxt/ccxt/issues/7487
            },
        })

    def currency_to_precision(self, code, fee, networkCode=None):
        return self.decimal_to_precision(fee, 0, self.currencies[code]['precision'], DECIMAL_PLACES)

    def amount_to_precision(self, symbol, amount):
        # https://docs.bitfinex.com/docs/introduction#amount-precision
        # The amount field allows up to 8 decimals.
        # Anything exceeding self will be rounded to the 8th decimal.
        return self.decimal_to_precision(amount, TRUNCATE, self.markets[symbol]['precision']['amount'], DECIMAL_PLACES)

    def price_to_precision(self, symbol, price):
        price = self.decimal_to_precision(price, ROUND, self.markets[symbol]['precision']['price'], self.precisionMode)
        # https://docs.bitfinex.com/docs/introduction#price-precision
        # The precision level of all trading prices is based on significant figures.
        # All pairs on Bitfinex use up to 5 significant digits and up to 8 decimals(e.g. 1.2345, 123.45, 1234.5, 0.00012345).
        # Prices submit with a precision larger than 5 will be cut by the API.
        return self.decimal_to_precision(price, TRUNCATE, 8, DECIMAL_PLACES)

    def fetch_time(self, params={}):
        """
        fetches the current integer timestamp in milliseconds from the exchange server
        :param dict params: extra parameters specific to the bitvavo api endpoint
        :returns int: the current integer timestamp in milliseconds from the exchange server
        """
        response = self.publicGetTime(params)
        #
        #     {"time": 1590379519148}
        #
        return self.safe_integer(response, 'time')

    def fetch_markets(self, params={}):
        """
        retrieves data on all markets for bitvavo
        :param dict params: extra parameters specific to the exchange api endpoint
        :returns [dict]: an array of objects representing market data
        """
        response = self.publicGetMarkets(params)
        currencies = self.fetch_currencies_from_cache(params)
        currenciesById = self.index_by(currencies, 'symbol')
        #
        #     [
        #         {
        #             "market":"ADA-BTC",
        #             "status":"trading",  # "trading" "halted" "auction"
        #             "base":"ADA",
        #             "quote":"BTC",
        #             "pricePrecision":5,
        #             "minOrderInBaseAsset":"100",
        #             "minOrderInQuoteAsset":"0.001",
        #             "orderTypes": ["market", "limit"]
        #         }
        #     ]
        #
        result = []
        for i in range(0, len(response)):
            market = response[i]
            id = self.safe_string(market, 'market')
            baseId = self.safe_string(market, 'base')
            quoteId = self.safe_string(market, 'quote')
            base = self.safe_currency_code(baseId)
            quote = self.safe_currency_code(quoteId)
            status = self.safe_string(market, 'status')
            baseCurrency = self.safe_value(currenciesById, baseId)
            result.append({
                'id': id,
                'symbol': base + '/' + quote,
                'base': base,
                'quote': quote,
                'settle': None,
                'baseId': baseId,
                'quoteId': quoteId,
                'settleId': None,
                'type': 'spot',
                'spot': True,
                'margin': False,
                'swap': False,
                'future': False,
                'option': False,
                'active': (status == 'trading'),
                'contract': False,
                'linear': None,
                'inverse': None,
                'contractSize': None,
                'expiry': None,
                'expiryDatetime': None,
                'strike': None,
                'optionType': None,
                'precision': {
                    'amount': self.safe_integer(baseCurrency, 'decimals', 8),
                    'price': self.safe_integer(market, 'pricePrecision'),
                },
                'limits': {
                    'leverage': {
                        'min': None,
                        'max': None,
                    },
                    'amount': {
                        'min': self.safe_number(market, 'minOrderInBaseAsset'),
                        'max': None,
                    },
                    'price': {
                        'min': None,
                        'max': None,
                    },
                    'cost': {
                        'min': self.safe_number(market, 'minOrderInQuoteAsset'),
                        'max': None,
                    },
                },
                'info': market,
            })
        return result

    def fetch_currencies_from_cache(self, params={}):
        # self method is now redundant
        # currencies are now fetched before markets
        options = self.safe_value(self.options, 'fetchCurrencies', {})
        timestamp = self.safe_integer(options, 'timestamp')
        expires = self.safe_integer(options, 'expires', 1000)
        now = self.milliseconds()
        if (timestamp is None) or ((now - timestamp) > expires):
            response = self.publicGetAssets(params)
            self.options['fetchCurrencies'] = self.extend(options, {
                'response': response,
                'timestamp': now,
            })
        return self.safe_value(self.options['fetchCurrencies'], 'response')

    def fetch_currencies(self, params={}):
        """
        fetches all available currencies on an exchange
        :param dict params: extra parameters specific to the bitvavo api endpoint
        :returns dict: an associative dictionary of currencies
        """
        response = self.fetch_currencies_from_cache(params)
        #
        #     [
        #         {
        #             "symbol":"ADA",
        #             "name":"Cardano",
        #             "decimals":6,
        #             "depositFee":"0",
        #             "depositConfirmations":15,
        #             "depositStatus":"OK",  # "OK", "MAINTENANCE", "DELISTED"
        #             "withdrawalFee":"0.2",
        #             "withdrawalMinAmount":"0.2",
        #             "withdrawalStatus":"OK",  # "OK", "MAINTENANCE", "DELISTED"
        #             "networks": ["Mainnet"],  # "ETH", "NEO", "ONT", "SEPA", "VET"
        #             "message":"",
        #         },
        #     ]
        #
        result = {}
        for i in range(0, len(response)):
            currency = response[i]
            id = self.safe_string(currency, 'symbol')
            code = self.safe_currency_code(id)
            depositStatus = self.safe_value(currency, 'depositStatus')
            deposit = (depositStatus == 'OK')
            withdrawalStatus = self.safe_value(currency, 'withdrawalStatus')
            withdrawal = (withdrawalStatus == 'OK')
            active = deposit and withdrawal
            name = self.safe_string(currency, 'name')
            result[code] = {
                'id': id,
                'info': currency,
                'code': code,
                'name': name,
                'active': active,
                'deposit': deposit,
                'withdraw': withdrawal,
                'fee': self.safe_number(currency, 'withdrawalFee'),
                'precision': self.safe_integer(currency, 'decimals', 8),
                'limits': {
                    'amount': {
                        'min': None,
                        'max': None,
                    },
                    'withdraw': {
                        'min': self.safe_number(currency, 'withdrawalMinAmount'),
                        'max': None,
                    },
                },
            }
        return result

    def fetch_ticker(self, symbol, params={}):
        """
        fetches a price ticker, a statistical calculation with the information calculated over the past 24 hours for a specific market
        :param str symbol: unified symbol of the market to fetch the ticker for
        :param dict params: extra parameters specific to the bitvavo api endpoint
        :returns dict: a `ticker structure <https://docs.ccxt.com/#/?id=ticker-structure>`
        """
        self.load_markets()
        market = self.market(symbol)
        request = {
            'market': market['id'],
        }
        response = self.publicGetTicker24h(self.extend(request, params))
        #
        #     {
        #         "market":"ETH-BTC",
        #         "open":"0.022578",
        #         "high":"0.023019",
        #         "low":"0.022573",
        #         "last":"0.023019",
        #         "volume":"25.16366324",
        #         "volumeQuote":"0.57333305",
        #         "bid":"0.023039",
        #         "bidSize":"0.53500578",
        #         "ask":"0.023041",
        #         "askSize":"0.47859202",
        #         "timestamp":1590381666900
        #     }
        #
        return self.parse_ticker(response, market)

    def parse_ticker(self, ticker, market=None):
        #
        # fetchTicker
        #
        #     {
        #         "market":"ETH-BTC",
        #         "open":"0.022578",
        #         "high":"0.023019",
        #         "low":"0.022573",
        #         "last":"0.023019",
        #         "volume":"25.16366324",
        #         "volumeQuote":"0.57333305",
        #         "bid":"0.023039",
        #         "bidSize":"0.53500578",
        #         "ask":"0.023041",
        #         "askSize":"0.47859202",
        #         "timestamp":1590381666900
        #     }
        #
        marketId = self.safe_string(ticker, 'market')
        symbol = self.safe_symbol(marketId, market, '-')
        timestamp = self.safe_integer(ticker, 'timestamp')
        last = self.safe_string(ticker, 'last')
        baseVolume = self.safe_string(ticker, 'volume')
        quoteVolume = self.safe_string(ticker, 'volumeQuote')
        open = self.safe_string(ticker, 'open')
        return self.safe_ticker({
            'symbol': symbol,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'high': self.safe_string(ticker, 'high'),
            'low': self.safe_string(ticker, 'low'),
            'bid': self.safe_string(ticker, 'bid'),
            'bidVolume': self.safe_string(ticker, 'bidSize'),
            'ask': self.safe_string(ticker, 'ask'),
            'askVolume': self.safe_string(ticker, 'askSize'),
            'vwap': None,
            'open': open,
            'close': last,
            'last': last,
            'previousClose': None,  # previous day close
            'change': None,
            'percentage': None,
            'average': None,
            'baseVolume': baseVolume,
            'quoteVolume': quoteVolume,
            'info': ticker,
        }, market)

    def fetch_tickers(self, symbols=None, params={}):
        """
        fetches price tickers for multiple markets, statistical calculations with the information calculated over the past 24 hours each market
        :param [str]|None symbols: unified symbols of the markets to fetch the ticker for, all market tickers are returned if not assigned
        :param dict params: extra parameters specific to the bitvavo api endpoint
        :returns dict: a dictionary of `ticker structures <https://docs.ccxt.com/#/?id=ticker-structure>`
        """
        self.load_markets()
        response = self.publicGetTicker24h(params)
        #
        #     [
        #         {
        #             "market":"ADA-BTC",
        #             "open":"0.0000059595",
        #             "high":"0.0000059765",
        #             "low":"0.0000059595",
        #             "last":"0.0000059765",
        #             "volume":"2923.172",
        #             "volumeQuote":"0.01743483",
        #             "bid":"0.0000059515",
        #             "bidSize":"1117.630919",
        #             "ask":"0.0000059585",
        #             "askSize":"809.999739",
        #             "timestamp":1590382266324
        #         }
        #     ]
        #
        return self.parse_tickers(response, symbols)

    def fetch_trades(self, symbol, since=None, limit=None, params={}):
        """
        get the list of most recent trades for a particular symbol
        :param str symbol: unified symbol of the market to fetch trades for
        :param int|None since: timestamp in ms of the earliest trade to fetch
        :param int|None limit: the maximum amount of trades to fetch
        :param dict params: extra parameters specific to the bitvavo api endpoint
        :returns [dict]: a list of `trade structures <https://docs.ccxt.com/en/latest/manual.html?#public-trades>`
        """
        self.load_markets()
        market = self.market(symbol)
        request = {
            'market': market['id'],
            # 'limit': 500,  # default 500, max 1000
            # 'start': since,
            # 'end': self.milliseconds(),
            # 'tradeIdFrom': '57b1159b-6bf5-4cde-9e2c-6bd6a5678baf',
            # 'tradeIdTo': '57b1159b-6bf5-4cde-9e2c-6bd6a5678baf',
        }
        if limit is not None:
            request['limit'] = limit
        if since is not None:
            request['start'] = since
        response = self.publicGetMarketTrades(self.extend(request, params))
        #
        #     [
        #         {
        #             "id":"94154c98-6e8b-4e33-92a8-74e33fc05650",
        #             "timestamp":1590382761859,
        #             "amount":"0.06026079",
        #             "price":"8095.3",
        #             "side":"buy"
        #         }
        #     ]
        #
        return self.parse_trades(response, market, since, limit)

    def parse_trade(self, trade, market=None):
        #
        # fetchTrades(public)
        #
        #     {
        #         "id":"94154c98-6e8b-4e33-92a8-74e33fc05650",
        #         "timestamp":1590382761859,
        #         "amount":"0.06026079",
        #         "price":"8095.3",
        #         "side":"buy"
        #     }
        #
        # createOrder, fetchOpenOrders, fetchOrders, editOrder(private)
        #
        #     {
        #         "id":"b0c86aa5-6ed3-4a2d-ba3a-be9a964220f4",
        #         "timestamp":1590505649245,
        #         "amount":"0.249825",
        #         "price":"183.49",
        #         "taker":true,
        #         "fee":"0.12038925",
        #         "feeCurrency":"EUR",
        #         "settled":true
        #     }
        #
        # fetchMyTrades(private)
        #
        #     {
        #         "id":"b0c86aa5-6ed3-4a2d-ba3a-be9a964220f4",
        #         "orderId":"af76d6ce-9f7c-4006-b715-bb5d430652d0",
        #         "timestamp":1590505649245,
        #         "market":"ETH-EUR",
        #         "side":"sell",
        #         "amount":"0.249825",
        #         "price":"183.49",
        #         "taker":true,
        #         "fee":"0.12038925",
        #         "feeCurrency":"EUR",
        #         "settled":true
        #     }
        #
        # watchMyTrades(private)
        #
        #     {
        #         event: 'fill',
        #         timestamp: 1590964470132,
        #         market: 'ETH-EUR',
        #         orderId: '85d082e1-eda4-4209-9580-248281a29a9a',
        #         fillId: '861d2da5-aa93-475c-8d9a-dce431bd4211',
        #         side: 'sell',
        #         amount: '0.1',
        #         price: '211.46',
        #         taker: True,
        #         fee: '0.056',
        #         feeCurrency: 'EUR'
        #     }
        #
        priceString = self.safe_string(trade, 'price')
        amountString = self.safe_string(trade, 'amount')
        timestamp = self.safe_integer(trade, 'timestamp')
        side = self.safe_string(trade, 'side')
        id = self.safe_string_2(trade, 'id', 'fillId')
        marketId = self.safe_string(trade, 'market')
        symbol = self.safe_symbol(marketId, market, '-')
        taker = self.safe_value(trade, 'taker')
        takerOrMaker = None
        if taker is not None:
            takerOrMaker = 'taker' if taker else 'maker'
        feeCostString = self.safe_string(trade, 'fee')
        fee = None
        if feeCostString is not None:
            feeCurrencyId = self.safe_string(trade, 'feeCurrency')
            feeCurrencyCode = self.safe_currency_code(feeCurrencyId)
            fee = {
                'cost': feeCostString,
                'currency': feeCurrencyCode,
            }
        orderId = self.safe_string(trade, 'orderId')
        return self.safe_trade({
            'info': trade,
            'id': id,
            'symbol': symbol,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'order': orderId,
            'type': None,
            'side': side,
            'takerOrMaker': takerOrMaker,
            'price': priceString,
            'amount': amountString,
            'cost': None,
            'fee': fee,
        }, market)

    def fetch_trading_fees(self, params={}):
        """
        fetch the trading fees for multiple markets
        :param dict params: extra parameters specific to the bitvavo api endpoint
        :returns dict: a dictionary of `fee structures <https://docs.ccxt.com/#/?id=fee-structure>` indexed by market symbols
        """
        self.load_markets()
        response = self.privateGetAccount(params)
        #
        #     {
        #         "fees": {
        #           "taker": "0.0025",
        #           "maker": "0.0015",
        #           "volume": "10000.00"
        #         }
        #     }
        #
        fees = self.safe_value(response, 'fees')
        maker = self.safe_number(fees, 'maker')
        taker = self.safe_number(fees, 'taker')
        result = {}
        for i in range(0, len(self.symbols)):
            symbol = self.symbols[i]
            result[symbol] = {
                'info': response,
                'symbol': symbol,
                'maker': maker,
                'taker': taker,
                'percentage': True,
                'tierBased': True,
            }
        return result

    def fetch_order_book(self, symbol, limit=None, params={}):
        """
        fetches information on open orders with bid(buy) and ask(sell) prices, volumes and other data
        :param str symbol: unified symbol of the market to fetch the order book for
        :param int|None limit: the maximum amount of order book entries to return
        :param dict params: extra parameters specific to the bitvavo api endpoint
        :returns dict: A dictionary of `order book structures <https://docs.ccxt.com/#/?id=order-book-structure>` indexed by market symbols
        """
        self.load_markets()
        market = self.market(symbol)
        request = {
            'market': market['id'],
        }
        if limit is not None:
            request['depth'] = limit
        response = self.publicGetMarketBook(self.extend(request, params))
        #
        #     {
        #         "market":"BTC-EUR",
        #         "nonce":35883831,
        #         "bids":[
        #             ["8097.4","0.6229099"],
        #             ["8097.2","0.64151283"],
        #             ["8097.1","0.24966294"],
        #         ],
        #         "asks":[
        #             ["8097.5","1.36916911"],
        #             ["8098.8","0.33462248"],
        #             ["8099.3","1.12908646"],
        #         ]
        #     }
        #
        orderbook = self.parse_order_book(response, market['symbol'])
        orderbook['nonce'] = self.safe_integer(response, 'nonce')
        return orderbook

    def parse_ohlcv(self, ohlcv, market=None):
        #
        #     [
        #         1590383700000,
        #         "8088.5",
        #         "8088.5",
        #         "8088.5",
        #         "8088.5",
        #         "0.04788623"
        #     ]
        #
        return [
            self.safe_integer(ohlcv, 0),
            self.safe_number(ohlcv, 1),
            self.safe_number(ohlcv, 2),
            self.safe_number(ohlcv, 3),
            self.safe_number(ohlcv, 4),
            self.safe_number(ohlcv, 5),
        ]

    def fetch_ohlcv(self, symbol, timeframe='1m', since=None, limit=None, params={}):
        """
        fetches historical candlestick data containing the open, high, low, and close price, and the volume of a market
        :param str symbol: unified symbol of the market to fetch OHLCV data for
        :param str timeframe: the length of time each candle represents
        :param int|None since: timestamp in ms of the earliest candle to fetch
        :param int|None limit: the maximum amount of candles to fetch
        :param dict params: extra parameters specific to the bitvavo api endpoint
        :returns [[int]]: A list of candles ordered, open, high, low, close, volume
        """
        self.load_markets()
        market = self.market(symbol)
        request = {
            'market': market['id'],
            'interval': self.safe_string(self.timeframes, timeframe, timeframe),
            # 'limit': 1440,  # default 1440, max 1440
            # 'start': since,
            # 'end': self.milliseconds(),
        }
        if since is not None:
            # https://github.com/ccxt/ccxt/issues/9227
            duration = self.parse_timeframe(timeframe)
            request['start'] = since
            if limit is None:
                limit = 1440
            request['end'] = self.sum(since, limit * duration * 1000)
        if limit is not None:
            request['limit'] = limit  # default 1440, max 1440
        response = self.publicGetMarketCandles(self.extend(request, params))
        #
        #     [
        #         [1590383700000,"8088.5","8088.5","8088.5","8088.5","0.04788623"],
        #         [1590383580000,"8091.3","8091.5","8091.3","8091.5","0.04931221"],
        #         [1590383520000,"8090.3","8092.7","8090.3","8092.5","0.04001286"],
        #     ]
        #
        return self.parse_ohlcvs(response, market, timeframe, since, limit)

    def parse_balance(self, response):
        result = {
            'info': response,
            'timestamp': None,
            'datetime': None,
        }
        for i in range(0, len(response)):
            balance = response[i]
            currencyId = self.safe_string(balance, 'symbol')
            code = self.safe_currency_code(currencyId)
            account = self.account()
            account['free'] = self.safe_string(balance, 'available')
            account['used'] = self.safe_string(balance, 'inOrder')
            result[code] = account
        return self.safe_balance(result)

    def fetch_balance(self, params={}):
        """
        query for balance and get the amount of funds available for trading or funds locked in orders
        :param dict params: extra parameters specific to the bitvavo api endpoint
        :returns dict: a `balance structure <https://docs.ccxt.com/en/latest/manual.html?#balance-structure>`
        """
        self.load_markets()
        response = self.privateGetBalance(params)
        #
        #     [
        #         {
        #             "symbol": "BTC",
        #             "available": "1.57593193",
        #             "inOrder": "0.74832374"
        #         }
        #     ]
        #
        return self.parse_balance(response)

    def fetch_deposit_address(self, code, params={}):
        """
        fetch the deposit address for a currency associated with self account
        :param str code: unified currency code
        :param dict params: extra parameters specific to the bitvavo api endpoint
        :returns dict: an `address structure <https://docs.ccxt.com/#/?id=address-structure>`
        """
        self.load_markets()
        currency = self.currency(code)
        request = {
            'symbol': currency['id'],
        }
        response = self.privateGetDeposit(self.extend(request, params))
        #
        #     {
        #         "address": "0x449889e3234514c45d57f7c5a571feba0c7ad567",
        #         "paymentId": "10002653"
        #     }
        #
        address = self.safe_string(response, 'address')
        tag = self.safe_string(response, 'paymentId')
        self.check_address(address)
        return {
            'currency': code,
            'address': address,
            'tag': tag,
            'network': None,
            'info': response,
        }

    def create_order(self, symbol, type, side, amount, price=None, params={}):
        """
        create a trade order
        see https://docs.bitvavo.com/#tag/Orders/paths/~1order/post
        :param str symbol: unified symbol of the market to create an order in
        :param str type: 'market' or 'limit'
        :param str side: 'buy' or 'sell'
        :param float amount: how much of currency you want to trade in units of base currency
        :param float|None price: the price at which the order is to be fullfilled, in units of the quote currency, ignored in market orders
        :param dict params: extra parameters specific to the bitvavo api endpoint
        :param str|None params['timeInForce']: "GTC", "IOC", or "PO"
        :param float|None params['stopPrice']: The price at which a trigger order is triggered at
        :param float|None params['triggerPrice']: The price at which a trigger order is triggered at
        :param bool|None params['postOnly']: If True, the order will only be posted to the order book and not executed immediately
        :param float|None params['stopLossPrice']: The price at which a stop loss order is triggered at
        :param float|None params['takeProfitPrice']: The price at which a take profit order is triggered at
        :param str|None params['triggerType']: "price"
        :param str|None params['triggerReference']: "lastTrade", "bestBid", "bestAsk", "midPrice" Only for stop orders: Use self to determine which parameter will trigger the order
        :param str|None params['selfTradePrevention']: "decrementAndCancel", "cancelOldest", "cancelNewest", "cancelBoth"
        :param bool|None params['disableMarketProtection']: don't cancel if the next fill price is 10% worse than the best fill price
        :param bool|None params['responseRequired']: Set self to 'false' when only an acknowledgement of success or failure is required, self is faster.
        :returns dict: an `order structure <https://docs.ccxt.com/#/?id=order-structure>`
        """
        self.load_markets()
        market = self.market(symbol)
        request = {
            'market': market['id'],
            'side': side,
            'orderType': type,
        }
        isMarketOrder = (type == 'market') or (type == 'stopLoss') or (type == 'takeProfit')
        isLimitOrder = (type == 'limit') or (type == 'stopLossLimit') or (type == 'takeProfitLimit')
        timeInForce = self.safe_string(params, 'timeInForce')
        triggerPrice = self.safe_string_n(params, ['triggerPrice', 'stopPrice', 'triggerAmount'])
        postOnly = self.is_post_only(isMarketOrder, False, params)
        stopLossPrice = self.safe_value(params, 'stopLossPrice')  # trigger when price crosses from above to below self value
        takeProfitPrice = self.safe_value(params, 'takeProfitPrice')  # trigger when price crosses from below to above self value
        params = self.omit(params, ['timeInForce', 'triggerPrice', 'stopPrice', 'stopLossPrice', 'takeProfitPrice'])
        if isMarketOrder:
            cost = None
            if price is not None:
                priceString = self.number_to_string(price)
                amountString = self.number_to_string(amount)
                quoteAmount = Precise.string_mul(amountString, priceString)
                cost = self.parse_number(quoteAmount)
            else:
                cost = self.safe_number(params, 'cost')
            if cost is not None:
                precision = self.currency(market['quote'])['precision']
                request['amountQuote'] = self.decimal_to_precision(cost, TRUNCATE, precision, self.precisionMode)
            else:
                request['amount'] = self.amount_to_precision(symbol, amount)
            params = self.omit(params, ['cost'])
        elif isLimitOrder:
            request['price'] = self.price_to_precision(symbol, price)
            request['amount'] = self.amount_to_precision(symbol, amount)
        isTakeProfit = (takeProfitPrice is not None) or (type == 'takeProfit') or (type == 'takeProfitLimit')
        isStopLoss = (stopLossPrice is not None) or (triggerPrice is not None) and (not isTakeProfit) or (type == 'stopLoss') or (type == 'stopLossLimit')
        if isStopLoss:
            if stopLossPrice is not None:
                triggerPrice = stopLossPrice
            request['orderType'] = 'stopLoss' if isMarketOrder else 'stopLossLimit'
        elif isTakeProfit:
            if takeProfitPrice is not None:
                triggerPrice = takeProfitPrice
            request['orderType'] = 'takeProfit' if isMarketOrder else 'takeProfitLimit'
        if triggerPrice is not None:
            request['triggerAmount'] = self.price_to_precision(symbol, triggerPrice)
            request['triggerType'] = 'price'
            request['triggerReference'] = 'lastTrade'  # 'bestBid', 'bestAsk', 'midPrice'
        if (timeInForce is not None) and (timeInForce != 'PO'):
            request['timeInForce'] = timeInForce
        if postOnly:
            request['postOnly'] = True
        response = self.privatePostOrder(self.extend(request, params))
        #
        #      {
        #          "orderId":"dec6a640-5b4c-45bc-8d22-3b41c6716630",
        #          "market":"DOGE-EUR",
        #          "created":1654789135146,
        #          "updated":1654789135153,
        #          "status":"new",
        #          "side":"buy",
        #          "orderType":"stopLossLimit",
        #          "amount":"200",
        #          "amountRemaining":"200",
        #          "price":"0.07471",
        #          "triggerPrice":"0.0747",
        #          "triggerAmount":"0.0747",
        #          "triggerType":"price",
        #          "triggerReference":"lastTrade",
        #          "onHold":"14.98",
        #          "onHoldCurrency":"EUR",
        #          "filledAmount":"0",
        #          "filledAmountQuote":"0",
        #          "feePaid":"0",
        #          "feeCurrency":"EUR",
        #          "fills":[ # filled with market orders only
        #             {
        #                 "id":"b0c86aa5-6ed3-4a2d-ba3a-be9a964220f4",
        #                 "timestamp":1590505649245,
        #                 "amount":"0.249825",
        #                 "price":"183.49",
        #                 "taker":true,
        #                 "fee":"0.12038925",
        #                 "feeCurrency":"EUR",
        #                 "settled":true
        #             }
        #          ],
        #          "selfTradePrevention":"decrementAndCancel",
        #          "visible":true,
        #          "timeInForce":"GTC",
        #          "postOnly":false
        #      }
        #
        return self.parse_order(response, market)

    def edit_order(self, id, symbol, type, side, amount=None, price=None, params={}):
        self.load_markets()
        market = self.market(symbol)
        request = {}
        amountRemaining = self.safe_number(params, 'amountRemaining')
        params = self.omit(params, 'amountRemaining')
        if price is not None:
            request['price'] = self.price_to_precision(symbol, price)
        if amount is not None:
            request['amount'] = self.amount_to_precision(symbol, amount)
        if amountRemaining is not None:
            request['amountRemaining'] = self.amount_to_precision(symbol, amountRemaining)
        request = self.extend(request, params)
        if request:
            request['orderId'] = id
            request['market'] = market['id']
            response = self.privatePutOrder(self.extend(request, params))
            return self.parse_order(response, market)
        else:
            raise ArgumentsRequired(self.id + ' editOrder() requires an amount argument, or a price argument, or non-empty params')

    def cancel_order(self, id, symbol=None, params={}):
        """
        cancels an open order
        :param str id: order id
        :param str symbol: unified symbol of the market the order was made in
        :param dict params: extra parameters specific to the bitvavo api endpoint
        :returns dict: An `order structure <https://docs.ccxt.com/#/?id=order-structure>`
        """
        if symbol is None:
            raise ArgumentsRequired(self.id + ' cancelOrder() requires a symbol argument')
        self.load_markets()
        market = self.market(symbol)
        request = {
            'orderId': id,
            'market': market['id'],
        }
        response = self.privateDeleteOrder(self.extend(request, params))
        #
        #     {
        #         "orderId": "2e7ce7fc-44e2-4d80-a4a7-d079c4750b61"
        #     }
        #
        return self.parse_order(response, market)

    def cancel_all_orders(self, symbol=None, params={}):
        """
        cancel all open orders
        :param str|None symbol: unified market symbol, only orders in the market of self symbol are cancelled when symbol is not None
        :param dict params: extra parameters specific to the bitvavo api endpoint
        :returns [dict]: a list of `order structures <https://docs.ccxt.com/#/?id=order-structure>`
        """
        self.load_markets()
        request = {}
        market = None
        if symbol is not None:
            market = self.market(symbol)
            request['market'] = market['id']
        response = self.privateDeleteOrders(self.extend(request, params))
        #
        #     [
        #         {
        #             "orderId": "1be6d0df-d5dc-4b53-a250-3376f3b393e6"
        #         }
        #     ]
        #
        return self.parse_orders(response, market)

    def fetch_order(self, id, symbol=None, params={}):
        """
        fetches information on an order made by the user
        :param str symbol: unified symbol of the market the order was made in
        :param dict params: extra parameters specific to the bitvavo api endpoint
        :returns dict: An `order structure <https://docs.ccxt.com/#/?id=order-structure>`
        """
        if symbol is None:
            raise ArgumentsRequired(self.id + ' fetchOrder() requires a symbol argument')
        self.load_markets()
        market = self.market(symbol)
        request = {
            'orderId': id,
            'market': market['id'],
        }
        response = self.privateGetOrder(self.extend(request, params))
        #
        #     {
        #         "orderId":"af76d6ce-9f7c-4006-b715-bb5d430652d0",
        #         "market":"ETH-EUR",
        #         "created":1590505649241,
        #         "updated":1590505649241,
        #         "status":"filled",
        #         "side":"sell",
        #         "orderType":"market",
        #         "amount":"0.249825",
        #         "amountRemaining":"0",
        #         "onHold":"0",
        #         "onHoldCurrency":"ETH",
        #         "filledAmount":"0.249825",
        #         "filledAmountQuote":"45.84038925",
        #         "feePaid":"0.12038925",
        #         "feeCurrency":"EUR",
        #         "fills":[
        #             {
        #                 "id":"b0c86aa5-6ed3-4a2d-ba3a-be9a964220f4",
        #                 "timestamp":1590505649245,
        #                 "amount":"0.249825",
        #                 "price":"183.49",
        #                 "taker":true,
        #                 "fee":"0.12038925",
        #                 "feeCurrency":"EUR",
        #                 "settled":true
        #             }
        #         ],
        #         "selfTradePrevention":"decrementAndCancel",
        #         "visible":false,
        #         "disableMarketProtection":false
        #     }
        #
        return self.parse_order(response, market)

    def fetch_orders(self, symbol=None, since=None, limit=None, params={}):
        """
        fetches information on multiple orders made by the user
        :param str symbol: unified market symbol of the market orders were made in
        :param int|None since: the earliest time in ms to fetch orders for
        :param int|None limit: the maximum number of  orde structures to retrieve
        :param dict params: extra parameters specific to the bitvavo api endpoint
        :returns [dict]: a list of `order structures <https://docs.ccxt.com/#/?id=order-structure>`
        """
        if symbol is None:
            raise ArgumentsRequired(self.id + ' fetchOrders() requires a symbol argument')
        self.load_markets()
        market = self.market(symbol)
        request = {
            'market': market['id'],
            # 'limit': 500,
            # 'start': since,
            # 'end': self.milliseconds(),
            # 'orderIdFrom': 'af76d6ce-9f7c-4006-b715-bb5d430652d0',
            # 'orderIdTo': 'af76d6ce-9f7c-4006-b715-bb5d430652d0',
        }
        if since is not None:
            request['start'] = since
        if limit is not None:
            request['limit'] = limit  # default 500, max 1000
        response = self.privateGetOrders(self.extend(request, params))
        #
        #     [
        #         {
        #             "orderId":"af76d6ce-9f7c-4006-b715-bb5d430652d0",
        #             "market":"ETH-EUR",
        #             "created":1590505649241,
        #             "updated":1590505649241,
        #             "status":"filled",
        #             "side":"sell",
        #             "orderType":"market",
        #             "amount":"0.249825",
        #             "amountRemaining":"0",
        #             "onHold":"0",
        #             "onHoldCurrency":"ETH",
        #             "filledAmount":"0.249825",
        #             "filledAmountQuote":"45.84038925",
        #             "feePaid":"0.12038925",
        #             "feeCurrency":"EUR",
        #             "fills":[
        #                 {
        #                     "id":"b0c86aa5-6ed3-4a2d-ba3a-be9a964220f4",
        #                     "timestamp":1590505649245,
        #                     "amount":"0.249825",
        #                     "price":"183.49",
        #                     "taker":true,
        #                     "fee":"0.12038925",
        #                     "feeCurrency":"EUR",
        #                     "settled":true
        #                 }
        #             ],
        #             "selfTradePrevention":"decrementAndCancel",
        #             "visible":false,
        #             "disableMarketProtection":false
        #         }
        #     ]
        #
        return self.parse_orders(response, market, since, limit)

    def fetch_open_orders(self, symbol=None, since=None, limit=None, params={}):
        """
        fetch all unfilled currently open orders
        :param str|None symbol: unified market symbol
        :param int|None since: the earliest time in ms to fetch open orders for
        :param int|None limit: the maximum number of  open orders structures to retrieve
        :param dict params: extra parameters specific to the bitvavo api endpoint
        :returns [dict]: a list of `order structures <https://docs.ccxt.com/#/?id=order-structure>`
        """
        self.load_markets()
        request = {
            # 'market': market['id'],  # rate limit 25 without a market, 1 with market specified
        }
        market = None
        if symbol is not None:
            market = self.market(symbol)
            request['market'] = market['id']
        response = self.privateGetOrdersOpen(self.extend(request, params))
        #
        #     [
        #         {
        #             "orderId":"af76d6ce-9f7c-4006-b715-bb5d430652d0",
        #             "market":"ETH-EUR",
        #             "created":1590505649241,
        #             "updated":1590505649241,
        #             "status":"filled",
        #             "side":"sell",
        #             "orderType":"market",
        #             "amount":"0.249825",
        #             "amountRemaining":"0",
        #             "onHold":"0",
        #             "onHoldCurrency":"ETH",
        #             "filledAmount":"0.249825",
        #             "filledAmountQuote":"45.84038925",
        #             "feePaid":"0.12038925",
        #             "feeCurrency":"EUR",
        #             "fills":[
        #                 {
        #                     "id":"b0c86aa5-6ed3-4a2d-ba3a-be9a964220f4",
        #                     "timestamp":1590505649245,
        #                     "amount":"0.249825",
        #                     "price":"183.49",
        #                     "taker":true,
        #                     "fee":"0.12038925",
        #                     "feeCurrency":"EUR",
        #                     "settled":true
        #                 }
        #             ],
        #             "selfTradePrevention":"decrementAndCancel",
        #             "visible":false,
        #             "disableMarketProtection":false
        #         }
        #     ]
        #
        return self.parse_orders(response, market, since, limit)

    def parse_order_status(self, status):
        statuses = {
            'new': 'open',
            'canceled': 'canceled',
            'canceledAuction': 'canceled',
            'canceledSelfTradePrevention': 'canceled',
            'canceledIOC': 'canceled',
            'canceledFOK': 'canceled',
            'canceledMarketProtection': 'canceled',
            'canceledPostOnly': 'canceled',
            'filled': 'closed',
            'partiallyFilled': 'open',
            'expired': 'canceled',
            'rejected': 'canceled',
            'awaitingTrigger': 'open',  # https://github.com/ccxt/ccxt/issues/8489
        }
        return self.safe_string(statuses, status, status)

    def parse_order(self, order, market=None):
        #
        # cancelOrder, cancelAllOrders
        #
        #     {
        #         "orderId": "2e7ce7fc-44e2-4d80-a4a7-d079c4750b61"
        #     }
        #
        # createOrder, fetchOrder, fetchOpenOrders, fetchOrders, editOrder
        #
        #     {
        #         "orderId":"af76d6ce-9f7c-4006-b715-bb5d430652d0",
        #         "market":"ETH-EUR",
        #         "created":1590505649241,
        #         "updated":1590505649241,
        #         "status":"filled",
        #         "side":"sell",
        #         "orderType":"market",
        #         "amount":"0.249825",
        #         "amountRemaining":"0",
        #         "price": "183.49",  # limit orders only
        #         "onHold":"0",
        #         "onHoldCurrency":"ETH",
        #         "filledAmount":"0.249825",
        #         "filledAmountQuote":"45.84038925",
        #         "feePaid":"0.12038925",
        #         "feeCurrency":"EUR",
        #         "fills":[
        #             {
        #                 "id":"b0c86aa5-6ed3-4a2d-ba3a-be9a964220f4",
        #                 "timestamp":1590505649245,
        #                 "amount":"0.249825",
        #                 "price":"183.49",
        #                 "taker":true,
        #                 "fee":"0.12038925",
        #                 "feeCurrency":"EUR",
        #                 "settled":true
        #             }
        #         ],
        #         "selfTradePrevention":"decrementAndCancel",
        #         "visible":false,
        #         "disableMarketProtection":false
        #         "timeInForce": "GTC",
        #         "postOnly": True,
        #     }
        #
        id = self.safe_string(order, 'orderId')
        timestamp = self.safe_integer(order, 'created')
        marketId = self.safe_string(order, 'market')
        market = self.safe_market(marketId, market, '-')
        symbol = market['symbol']
        status = self.parse_order_status(self.safe_string(order, 'status'))
        side = self.safe_string(order, 'side')
        type = self.safe_string(order, 'orderType')
        price = self.safe_string(order, 'price')
        amount = self.safe_string(order, 'amount')
        remaining = self.safe_string(order, 'amountRemaining')
        filled = self.safe_string(order, 'filledAmount')
        cost = self.safe_string(order, 'filledAmountQuote')
        if cost is None:
            amountQuote = self.safe_string(order, 'amountQuote')
            amountQuoteRemaining = self.safe_string(order, 'amountQuoteRemaining')
            cost = Precise.string_sub(amountQuote, amountQuoteRemaining)
        fee = None
        feeCost = self.safe_number(order, 'feePaid')
        if feeCost is not None:
            feeCurrencyId = self.safe_string(order, 'feeCurrency')
            feeCurrencyCode = self.safe_currency_code(feeCurrencyId)
            fee = {
                'cost': feeCost,
                'currency': feeCurrencyCode,
            }
        rawTrades = self.safe_value(order, 'fills', [])
        timeInForce = self.safe_string(order, 'timeInForce')
        postOnly = self.safe_value(order, 'postOnly')
        # https://github.com/ccxt/ccxt/issues/8489
        stopPrice = self.safe_number(order, 'triggerPrice')
        return self.safe_order({
            'info': order,
            'id': id,
            'clientOrderId': None,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'lastTradeTimestamp': None,
            'symbol': symbol,
            'type': type,
            'timeInForce': timeInForce,
            'postOnly': postOnly,
            'side': side,
            'price': price,
            'stopPrice': stopPrice,
            'triggerPrice': stopPrice,
            'amount': amount,
            'cost': cost,
            'average': None,
            'filled': filled,
            'remaining': remaining,
            'status': status,
            'fee': fee,
            'trades': rawTrades,
        }, market)

    def fetch_my_trades(self, symbol=None, since=None, limit=None, params={}):
        """
        fetch all trades made by the user
        :param str|None symbol: unified market symbol
        :param int|None since: the earliest time in ms to fetch trades for
        :param int|None limit: the maximum number of trades structures to retrieve
        :param dict params: extra parameters specific to the bitvavo api endpoint
        :returns [dict]: a list of `trade structures <https://docs.ccxt.com/#/?id=trade-structure>`
        """
        if symbol is None:
            raise ArgumentsRequired(self.id + ' fetchMyTrades() requires a symbol argument')
        self.load_markets()
        market = self.market(symbol)
        request = {
            'market': market['id'],
            # 'limit': 500,
            # 'start': since,
            # 'end': self.milliseconds(),
            # 'tradeIdFrom': 'af76d6ce-9f7c-4006-b715-bb5d430652d0',
            # 'tradeIdTo': 'af76d6ce-9f7c-4006-b715-bb5d430652d0',
        }
        if since is not None:
            request['start'] = since
        if limit is not None:
            request['limit'] = limit  # default 500, max 1000
        response = self.privateGetTrades(self.extend(request, params))
        #
        #     [
        #         {
        #             "id":"b0c86aa5-6ed3-4a2d-ba3a-be9a964220f4",
        #             "orderId":"af76d6ce-9f7c-4006-b715-bb5d430652d0",
        #             "timestamp":1590505649245,
        #             "market":"ETH-EUR",
        #             "side":"sell",
        #             "amount":"0.249825",
        #             "price":"183.49",
        #             "taker":true,
        #             "fee":"0.12038925",
        #             "feeCurrency":"EUR",
        #             "settled":true
        #         }
        #     ]
        #
        return self.parse_trades(response, market, since, limit)

    def withdraw(self, code, amount, address, tag=None, params={}):
        """
        make a withdrawal
        :param str code: unified currency code
        :param float amount: the amount to withdraw
        :param str address: the address to withdraw to
        :param str|None tag:
        :param dict params: extra parameters specific to the bitvavo api endpoint
        :returns dict: a `transaction structure <https://docs.ccxt.com/#/?id=transaction-structure>`
        """
        tag, params = self.handle_withdraw_tag_and_params(tag, params)
        self.check_address(address)
        self.load_markets()
        currency = self.currency(code)
        request = {
            'symbol': currency['id'],
            'amount': self.currency_to_precision(code, amount),
            'address': address,  # address or IBAN
            # 'internal': False,  # transfer to another Bitvavo user address, no fees
            # 'addWithdrawalFee': False,  # True = add the fee on top, otherwise the fee is subtracted from the amount
        }
        if tag is not None:
            request['paymentId'] = tag
        response = self.privatePostWithdrawal(self.extend(request, params))
        #
        #     {
        #         "success": True,
        #         "symbol": "BTC",
        #         "amount": "1.5"
        #     }
        #
        return self.parse_transaction(response, currency)

    def fetch_withdrawals(self, code=None, since=None, limit=None, params={}):
        """
        fetch all withdrawals made from an account
        :param str|None code: unified currency code
        :param int|None since: the earliest time in ms to fetch withdrawals for
        :param int|None limit: the maximum number of withdrawals structures to retrieve
        :param dict params: extra parameters specific to the bitvavo api endpoint
        :returns [dict]: a list of `transaction structures <https://docs.ccxt.com/#/?id=transaction-structure>`
        """
        self.load_markets()
        request = {
            # 'symbol': currency['id'],
            # 'limit': 500,  # default 500, max 1000
            # 'start': since,
            # 'end': self.milliseconds(),
        }
        currency = None
        if code is not None:
            currency = self.currency(code)
            request['symbol'] = currency['id']
        if since is not None:
            request['start'] = since
        if limit is not None:
            request['limit'] = limit  # default 500, max 1000
        response = self.privateGetWithdrawalHistory(self.extend(request, params))
        #
        #     [
        #         {
        #             "timestamp":1590531212000,
        #             "symbol":"ETH",
        #             "amount":"0.091",
        #             "fee":"0.009",
        #             "status":"awaiting_bitvavo_inspection",
        #             "address":"0xe42b309f1eE9F0cbf7f54CcF3bc2159eBfA6735b",
        #             "paymentId": "10002653",
        #             "txId": "927b3ea50c5bb52c6854152d305dfa1e27fc01d10464cf10825d96d69d235eb3",
        #         }
        #     ]
        #
        return self.parse_transactions(response, currency, since, limit, {'type': 'withdrawal'})

    def fetch_deposits(self, code=None, since=None, limit=None, params={}):
        """
        fetch all deposits made to an account
        :param str|None code: unified currency code
        :param int|None since: the earliest time in ms to fetch deposits for
        :param int|None limit: the maximum number of deposits structures to retrieve
        :param dict params: extra parameters specific to the bitvavo api endpoint
        :returns [dict]: a list of `transaction structures <https://docs.ccxt.com/#/?id=transaction-structure>`
        """
        self.load_markets()
        request = {
            # 'symbol': currency['id'],
            # 'limit': 500,  # default 500, max 1000
            # 'start': since,
            # 'end': self.milliseconds(),
        }
        currency = None
        if code is not None:
            currency = self.currency(code)
            request['symbol'] = currency['id']
        if since is not None:
            request['start'] = since
        if limit is not None:
            request['limit'] = limit  # default 500, max 1000
        response = self.privateGetDepositHistory(self.extend(request, params))
        #
        #     [
        #         {
        #             "timestamp":1590492401000,
        #             "symbol":"ETH",
        #             "amount":"0.249825",
        #             "fee":"0",
        #             "status":"completed",
        #             "txId":"0x5167b473fd37811f9ef22364c3d54726a859ef9d98934b3a1e11d7baa8d2c2e2"
        #         }
        #     ]
        #
        return self.parse_transactions(response, currency, since, limit, {'type': 'deposit'})

    def parse_transaction_status(self, status):
        statuses = {
            'awaiting_processing': 'pending',
            'awaiting_email_confirmation': 'pending',
            'awaiting_bitvavo_inspection': 'pending',
            'approved': 'pending',
            'sending': 'pending',
            'in_mempool': 'pending',
            'processed': 'pending',
            'completed': 'ok',
            'canceled': 'canceled',
        }
        return self.safe_string(statuses, status, status)

    def parse_transaction(self, transaction, currency=None):
        #
        # withdraw
        #
        #     {
        #         "success": True,
        #         "symbol": "BTC",
        #         "amount": "1.5"
        #     }
        #
        # fetchWithdrawals
        #
        #     {
        #         "timestamp": 1542967486256,
        #         "symbol": "BTC",
        #         "amount": "0.99994",
        #         "address": "BitcoinAddress",
        #         "paymentId": "10002653",
        #         "txId": "927b3ea50c5bb52c6854152d305dfa1e27fc01d10464cf10825d96d69d235eb3",
        #         "fee": "0.00006",
        #         "status": "awaiting_processing"
        #     }
        #
        # fetchDeposits
        #
        #     {
        #         "timestamp":1590492401000,
        #         "symbol":"ETH",
        #         "amount":"0.249825",
        #         "fee":"0",
        #         "status":"completed",
        #         "txId":"0x5167b473fd37811f9ef22364c3d54726a859ef9d98934b3a1e11d7baa8d2c2e2"
        #     }
        #
        id = None
        timestamp = self.safe_integer(transaction, 'timestamp')
        currencyId = self.safe_string(transaction, 'symbol')
        code = self.safe_currency_code(currencyId, currency)
        status = self.parse_transaction_status(self.safe_string(transaction, 'status'))
        amount = self.safe_number(transaction, 'amount')
        address = self.safe_string(transaction, 'address')
        txid = self.safe_string(transaction, 'txId')
        fee = None
        feeCost = self.safe_number(transaction, 'fee')
        if feeCost is not None:
            fee = {
                'cost': feeCost,
                'currency': code,
            }
        type = None
        if ('success' in transaction) or ('address' in transaction):
            type = 'withdrawal'
        else:
            type = 'deposit'
        tag = self.safe_string(transaction, 'paymentId')
        return {
            'info': transaction,
            'id': id,
            'txid': txid,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'addressFrom': None,
            'address': address,
            'addressTo': address,
            'tagFrom': None,
            'tag': tag,
            'tagTo': tag,
            'type': type,
            'amount': amount,
            'currency': code,
            'status': status,
            'updated': None,
            'fee': fee,
        }

    def sign(self, path, api='public', method='GET', params={}, headers=None, body=None):
        query = self.omit(params, self.extract_params(path))
        url = '/' + self.version + '/' + self.implode_params(path, params)
        getOrDelete = (method == 'GET') or (method == 'DELETE')
        if getOrDelete:
            if query:
                url += '?' + self.urlencode(query)
        if api == 'private':
            self.check_required_credentials()
            payload = ''
            if not getOrDelete:
                if query:
                    body = self.json(query)
                    payload = body
            timestamp = str(self.milliseconds())
            auth = timestamp + method + url + payload
            signature = self.hmac(self.encode(auth), self.encode(self.secret))
            accessWindow = self.safe_string(self.options, 'BITVAVO-ACCESS-WINDOW', '10000')
            headers = {
                'BITVAVO-ACCESS-KEY': self.apiKey,
                'BITVAVO-ACCESS-SIGNATURE': signature,
                'BITVAVO-ACCESS-TIMESTAMP': timestamp,
                'BITVAVO-ACCESS-WINDOW': accessWindow,
            }
            if not getOrDelete:
                headers['Content-Type'] = 'application/json'
        url = self.urls['api'][api] + url
        return {'url': url, 'method': method, 'body': body, 'headers': headers}

    def handle_errors(self, httpCode, reason, url, method, headers, body, response, requestHeaders, requestBody):
        if response is None:
            return  # fallback to default error handler
        #
        #     {"errorCode":308,"error":"The signature length is invalid(HMAC-SHA256 should return a 64 length hexadecimal string)."}
        #     {"errorCode":203,"error":"symbol parameter is required."}
        #     {"errorCode":205,"error":"symbol parameter is invalid."}
        #
        errorCode = self.safe_string(response, 'errorCode')
        error = self.safe_string(response, 'error')
        if errorCode is not None:
            feedback = self.id + ' ' + body
            self.throw_broadly_matched_exception(self.exceptions['broad'], error, feedback)
            self.throw_exactly_matched_exception(self.exceptions['exact'], errorCode, feedback)
            raise ExchangeError(feedback)  # unknown message

    def calculate_rate_limiter_cost(self, api, method, path, params, config={}, context={}):
        if ('noMarket' in config) and not ('market' in params):
            return config['noMarket']
        return self.safe_value(config, 'cost', 1)
