"""Misc utils to download chemRxiv dump"""
import json
import logging
import os
import sys
from typing import Optional

from tqdm import tqdm

from .chemrxiv_api import ChemrxivAPI

logging.basicConfig(stream=sys.stdout, level=logging.DEBUG)
logger = logging.getLogger(__name__)


def get_author(author_list: list) -> str:
    """Parse ChemRxiv dump entry to extract author list

    Args:
        author_list (list): List of dicts, one per author.

    Returns:
        str: ;-concatenated author list.
    """

    return '; '.join([a['full_name'] for a in author_list])


def get_date(date: str) -> str:
    return date.split('T')[0]


def parse_dump(source_path: str, target_path: str) -> None:
    """
    Parses the dump as generated by the chemrXiv API and this repo:
    https://github.com/cthoyt/chemrxiv-summarize
    into a format that is equal to that of biorXiv and medRxiv.

    NOTE: This is a lazy parser trying to store all data in memory.

    Args:
        path (str): Path to the source dump
    """

    dump = []
    # Read source dump
    for file_name in os.listdir(source_path):

        if not file_name.endswith('.json'):
            continue
        filepath = os.path.join(source_path, file_name)
        with open(filepath, 'r') as f:
            source_paper = json.load(f)

        target_paper = {
            'title': source_paper['title'],
            'doi': source_paper['doi'],
            'authors': get_author(source_paper['authors']),
            'abstract': source_paper['description'],
            'date': get_date(source_paper['published_date']),
            'journal': 'chemRxiv',
            'link': source_paper['url_public_html'],
        }
        dump.append(target_paper)
        os.remove(filepath)
    # Write dump
    with open(target_path, 'w') as f:
        for idx, target_paper in enumerate(dump):
            if idx > 0:
                f.write(os.linesep)
            f.write(json.dumps(target_paper))
    logger.info('Done, shutting down')


def download_full(save_dir: str, api: Optional[ChemrxivAPI] = None) -> None:
    if api is None:
        api = ChemrxivAPI()

    os.makedirs(save_dir, exist_ok=True)
    for preprint in tqdm(api.all_preprints()):
        preprint_id = preprint['id']

        path = os.path.join(save_dir, f'{preprint_id}.json')
        if os.path.exists(path):
            continue

        preprint = api.preprint(preprint_id)
        with open(path, 'w') as file:
            json.dump(preprint, file, indent=2)
