"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const s3_assets = require("@aws-cdk/aws-s3-assets");
/**
 * Represents an OpenAPI definition asset.
 * @experimental
 */
class ApiDefinition {
    /**
     * Creates an API definition from a specification file in an S3 bucket
     * @experimental
     */
    static fromBucket(bucket, key, objectVersion) {
        return new S3ApiDefinition(bucket, key, objectVersion);
    }
    /**
     * Create an API definition from an inline object. The inline object must follow the
     * schema of OpenAPI 2.0 or OpenAPI 3.0
     *
     * @example
     *   ApiDefinition.fromInline({
     *     openapi: '3.0.2',
     *     paths: {
     *       '/pets': {
     *         get: {
     *           'responses': {
     *             200: {
     *               content: {
     *                 'application/json': {
     *                   schema: {
     *                     $ref: '#/components/schemas/Empty',
     *                   },
     *                 },
     *               },
     *             },
     *           },
     *           'x-amazon-apigateway-integration': {
     *             responses: {
     *               default: {
     *                 statusCode: '200',
     *               },
     *             },
     *             requestTemplates: {
     *               'application/json': '{"statusCode": 200}',
     *             },
     *             passthroughBehavior: 'when_no_match',
     *             type: 'mock',
     *           },
     *         },
     *       },
     *     },
     *     components: {
     *       schemas: {
     *         Empty: {
     *           title: 'Empty Schema',
     *           type: 'object',
     *         },
     *       },
     *     },
     *   });
     */
    static fromInline(definition) {
        return new InlineApiDefinition(definition);
    }
    /**
     * Loads the API specification from a local disk asset.
     * @experimental
     */
    static fromAsset(file, options) {
        return new AssetApiDefinition(file, options);
    }
}
exports.ApiDefinition = ApiDefinition;
/**
 * OpenAPI specification from an S3 archive.
 * @experimental
 */
class S3ApiDefinition extends ApiDefinition {
    constructor(bucket, key, objectVersion) {
        super();
        this.key = key;
        this.objectVersion = objectVersion;
        if (!bucket.bucketName) {
            throw new Error('bucketName is undefined for the provided bucket');
        }
        this.bucketName = bucket.bucketName;
    }
    bind(_scope) {
        return {
            s3Location: {
                bucket: this.bucketName,
                key: this.key,
                version: this.objectVersion,
            },
        };
    }
}
exports.S3ApiDefinition = S3ApiDefinition;
/**
 * OpenAPI specification from an inline JSON object.
 * @experimental
 */
class InlineApiDefinition extends ApiDefinition {
    constructor(definition) {
        super();
        this.definition = definition;
        if (typeof (definition) !== 'object') {
            throw new Error('definition should be of type object');
        }
        if (Object.keys(definition).length === 0) {
            throw new Error('JSON definition cannot be empty');
        }
    }
    bind(_scope) {
        return {
            inlineDefinition: this.definition,
        };
    }
}
exports.InlineApiDefinition = InlineApiDefinition;
/**
 * OpenAPI specification from a local file.
 * @experimental
 */
class AssetApiDefinition extends ApiDefinition {
    constructor(path, options = {}) {
        super();
        this.path = path;
        this.options = options;
    }
    bind(scope) {
        // If the same AssetAPIDefinition is used multiple times, retain only the first instantiation.
        if (this.asset === undefined) {
            this.asset = new s3_assets.Asset(scope, 'APIDefinition', {
                path: this.path,
                ...this.options,
            });
        }
        if (this.asset.isZipArchive) {
            throw new Error(`Asset cannot be a .zip file or a directory (${this.path})`);
        }
        return {
            s3Location: {
                bucket: this.asset.s3BucketName,
                key: this.asset.s3ObjectKey,
            },
        };
    }
}
exports.AssetApiDefinition = AssetApiDefinition;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXBpLWRlZmluaXRpb24uanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJhcGktZGVmaW5pdGlvbi50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUNBLG9EQUFvRDtBQUdwRDs7O0dBR0c7QUFDSCxNQUFzQixhQUFhO0lBQ2pDOzs7T0FHRztJQUNJLE1BQU0sQ0FBQyxVQUFVLENBQUMsTUFBa0IsRUFBRSxHQUFXLEVBQUUsYUFBc0I7UUFDOUUsT0FBTyxJQUFJLGVBQWUsQ0FBQyxNQUFNLEVBQUUsR0FBRyxFQUFFLGFBQWEsQ0FBQyxDQUFDO0lBQ3pELENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O09BNkNHO0lBQ0ksTUFBTSxDQUFDLFVBQVUsQ0FBQyxVQUFlO1FBQ3RDLE9BQU8sSUFBSSxtQkFBbUIsQ0FBQyxVQUFVLENBQUMsQ0FBQztJQUM3QyxDQUFDO0lBRUQ7OztPQUdHO0lBQ0ksTUFBTSxDQUFDLFNBQVMsQ0FBQyxJQUFZLEVBQUUsT0FBZ0M7UUFDcEUsT0FBTyxJQUFJLGtCQUFrQixDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsQ0FBQztJQUMvQyxDQUFDO0NBVUY7QUEzRUQsc0NBMkVDO0FBc0NEOzs7R0FHRztBQUNILE1BQWEsZUFBZ0IsU0FBUSxhQUFhO0lBR2hELFlBQVksTUFBa0IsRUFBVSxHQUFXLEVBQVUsYUFBc0I7UUFDakYsS0FBSyxFQUFFLENBQUM7UUFEOEIsUUFBRyxHQUFILEdBQUcsQ0FBUTtRQUFVLGtCQUFhLEdBQWIsYUFBYSxDQUFTO1FBR2pGLElBQUksQ0FBQyxNQUFNLENBQUMsVUFBVSxFQUFFO1lBQ3RCLE1BQU0sSUFBSSxLQUFLLENBQUMsaURBQWlELENBQUMsQ0FBQztTQUNwRTtRQUVELElBQUksQ0FBQyxVQUFVLEdBQUcsTUFBTSxDQUFDLFVBQVUsQ0FBQztJQUN0QyxDQUFDO0lBRU0sSUFBSSxDQUFDLE1BQXFCO1FBQy9CLE9BQU87WUFDTCxVQUFVLEVBQUU7Z0JBQ1YsTUFBTSxFQUFFLElBQUksQ0FBQyxVQUFVO2dCQUN2QixHQUFHLEVBQUUsSUFBSSxDQUFDLEdBQUc7Z0JBQ2IsT0FBTyxFQUFFLElBQUksQ0FBQyxhQUFhO2FBQzVCO1NBQ0YsQ0FBQztJQUNKLENBQUM7Q0FDRjtBQXRCRCwwQ0FzQkM7QUFFRDs7O0dBR0c7QUFDSCxNQUFhLG1CQUFvQixTQUFRLGFBQWE7SUFDcEQsWUFBb0IsVUFBZTtRQUNqQyxLQUFLLEVBQUUsQ0FBQztRQURVLGVBQVUsR0FBVixVQUFVLENBQUs7UUFHakMsSUFBSSxPQUFNLENBQUMsVUFBVSxDQUFDLEtBQUssUUFBUSxFQUFFO1lBQ25DLE1BQU0sSUFBSSxLQUFLLENBQUMscUNBQXFDLENBQUMsQ0FBQztTQUN4RDtRQUVELElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQ3hDLE1BQU0sSUFBSSxLQUFLLENBQUMsaUNBQWlDLENBQUMsQ0FBQztTQUNwRDtJQUNILENBQUM7SUFFTSxJQUFJLENBQUMsTUFBcUI7UUFDL0IsT0FBTztZQUNMLGdCQUFnQixFQUFFLElBQUksQ0FBQyxVQUFVO1NBQ2xDLENBQUM7SUFDSixDQUFDO0NBQ0Y7QUFsQkQsa0RBa0JDO0FBRUQ7OztHQUdHO0FBQ0gsTUFBYSxrQkFBbUIsU0FBUSxhQUFhO0lBR25ELFlBQTZCLElBQVksRUFBbUIsVUFBa0MsRUFBRztRQUMvRixLQUFLLEVBQUUsQ0FBQztRQURtQixTQUFJLEdBQUosSUFBSSxDQUFRO1FBQW1CLFlBQU8sR0FBUCxPQUFPLENBQThCO0lBRWpHLENBQUM7SUFFTSxJQUFJLENBQUMsS0FBb0I7UUFDOUIsOEZBQThGO1FBQzlGLElBQUksSUFBSSxDQUFDLEtBQUssS0FBSyxTQUFTLEVBQUU7WUFDNUIsSUFBSSxDQUFDLEtBQUssR0FBRyxJQUFJLFNBQVMsQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLGVBQWUsRUFBRTtnQkFDdkQsSUFBSSxFQUFFLElBQUksQ0FBQyxJQUFJO2dCQUNmLEdBQUcsSUFBSSxDQUFDLE9BQU87YUFDaEIsQ0FBQyxDQUFDO1NBQ0o7UUFFRCxJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsWUFBWSxFQUFFO1lBQzNCLE1BQU0sSUFBSSxLQUFLLENBQUMsK0NBQStDLElBQUksQ0FBQyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1NBQzlFO1FBRUQsT0FBTztZQUNMLFVBQVUsRUFBRTtnQkFDVixNQUFNLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxZQUFZO2dCQUMvQixHQUFHLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxXQUFXO2FBQzVCO1NBQ0YsQ0FBQztJQUNKLENBQUM7Q0FDRjtBQTNCRCxnREEyQkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBzMyBmcm9tICdAYXdzLWNkay9hd3MtczMnO1xuaW1wb3J0ICogYXMgczNfYXNzZXRzIGZyb20gJ0Bhd3MtY2RrL2F3cy1zMy1hc3NldHMnO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuXG4vKipcbiAqIFJlcHJlc2VudHMgYW4gT3BlbkFQSSBkZWZpbml0aW9uIGFzc2V0LlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgYWJzdHJhY3QgY2xhc3MgQXBpRGVmaW5pdGlvbiB7XG4gIC8qKlxuICAgKiBDcmVhdGVzIGFuIEFQSSBkZWZpbml0aW9uIGZyb20gYSBzcGVjaWZpY2F0aW9uIGZpbGUgaW4gYW4gUzMgYnVja2V0XG4gICAqIEBleHBlcmltZW50YWxcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUJ1Y2tldChidWNrZXQ6IHMzLklCdWNrZXQsIGtleTogc3RyaW5nLCBvYmplY3RWZXJzaW9uPzogc3RyaW5nKTogUzNBcGlEZWZpbml0aW9uIHtcbiAgICByZXR1cm4gbmV3IFMzQXBpRGVmaW5pdGlvbihidWNrZXQsIGtleSwgb2JqZWN0VmVyc2lvbik7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlIGFuIEFQSSBkZWZpbml0aW9uIGZyb20gYW4gaW5saW5lIG9iamVjdC4gVGhlIGlubGluZSBvYmplY3QgbXVzdCBmb2xsb3cgdGhlXG4gICAqIHNjaGVtYSBvZiBPcGVuQVBJIDIuMCBvciBPcGVuQVBJIDMuMFxuICAgKlxuICAgKiBAZXhhbXBsZVxuICAgKiAgIEFwaURlZmluaXRpb24uZnJvbUlubGluZSh7XG4gICAqICAgICBvcGVuYXBpOiAnMy4wLjInLFxuICAgKiAgICAgcGF0aHM6IHtcbiAgICogICAgICAgJy9wZXRzJzoge1xuICAgKiAgICAgICAgIGdldDoge1xuICAgKiAgICAgICAgICAgJ3Jlc3BvbnNlcyc6IHtcbiAgICogICAgICAgICAgICAgMjAwOiB7XG4gICAqICAgICAgICAgICAgICAgY29udGVudDoge1xuICAgKiAgICAgICAgICAgICAgICAgJ2FwcGxpY2F0aW9uL2pzb24nOiB7XG4gICAqICAgICAgICAgICAgICAgICAgIHNjaGVtYToge1xuICAgKiAgICAgICAgICAgICAgICAgICAgICRyZWY6ICcjL2NvbXBvbmVudHMvc2NoZW1hcy9FbXB0eScsXG4gICAqICAgICAgICAgICAgICAgICAgIH0sXG4gICAqICAgICAgICAgICAgICAgICB9LFxuICAgKiAgICAgICAgICAgICAgIH0sXG4gICAqICAgICAgICAgICAgIH0sXG4gICAqICAgICAgICAgICB9LFxuICAgKiAgICAgICAgICAgJ3gtYW1hem9uLWFwaWdhdGV3YXktaW50ZWdyYXRpb24nOiB7XG4gICAqICAgICAgICAgICAgIHJlc3BvbnNlczoge1xuICAgKiAgICAgICAgICAgICAgIGRlZmF1bHQ6IHtcbiAgICogICAgICAgICAgICAgICAgIHN0YXR1c0NvZGU6ICcyMDAnLFxuICAgKiAgICAgICAgICAgICAgIH0sXG4gICAqICAgICAgICAgICAgIH0sXG4gICAqICAgICAgICAgICAgIHJlcXVlc3RUZW1wbGF0ZXM6IHtcbiAgICogICAgICAgICAgICAgICAnYXBwbGljYXRpb24vanNvbic6ICd7XCJzdGF0dXNDb2RlXCI6IDIwMH0nLFxuICAgKiAgICAgICAgICAgICB9LFxuICAgKiAgICAgICAgICAgICBwYXNzdGhyb3VnaEJlaGF2aW9yOiAnd2hlbl9ub19tYXRjaCcsXG4gICAqICAgICAgICAgICAgIHR5cGU6ICdtb2NrJyxcbiAgICogICAgICAgICAgIH0sXG4gICAqICAgICAgICAgfSxcbiAgICogICAgICAgfSxcbiAgICogICAgIH0sXG4gICAqICAgICBjb21wb25lbnRzOiB7XG4gICAqICAgICAgIHNjaGVtYXM6IHtcbiAgICogICAgICAgICBFbXB0eToge1xuICAgKiAgICAgICAgICAgdGl0bGU6ICdFbXB0eSBTY2hlbWEnLFxuICAgKiAgICAgICAgICAgdHlwZTogJ29iamVjdCcsXG4gICAqICAgICAgICAgfSxcbiAgICogICAgICAgfSxcbiAgICogICAgIH0sXG4gICAqICAgfSk7XG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21JbmxpbmUoZGVmaW5pdGlvbjogYW55KTogSW5saW5lQXBpRGVmaW5pdGlvbiB7XG4gICAgcmV0dXJuIG5ldyBJbmxpbmVBcGlEZWZpbml0aW9uKGRlZmluaXRpb24pO1xuICB9XG5cbiAgLyoqXG4gICAqIExvYWRzIHRoZSBBUEkgc3BlY2lmaWNhdGlvbiBmcm9tIGEgbG9jYWwgZGlzayBhc3NldC5cbiAgICogQGV4cGVyaW1lbnRhbFxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tQXNzZXQoZmlsZTogc3RyaW5nLCBvcHRpb25zPzogczNfYXNzZXRzLkFzc2V0T3B0aW9ucyk6IEFzc2V0QXBpRGVmaW5pdGlvbiB7XG4gICAgcmV0dXJuIG5ldyBBc3NldEFwaURlZmluaXRpb24oZmlsZSwgb3B0aW9ucyk7XG4gIH1cblxuICAvKipcbiAgICogQ2FsbGVkIHdoZW4gdGhlIHNwZWNpZmljYXRpb24gaXMgaW5pdGlhbGl6ZWQgdG8gYWxsb3cgdGhpcyBvYmplY3QgdG8gYmluZFxuICAgKiB0byB0aGUgc3RhY2ssIGFkZCByZXNvdXJjZXMgYW5kIGhhdmUgZnVuLlxuICAgKlxuICAgKiBAcGFyYW0gc2NvcGUgVGhlIGJpbmRpbmcgc2NvcGUuIERvbid0IGJlIHNtYXJ0IGFib3V0IHRyeWluZyB0byBkb3duLWNhc3Qgb3JcbiAgICogYXNzdW1lIGl0J3MgaW5pdGlhbGl6ZWQuIFlvdSBtYXkganVzdCB1c2UgaXQgYXMgYSBjb25zdHJ1Y3Qgc2NvcGUuXG4gICAqL1xuICBwdWJsaWMgYWJzdHJhY3QgYmluZChzY29wZTogY2RrLkNvbnN0cnVjdCk6IEFwaURlZmluaXRpb25Db25maWc7XG59XG5cbi8qKlxuICogUzMgbG9jYXRpb24gb2YgdGhlIEFQSSBkZWZpbml0aW9uIGZpbGVcbiAqIEBleHBlcmltZW50YWxcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBBcGlEZWZpbml0aW9uUzNMb2NhdGlvbiB7XG4gIC8qKiBUaGUgUzMgYnVja2V0ICovXG4gIHJlYWRvbmx5IGJ1Y2tldDogc3RyaW5nO1xuICAvKiogVGhlIFMzIGtleSAqL1xuICByZWFkb25seSBrZXk6IHN0cmluZztcbiAgLyoqXG4gICAqIEFuIG9wdGlvbmFsIHZlcnNpb25cbiAgICogQGRlZmF1bHQgLSBsYXRlc3QgdmVyc2lvblxuICAgKi9cbiAgcmVhZG9ubHkgdmVyc2lvbj86IHN0cmluZztcbn1cblxuLyoqXG4gKiBQb3N0LUJpbmRpbmcgQ29uZmlndXJhdGlvbiBmb3IgYSBDREsgY29uc3RydWN0XG4gKiBAZXhwZXJpbWVudGFsXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQXBpRGVmaW5pdGlvbkNvbmZpZyB7XG4gIC8qKlxuICAgKiBUaGUgbG9jYXRpb24gb2YgdGhlIHNwZWNpZmljYXRpb24gaW4gUzMgKG11dHVhbGx5IGV4Y2x1c2l2ZSB3aXRoIGBpbmxpbmVEZWZpbml0aW9uYCkuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gQVBJIGRlZmluaXRpb24gaXMgbm90IGFuIFMzIGxvY2F0aW9uXG4gICAqL1xuICByZWFkb25seSBzM0xvY2F0aW9uPzogQXBpRGVmaW5pdGlvblMzTG9jYXRpb247XG5cbiAgLyoqXG4gICAqIElubGluZSBzcGVjaWZpY2F0aW9uIChtdXR1YWxseSBleGNsdXNpdmUgd2l0aCBgczNMb2NhdGlvbmApLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIEFQSSBkZWZpbml0aW9uIGlzIG5vdCBkZWZpbmVkIGlubGluZVxuICAgKi9cbiAgcmVhZG9ubHkgaW5saW5lRGVmaW5pdGlvbj86IGFueTtcbn1cblxuLyoqXG4gKiBPcGVuQVBJIHNwZWNpZmljYXRpb24gZnJvbSBhbiBTMyBhcmNoaXZlLlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgY2xhc3MgUzNBcGlEZWZpbml0aW9uIGV4dGVuZHMgQXBpRGVmaW5pdGlvbiB7XG4gIHByaXZhdGUgYnVja2V0TmFtZTogc3RyaW5nO1xuXG4gIGNvbnN0cnVjdG9yKGJ1Y2tldDogczMuSUJ1Y2tldCwgcHJpdmF0ZSBrZXk6IHN0cmluZywgcHJpdmF0ZSBvYmplY3RWZXJzaW9uPzogc3RyaW5nKSB7XG4gICAgc3VwZXIoKTtcblxuICAgIGlmICghYnVja2V0LmJ1Y2tldE5hbWUpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignYnVja2V0TmFtZSBpcyB1bmRlZmluZWQgZm9yIHRoZSBwcm92aWRlZCBidWNrZXQnKTtcbiAgICB9XG5cbiAgICB0aGlzLmJ1Y2tldE5hbWUgPSBidWNrZXQuYnVja2V0TmFtZTtcbiAgfVxuXG4gIHB1YmxpYyBiaW5kKF9zY29wZTogY2RrLkNvbnN0cnVjdCk6IEFwaURlZmluaXRpb25Db25maWcge1xuICAgIHJldHVybiB7XG4gICAgICBzM0xvY2F0aW9uOiB7XG4gICAgICAgIGJ1Y2tldDogdGhpcy5idWNrZXROYW1lLFxuICAgICAgICBrZXk6IHRoaXMua2V5LFxuICAgICAgICB2ZXJzaW9uOiB0aGlzLm9iamVjdFZlcnNpb24sXG4gICAgICB9LFxuICAgIH07XG4gIH1cbn1cblxuLyoqXG4gKiBPcGVuQVBJIHNwZWNpZmljYXRpb24gZnJvbSBhbiBpbmxpbmUgSlNPTiBvYmplY3QuXG4gKiBAZXhwZXJpbWVudGFsXG4gKi9cbmV4cG9ydCBjbGFzcyBJbmxpbmVBcGlEZWZpbml0aW9uIGV4dGVuZHMgQXBpRGVmaW5pdGlvbiB7XG4gIGNvbnN0cnVjdG9yKHByaXZhdGUgZGVmaW5pdGlvbjogYW55KSB7XG4gICAgc3VwZXIoKTtcblxuICAgIGlmICh0eXBlb2YoZGVmaW5pdGlvbikgIT09ICdvYmplY3QnKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ2RlZmluaXRpb24gc2hvdWxkIGJlIG9mIHR5cGUgb2JqZWN0Jyk7XG4gICAgfVxuXG4gICAgaWYgKE9iamVjdC5rZXlzKGRlZmluaXRpb24pLmxlbmd0aCA9PT0gMCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdKU09OIGRlZmluaXRpb24gY2Fubm90IGJlIGVtcHR5Jyk7XG4gICAgfVxuICB9XG5cbiAgcHVibGljIGJpbmQoX3Njb3BlOiBjZGsuQ29uc3RydWN0KTogQXBpRGVmaW5pdGlvbkNvbmZpZyB7XG4gICAgcmV0dXJuIHtcbiAgICAgIGlubGluZURlZmluaXRpb246IHRoaXMuZGVmaW5pdGlvbixcbiAgICB9O1xuICB9XG59XG5cbi8qKlxuICogT3BlbkFQSSBzcGVjaWZpY2F0aW9uIGZyb20gYSBsb2NhbCBmaWxlLlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgY2xhc3MgQXNzZXRBcGlEZWZpbml0aW9uIGV4dGVuZHMgQXBpRGVmaW5pdGlvbiB7XG4gIHByaXZhdGUgYXNzZXQ/OiBzM19hc3NldHMuQXNzZXQ7XG5cbiAgY29uc3RydWN0b3IocHJpdmF0ZSByZWFkb25seSBwYXRoOiBzdHJpbmcsIHByaXZhdGUgcmVhZG9ubHkgb3B0aW9uczogczNfYXNzZXRzLkFzc2V0T3B0aW9ucyA9IHsgfSkge1xuICAgIHN1cGVyKCk7XG4gIH1cblxuICBwdWJsaWMgYmluZChzY29wZTogY2RrLkNvbnN0cnVjdCk6IEFwaURlZmluaXRpb25Db25maWcge1xuICAgIC8vIElmIHRoZSBzYW1lIEFzc2V0QVBJRGVmaW5pdGlvbiBpcyB1c2VkIG11bHRpcGxlIHRpbWVzLCByZXRhaW4gb25seSB0aGUgZmlyc3QgaW5zdGFudGlhdGlvbi5cbiAgICBpZiAodGhpcy5hc3NldCA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICB0aGlzLmFzc2V0ID0gbmV3IHMzX2Fzc2V0cy5Bc3NldChzY29wZSwgJ0FQSURlZmluaXRpb24nLCB7XG4gICAgICAgIHBhdGg6IHRoaXMucGF0aCxcbiAgICAgICAgLi4udGhpcy5vcHRpb25zLFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgaWYgKHRoaXMuYXNzZXQuaXNaaXBBcmNoaXZlKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYEFzc2V0IGNhbm5vdCBiZSBhIC56aXAgZmlsZSBvciBhIGRpcmVjdG9yeSAoJHt0aGlzLnBhdGh9KWApO1xuICAgIH1cblxuICAgIHJldHVybiB7XG4gICAgICBzM0xvY2F0aW9uOiB7XG4gICAgICAgIGJ1Y2tldDogdGhpcy5hc3NldC5zM0J1Y2tldE5hbWUsXG4gICAgICAgIGtleTogdGhpcy5hc3NldC5zM09iamVjdEtleSxcbiAgICAgIH0sXG4gICAgfTtcbiAgfVxufSJdfQ==