"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("./apigateway.generated");
const authorizer_1 = require("./authorizer");
const integration_1 = require("./integration");
const mock_1 = require("./integrations/mock");
const util_1 = require("./util");
class Method extends core_1.Resource {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.resource = props.resource;
        this.restApi = props.resource.restApi;
        this.httpMethod = props.httpMethod.toUpperCase();
        util_1.validateHttpMethod(this.httpMethod);
        const options = props.options || {};
        const defaultMethodOptions = props.resource.defaultMethodOptions || {};
        const authorizer = options.authorizer || defaultMethodOptions.authorizer;
        const authorizerId = authorizer === null || authorizer === void 0 ? void 0 : authorizer.authorizerId;
        const authorizationTypeOption = options.authorizationType || defaultMethodOptions.authorizationType;
        const authorizationType = (authorizer === null || authorizer === void 0 ? void 0 : authorizer.authorizationType) || authorizationTypeOption || AuthorizationType.NONE;
        // if the authorizer defines an authorization type and we also have an explicit option set, check that they are the same
        if ((authorizer === null || authorizer === void 0 ? void 0 : authorizer.authorizationType) && authorizationTypeOption && (authorizer === null || authorizer === void 0 ? void 0 : authorizer.authorizationType) !== authorizationTypeOption) {
            throw new Error(`${this.resource}/${this.httpMethod} - Authorization type is set to ${authorizationTypeOption} ` +
                `which is different from what is required by the authorizer [${authorizer.authorizationType}]`);
        }
        if (authorizer_1.Authorizer.isAuthorizer(authorizer)) {
            authorizer._attachToApi(this.restApi);
        }
        const methodProps = {
            resourceId: props.resource.resourceId,
            restApiId: this.restApi.restApiId,
            httpMethod: this.httpMethod,
            operationName: options.operationName || defaultMethodOptions.operationName,
            apiKeyRequired: options.apiKeyRequired || defaultMethodOptions.apiKeyRequired,
            authorizationType,
            authorizerId,
            requestParameters: options.requestParameters || defaultMethodOptions.requestParameters,
            integration: this.renderIntegration(props.integration),
            methodResponses: this.renderMethodResponses(options.methodResponses),
            requestModels: this.renderRequestModels(options.requestModels),
            requestValidatorId: this.requestValidatorId(options),
            authorizationScopes: (_a = options.authorizationScopes) !== null && _a !== void 0 ? _a : defaultMethodOptions.authorizationScopes,
        };
        const resource = new apigateway_generated_1.CfnMethod(this, 'Resource', methodProps);
        this.methodId = resource.ref;
        props.resource.restApi._attachMethod(this);
        const deployment = props.resource.restApi.latestDeployment;
        if (deployment) {
            deployment.node.addDependency(resource);
            deployment.addToLogicalId({ method: methodProps });
        }
    }
    /**
     * Returns an execute-api ARN for this method:
     *
     *   arn:aws:execute-api:{region}:{account}:{restApiId}/{stage}/{method}/{path}
     *
     * NOTE: {stage} will refer to the `restApi.deploymentStage`, which will
     * automatically set if auto-deploy is enabled.
     *
     * @attribute
     */
    get methodArn() {
        if (!this.restApi.deploymentStage) {
            throw new Error(`Unable to determine ARN for method "${this.node.id}" since there is no stage associated with this API.\n` +
                'Either use the `deploy` prop or explicitly assign `deploymentStage` on the RestApi');
        }
        const stage = this.restApi.deploymentStage.stageName.toString();
        return this.restApi.arnForExecuteApi(this.httpMethod, pathForArn(this.resource.path), stage);
    }
    /**
     * Returns an execute-api ARN for this method's "test-invoke-stage" stage.
     * This stage is used by the AWS Console UI when testing the method.
     */
    get testMethodArn() {
        return this.restApi.arnForExecuteApi(this.httpMethod, pathForArn(this.resource.path), 'test-invoke-stage');
    }
    renderIntegration(integration) {
        if (!integration) {
            // use defaultIntegration from API if defined
            if (this.resource.defaultIntegration) {
                return this.renderIntegration(this.resource.defaultIntegration);
            }
            // fallback to mock
            return this.renderIntegration(new mock_1.MockIntegration());
        }
        integration.bind(this);
        const options = integration._props.options || {};
        let credentials;
        if (options.credentialsPassthrough !== undefined && options.credentialsRole !== undefined) {
            throw new Error('\'credentialsPassthrough\' and \'credentialsRole\' are mutually exclusive');
        }
        if (options.connectionType === integration_1.ConnectionType.VPC_LINK && options.vpcLink === undefined) {
            throw new Error('\'connectionType\' of VPC_LINK requires \'vpcLink\' prop to be set');
        }
        if (options.connectionType === integration_1.ConnectionType.INTERNET && options.vpcLink !== undefined) {
            throw new Error('cannot set \'vpcLink\' where \'connectionType\' is INTERNET');
        }
        if (options.credentialsRole) {
            credentials = options.credentialsRole.roleArn;
        }
        else if (options.credentialsPassthrough) {
            // arn:aws:iam::*:user/*
            // tslint:disable-next-line:max-line-length
            credentials = core_1.Stack.of(this).formatArn({ service: 'iam', region: '', account: '*', resource: 'user', sep: '/', resourceName: '*' });
        }
        return {
            type: integration._props.type,
            uri: integration._props.uri,
            cacheKeyParameters: options.cacheKeyParameters,
            cacheNamespace: options.cacheNamespace,
            contentHandling: options.contentHandling,
            integrationHttpMethod: integration._props.integrationHttpMethod,
            requestParameters: options.requestParameters,
            requestTemplates: options.requestTemplates,
            passthroughBehavior: options.passthroughBehavior,
            integrationResponses: options.integrationResponses,
            connectionType: options.connectionType,
            connectionId: options.vpcLink ? options.vpcLink.vpcLinkId : undefined,
            credentials,
        };
    }
    renderMethodResponses(methodResponses) {
        if (!methodResponses) {
            // Fall back to nothing
            return undefined;
        }
        return methodResponses.map(mr => {
            let responseModels;
            if (mr.responseModels) {
                responseModels = {};
                for (const contentType in mr.responseModels) {
                    if (mr.responseModels.hasOwnProperty(contentType)) {
                        responseModels[contentType] = mr.responseModels[contentType].modelId;
                    }
                }
            }
            const methodResponseProp = {
                statusCode: mr.statusCode,
                responseParameters: mr.responseParameters,
                responseModels,
            };
            return methodResponseProp;
        });
    }
    renderRequestModels(requestModels) {
        if (!requestModels) {
            // Fall back to nothing
            return undefined;
        }
        const models = {};
        for (const contentType in requestModels) {
            if (requestModels.hasOwnProperty(contentType)) {
                models[contentType] = requestModels[contentType].modelId;
            }
        }
        return models;
    }
    requestValidatorId(options) {
        var _a;
        if (options.requestValidator && options.requestValidatorOptions) {
            throw new Error('Only one of \'requestValidator\' or \'requestValidatorOptions\' must be specified.');
        }
        if (options.requestValidatorOptions) {
            const validator = this.restApi.addRequestValidator('validator', options.requestValidatorOptions);
            return validator.requestValidatorId;
        }
        // For backward compatibility
        return (_a = options.requestValidator) === null || _a === void 0 ? void 0 : _a.requestValidatorId;
    }
}
exports.Method = Method;
var AuthorizationType;
(function (AuthorizationType) {
    /**
     * Open access.
     */
    AuthorizationType["NONE"] = "NONE";
    /**
     * Use AWS IAM permissions.
     */
    AuthorizationType["IAM"] = "AWS_IAM";
    /**
     * Use a custom authorizer.
     */
    AuthorizationType["CUSTOM"] = "CUSTOM";
    /**
     * Use an AWS Cognito user pool.
     */
    AuthorizationType["COGNITO"] = "COGNITO_USER_POOLS";
})(AuthorizationType = exports.AuthorizationType || (exports.AuthorizationType = {}));
function pathForArn(path) {
    return path.replace(/\{[^\}]*\}/g, '*'); // replace path parameters (like '{bookId}') with asterisk
}
//# sourceMappingURL=data:application/json;base64,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