"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const elbv2 = require("@aws-cdk/aws-elasticloadbalancingv2");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const apigw = require("../lib");
const DUMMY_AUTHORIZER = {
    authorizerId: 'dummyauthorizer',
    authorizationType: apigw.AuthorizationType.CUSTOM,
};
module.exports = {
    'default setup'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: false });
        // WHEN
        new apigw.Method(stack, 'my-method', {
            httpMethod: 'POST',
            resource: api.root,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'POST',
            AuthorizationType: 'NONE',
            Integration: {
                Type: 'MOCK',
            },
        }));
        test.done();
    },
    'method options can be specified'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: false });
        // WHEN
        new apigw.Method(stack, 'my-method', {
            httpMethod: 'POST',
            resource: api.root,
            options: {
                apiKeyRequired: true,
                operationName: 'MyOperation',
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            ApiKeyRequired: true,
            OperationName: 'MyOperation',
        }));
        test.done();
    },
    'integration can be set via a property'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: false });
        // WHEN
        new apigw.Method(stack, 'my-method', {
            httpMethod: 'POST',
            resource: api.root,
            integration: new apigw.AwsIntegration({ service: 's3', path: 'bucket/key' }),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            Integration: {
                IntegrationHttpMethod: 'POST',
                Type: 'AWS',
                Uri: {
                    'Fn::Join': [
                        '',
                        [
                            'arn:', { Ref: 'AWS::Partition' }, ':apigateway:',
                            { Ref: 'AWS::Region' }, ':s3:path/bucket/key',
                        ],
                    ],
                },
            },
        }));
        test.done();
    },
    'integration with a custom http method can be set via a property'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: false });
        // WHEN
        new apigw.Method(stack, 'my-method', {
            httpMethod: 'POST',
            resource: api.root,
            integration: new apigw.AwsIntegration({ service: 's3', path: 'bucket/key', integrationHttpMethod: 'GET' }),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ApiGateway::Method', {
            Integration: {
                IntegrationHttpMethod: 'GET',
            },
        }));
        test.done();
    },
    'use default integration from api'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const defaultIntegration = new apigw.Integration({ type: apigw.IntegrationType.HTTP_PROXY, uri: 'https://amazon.com' });
        const api = new apigw.RestApi(stack, 'test-api', {
            cloudWatchRole: false,
            deploy: false,
            defaultIntegration,
        });
        // WHEN
        new apigw.Method(stack, 'my-method', {
            httpMethod: 'POST',
            resource: api.root,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            Integration: {
                Type: 'HTTP_PROXY',
                Uri: 'https://amazon.com',
            },
        }));
        test.done();
    },
    '"methodArn" returns the ARN execute-api ARN for this method in the current stage'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api');
        // WHEN
        const method = new apigw.Method(stack, 'my-method', {
            httpMethod: 'POST',
            resource: api.root,
        });
        // THEN
        test.deepEqual(stack.resolve(method.methodArn), {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':execute-api:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':',
                    { Ref: 'testapiD6451F70' },
                    '/',
                    { Ref: 'testapiDeploymentStageprod5C9E92A4' },
                    '/POST/',
                ],
            ],
        });
        test.done();
    },
    '"testMethodArn" returns the ARN of the "test-invoke-stage" stage (console UI)'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api');
        // WHEN
        const method = new apigw.Method(stack, 'my-method', {
            httpMethod: 'POST',
            resource: api.root,
        });
        // THEN
        test.deepEqual(stack.resolve(method.testMethodArn), {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':execute-api:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':',
                    { Ref: 'testapiD6451F70' },
                    '/test-invoke-stage/POST/',
                ],
            ],
        });
        test.done();
    },
    '"methodArn" fails if the API does not have a deployment stage'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        const method = new apigw.Method(stack, 'my-method', { httpMethod: 'POST', resource: api.root });
        // WHEN + THEN
        test.throws(() => method.methodArn, /Unable to determine ARN for method "my-method" since there is no stage associated with this API./);
        test.done();
    },
    '"methodArn" and "testMethodArn" replace path parameters with asterisks'(test) {
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api');
        const petId = api.root.addResource('pets').addResource('{petId}');
        const commentId = petId.addResource('comments').addResource('{commentId}');
        const method = commentId.addMethod('GET');
        test.deepEqual(stack.resolve(method.methodArn), {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':execute-api:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':',
                    { Ref: 'testapiD6451F70' },
                    '/',
                    { Ref: 'testapiDeploymentStageprod5C9E92A4' },
                    '/GET/pets/*/comments/*',
                ],
            ],
        });
        test.deepEqual(stack.resolve(method.testMethodArn), {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':execute-api:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':',
                    { Ref: 'testapiD6451F70' },
                    '/test-invoke-stage/GET/pets/*/comments/*',
                ],
            ],
        });
        test.done();
    },
    'integration "credentialsRole" can be used to assume a role when calling backend'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        const role = new iam.Role(stack, 'MyRole', { assumedBy: new iam.ServicePrincipal('foo') });
        // WHEN
        api.root.addMethod('GET', new apigw.Integration({
            type: apigw.IntegrationType.AWS_PROXY,
            options: {
                credentialsRole: role,
            },
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ApiGateway::Method', {
            Integration: {
                Credentials: { 'Fn::GetAtt': ['MyRoleF48FFE04', 'Arn'] },
            },
        }));
        test.done();
    },
    'integration "credentialsPassthrough" can be used to passthrough user credentials to backend'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        // WHEN
        api.root.addMethod('GET', new apigw.Integration({
            type: apigw.IntegrationType.AWS_PROXY,
            options: {
                credentialsPassthrough: true,
            },
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ApiGateway::Method', {
            Integration: {
                Credentials: { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::*:user/*']] },
            },
        }));
        test.done();
    },
    'integration "credentialsRole" and "credentialsPassthrough" are mutually exclusive'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        const role = new iam.Role(stack, 'MyRole', { assumedBy: new iam.ServicePrincipal('foo') });
        // WHEN
        const integration = new apigw.Integration({
            type: apigw.IntegrationType.AWS_PROXY,
            options: {
                credentialsPassthrough: true,
                credentialsRole: role,
            },
        });
        // THEN
        test.throws(() => api.root.addMethod('GET', integration), /'credentialsPassthrough' and 'credentialsRole' are mutually exclusive/);
        test.done();
    },
    'integration connectionType VpcLink requires vpcLink to be set'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        // WHEN
        const integration = new apigw.Integration({
            type: apigw.IntegrationType.HTTP_PROXY,
            integrationHttpMethod: 'ANY',
            options: {
                connectionType: apigw.ConnectionType.VPC_LINK,
            },
        });
        // THEN
        test.throws(() => api.root.addMethod('GET', integration), /'connectionType' of VPC_LINK requires 'vpcLink' prop to be set/);
        test.done();
    },
    'connectionType of INTERNET and vpcLink are mutually exclusive'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        const vpc = new ec2.Vpc(stack, 'VPC');
        const nlb = new elbv2.NetworkLoadBalancer(stack, 'NLB', {
            vpc,
        });
        const link = new apigw.VpcLink(stack, 'link', {
            targets: [nlb],
        });
        // WHEN
        const integration = new apigw.Integration({
            type: apigw.IntegrationType.HTTP_PROXY,
            integrationHttpMethod: 'ANY',
            options: {
                connectionType: apigw.ConnectionType.INTERNET,
                vpcLink: link,
            },
        });
        // THEN
        test.throws(() => api.root.addMethod('GET', integration), /cannot set 'vpcLink' where 'connectionType' is INTERNET/);
        test.done();
    },
    'methodResponse set one or more method responses via options'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        // WHEN
        new apigw.Method(stack, 'method-man', {
            httpMethod: 'GET',
            resource: api.root,
            options: {
                methodResponses: [{
                        statusCode: '200',
                    }, {
                        statusCode: '400',
                        responseParameters: {
                            'method.response.header.killerbees': false,
                        },
                    }, {
                        statusCode: '500',
                        responseParameters: {
                            'method.response.header.errthing': true,
                        },
                        responseModels: {
                            'application/json': apigw.Model.EMPTY_MODEL,
                            'text/plain': apigw.Model.ERROR_MODEL,
                        },
                    },
                ],
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'GET',
            MethodResponses: [{
                    StatusCode: '200',
                }, {
                    StatusCode: '400',
                    ResponseParameters: {
                        'method.response.header.killerbees': false,
                    },
                }, {
                    StatusCode: '500',
                    ResponseParameters: {
                        'method.response.header.errthing': true,
                    },
                    ResponseModels: {
                        'application/json': 'Empty',
                        'text/plain': 'Error',
                    },
                },
            ],
        }));
        test.done();
    },
    'multiple integration responses can be used'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        // WHEN
        api.root.addMethod('GET', new apigw.AwsIntegration({
            service: 'foo-service',
            action: 'BarAction',
            options: {
                integrationResponses: [
                    {
                        statusCode: '200',
                        responseTemplates: { 'application/json': JSON.stringify({ success: true }) },
                    },
                    {
                        selectionPattern: 'Invalid',
                        statusCode: '503',
                        responseTemplates: { 'application/json': JSON.stringify({ success: false, message: 'Invalid Request' }) },
                    },
                ],
            },
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            Integration: {
                IntegrationHttpMethod: 'POST',
                IntegrationResponses: [
                    {
                        ResponseTemplates: { 'application/json': '{"success":true}' },
                        StatusCode: '200',
                    },
                    {
                        ResponseTemplates: { 'application/json': '{"success":false,"message":"Invalid Request"}' },
                        SelectionPattern: 'Invalid',
                        StatusCode: '503',
                    },
                ],
                Type: 'AWS',
                Uri: { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':apigateway:', { Ref: 'AWS::Region' }, ':foo-service:action/BarAction']] },
            },
        }));
        test.done();
    },
    'method is always set as uppercase'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'api');
        // WHEN
        api.root.addMethod('get');
        api.root.addMethod('PoSt');
        api.root.addMethod('PUT');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', { HttpMethod: 'POST' }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', { HttpMethod: 'GET' }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', { HttpMethod: 'PUT' }));
        test.done();
    },
    'requestModel can be set'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        const model = api.addModel('test-model', {
            contentType: 'application/json',
            modelName: 'test-model',
            schema: {
                title: 'test',
                type: apigw.JsonSchemaType.OBJECT,
                properties: { message: { type: apigw.JsonSchemaType.STRING } },
            },
        });
        // WHEN
        new apigw.Method(stack, 'method-man', {
            httpMethod: 'GET',
            resource: api.root,
            options: {
                requestModels: {
                    'application/json': model,
                },
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'GET',
            RequestModels: {
                'application/json': { Ref: stack.getLogicalId(model.node.findChild('Resource')) },
            },
        }));
        test.done();
    },
    'methodResponse has a mix of response modes'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        const htmlModel = api.addModel('my-model', {
            schema: {
                schema: apigw.JsonSchemaVersion.DRAFT4,
                title: 'test',
                type: apigw.JsonSchemaType.OBJECT,
                properties: { message: { type: apigw.JsonSchemaType.STRING } },
            },
        });
        // WHEN
        new apigw.Method(stack, 'method-man', {
            httpMethod: 'GET',
            resource: api.root,
            options: {
                methodResponses: [{
                        statusCode: '200',
                    }, {
                        statusCode: '400',
                        responseParameters: {
                            'method.response.header.killerbees': false,
                        },
                    }, {
                        statusCode: '500',
                        responseParameters: {
                            'method.response.header.errthing': true,
                        },
                        responseModels: {
                            'application/json': apigw.Model.EMPTY_MODEL,
                            'text/plain': apigw.Model.ERROR_MODEL,
                            'text/html': htmlModel,
                        },
                    },
                ],
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'GET',
            MethodResponses: [{
                    StatusCode: '200',
                }, {
                    StatusCode: '400',
                    ResponseParameters: {
                        'method.response.header.killerbees': false,
                    },
                }, {
                    StatusCode: '500',
                    ResponseParameters: {
                        'method.response.header.errthing': true,
                    },
                    ResponseModels: {
                        'application/json': 'Empty',
                        'text/plain': 'Error',
                        'text/html': { Ref: stack.getLogicalId(htmlModel.node.findChild('Resource')) },
                    },
                },
            ],
        }));
        test.done();
    },
    'method has a request validator'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        const validator = api.addRequestValidator('validator', {
            validateRequestBody: true,
            validateRequestParameters: false,
        });
        // WHEN
        new apigw.Method(stack, 'method-man', {
            httpMethod: 'GET',
            resource: api.root,
            options: {
                requestValidator: validator,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            RequestValidatorId: { Ref: stack.getLogicalId(validator.node.findChild('Resource')) },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::RequestValidator', {
            RestApiId: { Ref: stack.getLogicalId(api.node.findChild('Resource')) },
            ValidateRequestBody: true,
            ValidateRequestParameters: false,
        }));
        test.done();
    },
    'use default requestParameters'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', {
            cloudWatchRole: false,
            deploy: false,
            defaultMethodOptions: {
                requestParameters: { 'method.request.path.proxy': true },
            },
        });
        // WHEN
        new apigw.Method(stack, 'defaultRequestParameters', {
            httpMethod: 'POST',
            resource: api.root,
            options: {
                operationName: 'defaultRequestParameters',
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            OperationName: 'defaultRequestParameters',
            RequestParameters: {
                'method.request.path.proxy': true,
            },
        }));
        test.done();
    },
    'authorizer is bound correctly'(test) {
        const stack = new cdk.Stack();
        const restApi = new apigw.RestApi(stack, 'myrestapi');
        restApi.root.addMethod('ANY', undefined, {
            authorizer: DUMMY_AUTHORIZER,
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'ANY',
            AuthorizationType: 'CUSTOM',
            AuthorizerId: DUMMY_AUTHORIZER.authorizerId,
        }));
        test.done();
    },
    'authorizer via default method options'(test) {
        const stack = new cdk.Stack();
        const func = new lambda.Function(stack, 'myfunction', {
            handler: 'handler',
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const auth = new apigw.TokenAuthorizer(stack, 'myauthorizer1', {
            authorizerName: 'myauthorizer1',
            handler: func,
        });
        const restApi = new apigw.RestApi(stack, 'myrestapi', {
            defaultMethodOptions: {
                authorizer: auth,
            },
        });
        restApi.root.addMethod('ANY');
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Authorizer', {
            Name: 'myauthorizer1',
            Type: 'TOKEN',
            RestApiId: stack.resolve(restApi.restApiId),
        }));
        test.done();
    },
    'fails when authorization type does not match the authorizer'(test) {
        const stack = new cdk.Stack();
        const restApi = new apigw.RestApi(stack, 'myrestapi');
        test.throws(() => {
            restApi.root.addMethod('ANY', undefined, {
                authorizationType: apigw.AuthorizationType.IAM,
                authorizer: DUMMY_AUTHORIZER,
            });
        }, /Authorization type is set to AWS_IAM which is different from what is required by the authorizer/);
        test.done();
    },
    'fails when authorization type does not match the authorizer in default method options'(test) {
        const stack = new cdk.Stack();
        const restApi = new apigw.RestApi(stack, 'myrestapi', {
            defaultMethodOptions: {
                authorizer: DUMMY_AUTHORIZER,
            },
        });
        test.throws(() => {
            restApi.root.addMethod('ANY', undefined, {
                authorizationType: apigw.AuthorizationType.NONE,
            });
        }, /Authorization type is set to NONE which is different from what is required by the authorizer/);
        test.done();
    },
    'method has Auth Scopes'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: false });
        // WHEN
        new apigw.Method(stack, 'my-method', {
            httpMethod: 'POST',
            resource: api.root,
            options: {
                apiKeyRequired: true,
                authorizationScopes: ['AuthScope1', 'AuthScope2'],
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            ApiKeyRequired: true,
            AuthorizationScopes: ['AuthScope1', 'AuthScope2'],
        }));
        test.done();
    },
    'use default Auth Scopes'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', {
            cloudWatchRole: false,
            deploy: false,
            defaultMethodOptions: {
                authorizationScopes: ['DefaultAuth'],
            },
        });
        // WHEN
        new apigw.Method(stack, 'defaultAuthScopes', {
            httpMethod: 'POST',
            resource: api.root,
            options: {
                operationName: 'defaultAuthScopes',
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            OperationName: 'defaultAuthScopes',
            AuthorizationScopes: ['DefaultAuth'],
        }));
        test.done();
    },
    'Method options Auth Scopes is picked up'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', {
            cloudWatchRole: false,
            deploy: false,
            defaultMethodOptions: {
                authorizationScopes: ['DefaultAuth'],
            },
        });
        // WHEN
        new apigw.Method(stack, 'MethodAuthScopeUsed', {
            httpMethod: 'POST',
            resource: api.root,
            options: {
                apiKeyRequired: true,
                authorizationScopes: ['MethodAuthScope'],
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            ApiKeyRequired: true,
            AuthorizationScopes: ['MethodAuthScope'],
        }));
        test.done();
    },
    'Auth Scopes absent'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', {
            cloudWatchRole: false,
            deploy: false,
        });
        // WHEN
        new apigw.Method(stack, 'authScopesAbsent', {
            httpMethod: 'POST',
            resource: api.root,
            options: {
                operationName: 'authScopesAbsent',
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            OperationName: 'authScopesAbsent',
            AuthorizationScopes: assert_1.ABSENT,
        }));
        test.done();
    },
    'method has a request validator with provided properties'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        // WHEN
        new apigw.Method(stack, 'method-man', {
            httpMethod: 'GET',
            resource: api.root,
            options: {
                requestValidatorOptions: {
                    requestValidatorName: 'test-validator',
                    validateRequestBody: true,
                    validateRequestParameters: false,
                },
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::RequestValidator', {
            RestApiId: stack.resolve(api.restApiId),
            ValidateRequestBody: true,
            ValidateRequestParameters: false,
            Name: 'test-validator',
        }));
        test.done();
    },
    'method does not have a request validator'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        // WHEN
        new apigw.Method(stack, 'method-man', {
            httpMethod: 'GET',
            resource: api.root,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            RequestValidatorId: assert_1.ABSENT,
        }));
        test.done();
    },
    'method does not support both request validator and request validator options'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        const validator = api.addRequestValidator('test-validator1', {
            validateRequestBody: true,
            validateRequestParameters: false,
        });
        // WHEN
        const methodProps = {
            httpMethod: 'GET',
            resource: api.root,
            options: {
                requestValidatorOptions: {
                    requestValidatorName: 'test-validator2',
                    validateRequestBody: true,
                    validateRequestParameters: false,
                },
                requestValidator: validator,
            },
        };
        // THEN
        test.throws(() => new apigw.Method(stack, 'method', methodProps), /Only one of 'requestValidator' or 'requestValidatorOptions' must be specified./);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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