"use strict";
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const apigw = require("../lib");
module.exports = {
    'minimal setup'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'my-api');
        api.root.addMethod('GET'); // must have at least one method or an API definition
        // THEN
        assert_1.expect(stack).toMatch({
            Resources: {
                myapi4C7BF186: {
                    Type: 'AWS::ApiGateway::RestApi',
                    Properties: {
                        Name: 'my-api',
                    },
                },
                myapiGETF990CE3C: {
                    Type: 'AWS::ApiGateway::Method',
                    Properties: {
                        HttpMethod: 'GET',
                        ResourceId: { 'Fn::GetAtt': ['myapi4C7BF186', 'RootResourceId'] },
                        RestApiId: { Ref: 'myapi4C7BF186' },
                        AuthorizationType: 'NONE',
                        Integration: {
                            Type: 'MOCK',
                        },
                    },
                },
                myapiDeployment92F2CB4972a890db5063ec679071ba7eefc76f2a: {
                    Type: 'AWS::ApiGateway::Deployment',
                    Properties: {
                        RestApiId: { Ref: 'myapi4C7BF186' },
                        Description: 'Automatically created by the RestApi construct',
                    },
                    DependsOn: ['myapiGETF990CE3C'],
                },
                myapiDeploymentStageprod298F01AF: {
                    Type: 'AWS::ApiGateway::Stage',
                    Properties: {
                        RestApiId: { Ref: 'myapi4C7BF186' },
                        DeploymentId: { Ref: 'myapiDeployment92F2CB4972a890db5063ec679071ba7eefc76f2a' },
                        StageName: 'prod',
                    },
                },
                myapiCloudWatchRole095452E5: {
                    Type: 'AWS::IAM::Role',
                    Properties: {
                        AssumeRolePolicyDocument: {
                            Statement: [
                                {
                                    Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'apigateway.amazonaws.com' },
                                },
                            ],
                            Version: '2012-10-17',
                        },
                        ManagedPolicyArns: [
                            { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::aws:policy/service-role/AmazonAPIGatewayPushToCloudWatchLogs']] },
                        ],
                    },
                },
                myapiAccountEC421A0A: {
                    Type: 'AWS::ApiGateway::Account',
                    Properties: {
                        CloudWatchRoleArn: { 'Fn::GetAtt': ['myapiCloudWatchRole095452E5', 'Arn'] },
                    },
                    DependsOn: ['myapi4C7BF186'],
                },
            },
            Outputs: {
                myapiEndpoint3628AFE3: {
                    Value: {
                        'Fn::Join': ['', [
                                'https://',
                                { Ref: 'myapi4C7BF186' },
                                '.execute-api.',
                                { Ref: 'AWS::Region' },
                                '.',
                                { Ref: 'AWS::URLSuffix' },
                                '/',
                                { Ref: 'myapiDeploymentStageprod298F01AF' },
                                '/',
                            ]],
                    },
                },
            },
        });
        test.done();
    },
    'defaultChild is set correctly'(test) {
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'my-api');
        test.ok(api.node.defaultChild instanceof apigw.CfnRestApi);
        test.done();
    },
    '"name" is defaulted to resource physical name'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'restapi', {
            deploy: false,
            cloudWatchRole: false,
        });
        api.root.addMethod('GET');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::RestApi', {
            Name: 'restapi',
        }));
        test.done();
    },
    'fails in synthesis if there are no methods or definition'(test) {
        // GIVEN
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'my-stack');
        const api = new apigw.RestApi(stack, 'API');
        // WHEN
        api.root.addResource('foo');
        api.root.addResource('bar').addResource('goo');
        // THEN
        test.throws(() => app.synth(), /The REST API doesn't contain any methods/);
        test.done();
    },
    '"addResource" can be used on "IRestApiResource" to form a tree'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'restapi', {
            deploy: false,
            cloudWatchRole: false,
            restApiName: 'my-rest-api',
        });
        api.root.addMethod('GET');
        // WHEN
        const foo = api.root.addResource('foo');
        api.root.addResource('bar');
        foo.addResource('{hello}');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Resource', {
            PathPart: 'foo',
            ParentId: { 'Fn::GetAtt': ['restapiC5611D27', 'RootResourceId'] },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Resource', {
            PathPart: 'bar',
            ParentId: { 'Fn::GetAtt': ['restapiC5611D27', 'RootResourceId'] },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Resource', {
            PathPart: '{hello}',
            ParentId: { Ref: 'restapifooF697E056' },
        }));
        test.done();
    },
    '"addResource" allows configuration of proxy paths'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'restapi', {
            deploy: false,
            cloudWatchRole: false,
            restApiName: 'my-rest-api',
        });
        // WHEN
        const proxy = api.root.addResource('{proxy+}');
        proxy.addMethod('ANY');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Resource', {
            PathPart: '{proxy+}',
            ParentId: { 'Fn::GetAtt': ['restapiC5611D27', 'RootResourceId'] },
        }));
        test.done();
    },
    '"addMethod" can be used to add methods to resources'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'restapi', { deploy: false, cloudWatchRole: false });
        const r1 = api.root.addResource('r1');
        // WHEN
        api.root.addMethod('GET');
        r1.addMethod('POST');
        // THEN
        assert_1.expect(stack).toMatch({
            Resources: {
                restapiC5611D27: {
                    Type: 'AWS::ApiGateway::RestApi',
                    Properties: {
                        Name: 'restapi',
                    },
                },
                restapir1CF2997EA: {
                    Type: 'AWS::ApiGateway::Resource',
                    Properties: {
                        ParentId: {
                            'Fn::GetAtt': [
                                'restapiC5611D27',
                                'RootResourceId',
                            ],
                        },
                        PathPart: 'r1',
                        RestApiId: {
                            Ref: 'restapiC5611D27',
                        },
                    },
                },
                restapir1POST766920C4: {
                    Type: 'AWS::ApiGateway::Method',
                    Properties: {
                        HttpMethod: 'POST',
                        ResourceId: {
                            Ref: 'restapir1CF2997EA',
                        },
                        RestApiId: {
                            Ref: 'restapiC5611D27',
                        },
                        AuthorizationType: 'NONE',
                        Integration: {
                            Type: 'MOCK',
                        },
                    },
                },
                restapiGET6FC1785A: {
                    Type: 'AWS::ApiGateway::Method',
                    Properties: {
                        HttpMethod: 'GET',
                        ResourceId: {
                            'Fn::GetAtt': [
                                'restapiC5611D27',
                                'RootResourceId',
                            ],
                        },
                        RestApiId: {
                            Ref: 'restapiC5611D27',
                        },
                        AuthorizationType: 'NONE',
                        Integration: {
                            Type: 'MOCK',
                        },
                    },
                },
            },
        });
        test.done();
    },
    'resourcePath returns the full path of the resource within the API'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'restapi');
        // WHEN
        const r1 = api.root.addResource('r1');
        const r11 = r1.addResource('r1_1');
        const r12 = r1.addResource('r1_2');
        const r121 = r12.addResource('r1_2_1');
        const r2 = api.root.addResource('r2');
        // THEN
        test.deepEqual(api.root.path, '/');
        test.deepEqual(r1.path, '/r1');
        test.deepEqual(r11.path, '/r1/r1_1');
        test.deepEqual(r12.path, '/r1/r1_2');
        test.deepEqual(r121.path, '/r1/r1_2/r1_2_1');
        test.deepEqual(r2.path, '/r2');
        test.done();
    },
    'resource path part validation'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'restapi');
        // THEN
        test.throws(() => api.root.addResource('foo/'));
        api.root.addResource('boom-bam');
        test.throws(() => api.root.addResource('illegal()'));
        api.root.addResource('{foo}');
        test.throws(() => api.root.addResource('foo{bar}'));
        test.done();
    },
    'fails if "deployOptions" is set with "deploy" disabled'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // THEN
        test.throws(() => new apigw.RestApi(stack, 'myapi', {
            deploy: false,
            deployOptions: { cachingEnabled: true },
        }), /Cannot set 'deployOptions' if 'deploy' is disabled/);
        test.done();
    },
    'CloudWatch role is created for API Gateway'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'myapi');
        api.root.addMethod('GET');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role'));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Account'));
        test.done();
    },
    'fromRestApiId'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const imported = apigw.RestApi.fromRestApiId(stack, 'imported-api', 'api-rxt4498f');
        // THEN
        test.deepEqual(stack.resolve(imported.restApiId), 'api-rxt4498f');
        test.done();
    },
    '"url" and "urlForPath" return the URL endpoints of the deployed API'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        api.root.addMethod('GET');
        // THEN
        test.deepEqual(stack.resolve(api.url), { 'Fn::Join': ['',
                ['https://',
                    { Ref: 'apiC8550315' },
                    '.execute-api.',
                    { Ref: 'AWS::Region' },
                    '.',
                    { Ref: 'AWS::URLSuffix' },
                    '/',
                    { Ref: 'apiDeploymentStageprod896C8101' },
                    '/']] });
        test.deepEqual(stack.resolve(api.urlForPath('/foo/bar')), { 'Fn::Join': ['',
                ['https://',
                    { Ref: 'apiC8550315' },
                    '.execute-api.',
                    { Ref: 'AWS::Region' },
                    '.',
                    { Ref: 'AWS::URLSuffix' },
                    '/',
                    { Ref: 'apiDeploymentStageprod896C8101' },
                    '/foo/bar']] });
        test.done();
    },
    '"urlForPath" would not work if there is no deployment'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api', { deploy: false });
        api.root.addMethod('GET');
        // THEN
        test.throws(() => api.url, /Cannot determine deployment stage for API from "deploymentStage". Use "deploy" or explicitly set "deploymentStage"/);
        test.throws(() => api.urlForPath('/foo'), /Cannot determine deployment stage for API from "deploymentStage". Use "deploy" or explicitly set "deploymentStage"/);
        test.done();
    },
    '"urlForPath" requires that path will begin with "/"'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        api.root.addMethod('GET');
        // THEN
        test.throws(() => api.urlForPath('foo'), /Path must begin with \"\/\": foo/);
        test.done();
    },
    '"executeApiArn" returns the execute-api ARN for a resource/method'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        api.root.addMethod('GET');
        // WHEN
        const arn = api.arnForExecuteApi('method', '/path', 'stage');
        // THEN
        test.deepEqual(stack.resolve(arn), { 'Fn::Join': ['',
                ['arn:',
                    { Ref: 'AWS::Partition' },
                    ':execute-api:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':',
                    { Ref: 'apiC8550315' },
                    '/stage/method/path']] });
        test.done();
    },
    '"executeApiArn" path must begin with "/"'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        api.root.addMethod('GET');
        // THEN
        test.throws(() => api.arnForExecuteApi('method', 'hey-path', 'stage'), /"path" must begin with a "\/": 'hey-path'/);
        test.done();
    },
    '"executeApiArn" will convert ANY to "*"'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        const method = api.root.addMethod('ANY');
        // THEN
        test.deepEqual(stack.resolve(method.methodArn), { 'Fn::Join': ['',
                ['arn:',
                    { Ref: 'AWS::Partition' },
                    ':execute-api:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':',
                    { Ref: 'apiC8550315' },
                    '/',
                    { Ref: 'apiDeploymentStageprod896C8101' },
                    '/*/']] });
        test.done();
    },
    '"endpointTypes" can be used to specify endpoint configuration for the api'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'api', {
            endpointTypes: [apigw.EndpointType.EDGE, apigw.EndpointType.PRIVATE],
        });
        api.root.addMethod('GET');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::RestApi', {
            EndpointConfiguration: {
                Types: [
                    'EDGE',
                    'PRIVATE',
                ],
            },
        }));
        test.done();
    },
    '"endpointConfiguration" can be used to specify endpoint types for the api'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'api', {
            endpointConfiguration: {
                types: [apigw.EndpointType.EDGE, apigw.EndpointType.PRIVATE],
            },
        });
        api.root.addMethod('GET');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::RestApi', {
            EndpointConfiguration: {
                Types: ['EDGE', 'PRIVATE'],
            },
        }));
        test.done();
    },
    '"endpointConfiguration" can be used to specify vpc endpoints on the API'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'api', {
            endpointConfiguration: {
                types: [apigw.EndpointType.EDGE, apigw.EndpointType.PRIVATE],
                vpcEndpoints: [
                    aws_ec2_1.GatewayVpcEndpoint.fromGatewayVpcEndpointId(stack, 'ImportedEndpoint', 'vpcEndpoint'),
                    aws_ec2_1.GatewayVpcEndpoint.fromGatewayVpcEndpointId(stack, 'ImportedEndpoint2', 'vpcEndpoint2'),
                ],
            },
        });
        api.root.addMethod('GET');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::RestApi', {
            EndpointConfiguration: {
                Types: [
                    'EDGE',
                    'PRIVATE',
                ],
                VpcEndpointIds: [
                    'vpcEndpoint',
                    'vpcEndpoint2',
                ],
            },
        }));
        test.done();
    },
    '"endpointTypes" and "endpointConfiguration" can NOT both be used to specify endpoint configuration for the api'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // THEN
        test.throws(() => new apigw.RestApi(stack, 'api', {
            endpointConfiguration: {
                types: [apigw.EndpointType.PRIVATE],
                vpcEndpoints: [aws_ec2_1.GatewayVpcEndpoint.fromGatewayVpcEndpointId(stack, 'ImportedEndpoint', 'vpcEndpoint')],
            },
            endpointTypes: [apigw.EndpointType.PRIVATE],
        }), /Only one of the RestApi props, endpointTypes or endpointConfiguration, is allowed/);
        test.done();
    },
    '"cloneFrom" can be used to clone an existing API'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cloneFrom = apigw.RestApi.fromRestApiId(stack, 'RestApi', 'foobar');
        // WHEN
        const api = new apigw.RestApi(stack, 'api', {
            cloneFrom,
        });
        api.root.addMethod('GET');
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::RestApi', {
            CloneFrom: 'foobar',
            Name: 'api',
        }));
        test.done();
    },
    'allow taking a dependency on the rest api (includes deployment and stage)'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'myapi');
        api.root.addMethod('GET');
        const resource = new core_1.CfnResource(stack, 'DependsOnRestApi', { type: 'My::Resource' });
        // WHEN
        resource.node.addDependency(api);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('My::Resource', {
            DependsOn: [
                'myapiAccountC3A4750C',
                'myapiCloudWatchRoleEB425128',
                'myapiGET9B7CD29E',
                'myapiDeploymentB7EF8EB7b8edc043bcd33e0d85a3c85151f47e98',
                'myapiDeploymentStageprod329F21FF',
                'myapi162F20B8',
            ],
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'defaultIntegration and defaultMethodOptions can be used at any level'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const rootInteg = new apigw.AwsIntegration({
            service: 's3',
            action: 'GetObject',
        });
        // WHEN
        const api = new apigw.RestApi(stack, 'myapi', {
            defaultIntegration: rootInteg,
            defaultMethodOptions: {
                authorizer: { authorizerId: 'AUTHID' },
                authorizationType: apigw.AuthorizationType.IAM,
            },
        });
        // CASE #1: should inherit integration and options from root resource
        api.root.addMethod('GET');
        const child = api.root.addResource('child');
        // CASE #2: should inherit integration from root and method options, but
        // "authorizationType" will be overridden to "None" instead of "IAM"
        child.addMethod('POST', undefined, {
            authorizationType: apigw.AuthorizationType.COGNITO,
        });
        const child2 = api.root.addResource('child2', {
            defaultIntegration: new apigw.MockIntegration(),
            defaultMethodOptions: {
                authorizer: { authorizerId: 'AUTHID2' },
            },
        });
        // CASE #3: integartion and authorizer ID are inherited from child2
        child2.addMethod('DELETE');
        // CASE #4: same as case #3, but integration is customized
        child2.addMethod('PUT', new apigw.AwsIntegration({ action: 'foo', service: 'bar' }));
        // THEN
        // CASE #1
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ApiGateway::Method', {
            HttpMethod: 'GET',
            ResourceId: { 'Fn::GetAtt': ['myapi162F20B8', 'RootResourceId'] },
            Integration: { Type: 'AWS' },
            AuthorizerId: 'AUTHID',
            AuthorizationType: 'AWS_IAM',
        }));
        // CASE #2
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ApiGateway::Method', {
            HttpMethod: 'POST',
            ResourceId: { Ref: 'myapichildA0A65412' },
            Integration: { Type: 'AWS' },
            AuthorizerId: 'AUTHID',
            AuthorizationType: 'COGNITO_USER_POOLS',
        }));
        // CASE #3
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ApiGateway::Method', {
            HttpMethod: 'DELETE',
            Integration: { Type: 'MOCK' },
            AuthorizerId: 'AUTHID2',
            AuthorizationType: 'AWS_IAM',
        }));
        // CASE #4
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ApiGateway::Method', {
            HttpMethod: 'PUT',
            Integration: { Type: 'AWS' },
            AuthorizerId: 'AUTHID2',
            AuthorizationType: 'AWS_IAM',
        }));
        test.done();
    },
    'addApiKey is supported'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'myapi');
        api.root.addMethod('OPTIONS');
        // WHEN
        api.addApiKey('myapikey', {
            apiKeyName: 'myApiKey1',
            value: '01234567890ABCDEFabcdef',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::ApiKey', {
            Enabled: true,
            Name: 'myApiKey1',
            StageKeys: [
                {
                    RestApiId: { Ref: 'myapi162F20B8' },
                    StageName: { Ref: 'myapiDeploymentStageprod329F21FF' },
                },
            ],
            Value: '01234567890ABCDEFabcdef',
        }));
        test.done();
    },
    'addModel is supported'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'myapi');
        api.root.addMethod('OPTIONS');
        // WHEN
        api.addModel('model', {
            schema: {
                schema: apigw.JsonSchemaVersion.DRAFT4,
                title: 'test',
                type: apigw.JsonSchemaType.OBJECT,
                properties: { message: { type: apigw.JsonSchemaType.STRING } },
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Model', {
            RestApiId: { Ref: stack.getLogicalId(api.node.findChild('Resource')) },
            Schema: {
                $schema: 'http://json-schema.org/draft-04/schema#',
                title: 'test',
                type: 'object',
                properties: { message: { type: 'string' } },
            },
        }));
        test.done();
    },
    'addRequestValidator is supported'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'myapi');
        api.root.addMethod('OPTIONS');
        // WHEN
        api.addRequestValidator('params-validator', {
            requestValidatorName: 'Parameters',
            validateRequestBody: false,
            validateRequestParameters: true,
        });
        api.addRequestValidator('body-validator', {
            requestValidatorName: 'Body',
            validateRequestBody: true,
            validateRequestParameters: false,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::RequestValidator', {
            RestApiId: { Ref: stack.getLogicalId(api.node.findChild('Resource')) },
            Name: 'Parameters',
            ValidateRequestBody: false,
            ValidateRequestParameters: true,
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::RequestValidator', {
            RestApiId: { Ref: stack.getLogicalId(api.node.findChild('Resource')) },
            Name: 'Body',
            ValidateRequestBody: true,
            ValidateRequestParameters: false,
        }));
        test.done();
    },
    'creates output with given "exportName"'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'myapi', { endpointExportName: 'my-given-export-name' });
        api.root.addMethod('GET');
        // THEN
        test.deepEqual(assert_1.SynthUtils.toCloudFormation(stack).Outputs, {
            myapiEndpoint8EB17201: {
                Value: {
                    'Fn::Join': [
                        '',
                        [
                            'https://',
                            { Ref: 'myapi162F20B8' },
                            '.execute-api.',
                            { Ref: 'AWS::Region' },
                            '.',
                            { Ref: 'AWS::URLSuffix' },
                            '/',
                            { Ref: 'myapiDeploymentStageprod329F21FF' },
                            '/',
                        ],
                    ],
                },
                Export: { Name: 'my-given-export-name' },
            },
        });
        test.done();
    },
    'creates output when "exportName" is not specified'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'myapi');
        api.root.addMethod('GET');
        // THEN
        test.deepEqual(assert_1.SynthUtils.toCloudFormation(stack).Outputs, {
            myapiEndpoint8EB17201: {
                Value: {
                    'Fn::Join': [
                        '',
                        [
                            'https://',
                            { Ref: 'myapi162F20B8' },
                            '.execute-api.',
                            { Ref: 'AWS::Region' },
                            '.',
                            { Ref: 'AWS::URLSuffix' },
                            '/',
                            { Ref: 'myapiDeploymentStageprod329F21FF' },
                            '/',
                        ],
                    ],
                },
            },
        });
        test.done();
    },
    'gateway response resource is created'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'restapi', {
            deploy: false,
            cloudWatchRole: false,
        });
        api.root.addMethod('GET');
        api.addGatewayResponse('test-response', {
            type: apigw.ResponseType.ACCESS_DENIED,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::GatewayResponse', {
            ResponseType: 'ACCESS_DENIED',
            RestApiId: stack.resolve(api.restApiId),
            StatusCode: assert_1.ABSENT,
            ResponseParameters: assert_1.ABSENT,
            ResponseTemplates: assert_1.ABSENT,
        }));
        test.done();
    },
    'gateway response resource is created with parameters'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'restapi', {
            deploy: false,
            cloudWatchRole: false,
        });
        api.root.addMethod('GET');
        api.addGatewayResponse('test-response', {
            type: apigw.ResponseType.AUTHORIZER_FAILURE,
            statusCode: '500',
            responseHeaders: {
                'Access-Control-Allow-Origin': 'test.com',
                'test-key': 'test-value',
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::GatewayResponse', {
            ResponseType: 'AUTHORIZER_FAILURE',
            RestApiId: stack.resolve(api.restApiId),
            StatusCode: '500',
            ResponseParameters: {
                'gatewayresponse.header.Access-Control-Allow-Origin': 'test.com',
                'gatewayresponse.header.test-key': 'test-value',
            },
            ResponseTemplates: assert_1.ABSENT,
        }));
        test.done();
    },
    'gateway response resource is created with templates'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'restapi', {
            deploy: false,
            cloudWatchRole: false,
        });
        api.root.addMethod('GET');
        api.addGatewayResponse('test-response', {
            type: apigw.ResponseType.AUTHORIZER_FAILURE,
            statusCode: '500',
            templates: {
                'application/json': '{ "message": $context.error.messageString, "statusCode": "488" }',
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::GatewayResponse', {
            ResponseType: 'AUTHORIZER_FAILURE',
            RestApiId: stack.resolve(api.restApiId),
            StatusCode: '500',
            ResponseParameters: assert_1.ABSENT,
            ResponseTemplates: {
                'application/json': '{ "message": $context.error.messageString, "statusCode": "488" }',
            },
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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