# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['sdvclient']

package_data = \
{'': ['*']}

install_requires = \
['pandas>=1.0.5,<2.0.0', 'pydantic>=1.5.1,<2.0.0', 'requests>=2.24.0,<3.0.0']

setup_kwargs = {
    'name': 'sdvclient',
    'version': '0.4.0',
    'description': 'Sport Data Valley client library for Python',
    'long_description': '# Sport Data Valley client library for Python\n\n[![Downloads](https://pepy.tech/badge/sdvclient)](https://pepy.tech/project/sdvclient)\n\n## Introduction\n`sdvclient` is a Python client library for the Sport Data Valley platform.\nIt is basically a wrapper around the REST API (documented [here](https://app.sportdatavalley.nl/api-docs/index.html)).\n\n## Installation\nIf you are working in the Sport Data Valley JupyterHub environment, this library is automatically installed.\n\nIf you are working in a different Python environment, this library can be installed from [PyPI](https://pypi.org/project/sdvclient/):\n```bash\npip install sdvclient\n```\n\nWhen you have previously installed the library and want to upgrade to a newer version:\n```bash\npip install --upgrade sdvclient\n```\n\n## Usage\n```python\nimport sdvclient as sdv\n\n\nfor dataset in sdv.my_datasets():\n    # Do something\n    pass\n```\nThe dataset summaries that are returned from `my_datasets()` have attributes like `title`, `event_start`, `event_end`, `owner`, `sport`, `tags` and more...\n```python\ndataset.sport\n>>> "sports.riding"\n```\n\nTo retrieve data from your network:\n```python\n\nimport sdvclient as sdv\n\n\nfor dataset in sdv.network_datasets():\n    # Do something\n    pass\n```\n\n### Limit the number of results\nBoth `sdv.my_datasets()` and `sdv.network_datasets()` accept an optional `limit` argument that can be used to limit the number of dataset summaries that are returned.\n```python\n\nimport sdvclient as sdv\n\nfor dataset in sdv.my_datasets(limit=10):\n    # Process maximum 10 datasets\n    pass\n```\nPlease note that if there are less datasets available then the `limit` you specify, the number of returned dataset summaries is lower than `limit`.\n\n\n### Filter network data\n`sdv.network_datasets()` accepts an optional `query` argument that can be used to filter the returned datasets:\n```python\nimport sdvclient as sdv\n\nfor dataset in sdv.network_datasets(query="strava"):\n    # Process datasets that are matched by the "strava" query\n    pass\n```\nPlease note that the query argument filters on *all* the fields of a dataset.\nThis means that filtering on the name of a user does not necessarily only retrieve data for that user, as this name may also occur *anywhere* else in a different dataset.\n\nN.B. The `query` argument is **not** available for `sdv.my_datasets()`.\n\n### Retrieving raw/full data\nAfter you have retrieved a dataset summary, you can then continue to download the raw/full data from this dataset by calling the `get_data()` method on this object:\n```python\ndataset = dataset.get_data(id=1337)\n```\n\nEvery object that is returned from `get_data()` has attributes like `title`, `event_start`, `event_end`, `owner`, `sport`, `type`, `tags` and more fields depending on the data_type. For example a dataset with type `strava_type` has an attribute `dataframe` that contains a [pandas.DataFrame](https://pandas.pydata.org/pandas-docs/stable/reference/api/pandas.DataFrame.html) with the data from this dataset.\n```python\ndataset.data_type\n>>> "strava_type"\ndataset.dataframe\n>>> <pandas.DataFrame>\n```\n\n#### Strava data type\nAs mentioned above, dasets of type `strava_type` have an attribute dataframe with the corresponding data in a dataframe:\n```python\ndataset.data_type\n>>> "strava_type"\ndataset.dataframe\n>>> <pandas.DataFrame>\n```\n\n#### Questionnaire data type\nDatasets of type questionnaire have a `questions` attribute which contains of all the questions and answers in the questionnaire.\nFor each question+answer, the question and answer are available on the `question` and `answer` attributes, respectivily.\n```python\ndataset.questions[2].question\n>>> "this is a question"\n\ndataset.questions[2].answer\n>>> "this is an answer"\n```\n\n#### Generic CSV data type\nFor generic tabular data like csv\'s the returned dataset has an attribute dataframe with the corresponding data in a dataframe:\n```python\ndataset.data_type\n>>> "generic_csv_type"\ndataset.dataframe\n>>> <pandas.DataFrame>\n```\n\n#### Daily activity data type\nFor daily activity data that is coming from e.g. Fitbit or Polar, the returned dataset has a range of attributes:\n\n- steps\n- distance\n- calories\n- floors\n- sleep_start\n- sleep_end\n- sleep_duration\n- resting_heart_rate\n- minutes_sedentary\n- minutes_lightly_active\n- minutes_fairly_active\n- minutes_very_active\n\n```python\ndataset.data_type\n>>> "fitbit_type"\ndataset.resting_heart_rate\n>>> 58\n```\n\nPlease note that not all attributes are always available, this is platform and device dependent.\n\n\n#### Unstructured data\nUnstructured data is data (files) that Sport Data Valley does not know how to process.\nThese files are stored "as is" in the platform and can be download via this client library as well:\nFor generic tabular data like csv\'s the returned dataset has an attribute dataframe with the corresponding data in a dataframe:\nUnstructured data has a `file_response` attribute that contains a [requests.Response](https://requests.readthedocs.io/en/latest/api/#requests.Response) object.\n\n```python\ndataset.data_type\n>>> "unstructured"\ndataset.file_response\n>>> <Response [200]>\n```\n\nRead more about processing files downloaded with the Python requests library [here](https://requests.readthedocs.io/en/master/user/quickstart/).\nE.g. to process binary response content, see [here](https://requests.readthedocs.io/en/master/user/quickstart/#binary-response-content).\n\n\n#### Other data types\nAlthough this library will be updated when new data types are added it can happen that a specific data type is not fully supported yet. In that case the returned dateset will be identical as unstructured data, with an `file_response` attribute that contains a [requests.Response](https://requests.readthedocs.io/en/latest/api/#requests.Response) object.\nUnstructured data is data (files) that Sport Data Valley does not know how to process.\n\n```python\ndataset.data_type\n>>> "some new data type"\ndataset.file_response\n>>> <Response [200]>\n```\n\n\n### Authentication\nThe library retrieves your API token from the `SDV_AUTH_TOKEN` environment variable.\nIf you are working in the Sport Data Valley JupyterHub, this is automatically set.\nIf you are working in a different environment, you can retrieve an API token from the "Advanced" page [here](https://app.sportdatavalley.nl/profile/edit) and set it like this:\n\n```python\nsdv.set_token("your API token here")\n```\n\n\n\n## Development\n\n### Adding Python versions\nThe supported Python versions are specified in `pyproject.toml[tool.poetry.dependencies]#python`.\nThe Python versions that are tested are specified in `pyproject.toml[tool.tox]#envlist` and in `Dockerfile.test`.\nIf you want to add a new supported Python version, or want to test against a newer version of an existing Python version, the versions at these locations need to be updated.\n\n\n## Contributors\n- [Aart Goossens](https://twitter.com/aartgoossens)\n\n## License\nSee [LICENSE](LICENSE) file.\n',
    'author': 'Aart Goossens',
    'author_email': 'aart@sportdatavalley.nl',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://info.sportdatavalley.nl',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6.1,<4.0.0',
}


setup(**setup_kwargs)
