"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Represents an identity source.
 *
 * The source can be specified either as a literal value (e.g: `Auth`) which
 * cannot be blank, or as an unresolved string token.
 */
class IdentitySource {
    /**
     * Provides a properly formatted header identity source.
     * @param headerName the name of the header the `IdentitySource` will represent.
     *
     * @returns a header identity source.
     */
    static header(headerName) {
        return IdentitySource.toString(headerName, 'method.request.header');
    }
    /**
     * Provides a properly formatted query string identity source.
     * @param queryString the name of the query string the `IdentitySource` will represent.
     *
     * @returns a query string identity source.
     */
    static queryString(queryString) {
        return IdentitySource.toString(queryString, 'method.request.querystring');
    }
    /**
     * Provides a properly formatted API Gateway stage variable identity source.
     * @param stageVariable the name of the stage variable the `IdentitySource` will represent.
     *
     * @returns an API Gateway stage variable identity source.
     */
    static stageVariable(stageVariable) {
        return IdentitySource.toString(stageVariable, 'stageVariables');
    }
    /**
     * Provides a properly formatted request context identity source.
     * @param context the name of the context variable the `IdentitySource` will represent.
     *
     * @returns a request context identity source.
     */
    static context(context) {
        return IdentitySource.toString(context, 'context');
    }
    static toString(source, type) {
        if (!source.trim()) {
            throw new Error(`IdentitySources must be a non-empty string.`);
        }
        return `${type}.${source}`;
    }
}
exports.IdentitySource = IdentitySource;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaWRlbnRpdHktc291cmNlLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiaWRlbnRpdHktc291cmNlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUE7Ozs7O0dBS0c7QUFDSCxNQUFhLGNBQWM7SUFDekI7Ozs7O09BS0c7SUFDSSxNQUFNLENBQUMsTUFBTSxDQUFDLFVBQWtCO1FBQ3JDLE9BQU8sY0FBYyxDQUFDLFFBQVEsQ0FBQyxVQUFVLEVBQUUsdUJBQXVCLENBQUMsQ0FBQztJQUN0RSxDQUFDO0lBRUQ7Ozs7O09BS0c7SUFDSSxNQUFNLENBQUMsV0FBVyxDQUFDLFdBQW1CO1FBQzNDLE9BQU8sY0FBYyxDQUFDLFFBQVEsQ0FBQyxXQUFXLEVBQUUsNEJBQTRCLENBQUMsQ0FBQztJQUM1RSxDQUFDO0lBRUQ7Ozs7O09BS0c7SUFDSSxNQUFNLENBQUMsYUFBYSxDQUFDLGFBQXFCO1FBQy9DLE9BQU8sY0FBYyxDQUFDLFFBQVEsQ0FBQyxhQUFhLEVBQUUsZ0JBQWdCLENBQUMsQ0FBQztJQUNsRSxDQUFDO0lBRUQ7Ozs7O09BS0c7SUFDSSxNQUFNLENBQUMsT0FBTyxDQUFDLE9BQWU7UUFDbkMsT0FBTyxjQUFjLENBQUMsUUFBUSxDQUFDLE9BQU8sRUFBRSxTQUFTLENBQUMsQ0FBQztJQUNyRCxDQUFDO0lBRU8sTUFBTSxDQUFDLFFBQVEsQ0FBQyxNQUFjLEVBQUUsSUFBWTtRQUNsRCxJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksRUFBRSxFQUFFO1lBQ2xCLE1BQU0sSUFBSSxLQUFLLENBQUMsNkNBQTZDLENBQUMsQ0FBQztTQUNoRTtRQUVELE9BQU8sR0FBRyxJQUFJLElBQUksTUFBTSxFQUFFLENBQUM7SUFDN0IsQ0FBQztDQUNGO0FBaERELHdDQWdEQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogUmVwcmVzZW50cyBhbiBpZGVudGl0eSBzb3VyY2UuXG4gKlxuICogVGhlIHNvdXJjZSBjYW4gYmUgc3BlY2lmaWVkIGVpdGhlciBhcyBhIGxpdGVyYWwgdmFsdWUgKGUuZzogYEF1dGhgKSB3aGljaFxuICogY2Fubm90IGJlIGJsYW5rLCBvciBhcyBhbiB1bnJlc29sdmVkIHN0cmluZyB0b2tlbi5cbiAqL1xuZXhwb3J0IGNsYXNzIElkZW50aXR5U291cmNlIHtcbiAgLyoqXG4gICAqIFByb3ZpZGVzIGEgcHJvcGVybHkgZm9ybWF0dGVkIGhlYWRlciBpZGVudGl0eSBzb3VyY2UuXG4gICAqIEBwYXJhbSBoZWFkZXJOYW1lIHRoZSBuYW1lIG9mIHRoZSBoZWFkZXIgdGhlIGBJZGVudGl0eVNvdXJjZWAgd2lsbCByZXByZXNlbnQuXG4gICAqXG4gICAqIEByZXR1cm5zIGEgaGVhZGVyIGlkZW50aXR5IHNvdXJjZS5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgaGVhZGVyKGhlYWRlck5hbWU6IHN0cmluZyk6IHN0cmluZyB7XG4gICAgcmV0dXJuIElkZW50aXR5U291cmNlLnRvU3RyaW5nKGhlYWRlck5hbWUsICdtZXRob2QucmVxdWVzdC5oZWFkZXInKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBQcm92aWRlcyBhIHByb3Blcmx5IGZvcm1hdHRlZCBxdWVyeSBzdHJpbmcgaWRlbnRpdHkgc291cmNlLlxuICAgKiBAcGFyYW0gcXVlcnlTdHJpbmcgdGhlIG5hbWUgb2YgdGhlIHF1ZXJ5IHN0cmluZyB0aGUgYElkZW50aXR5U291cmNlYCB3aWxsIHJlcHJlc2VudC5cbiAgICpcbiAgICogQHJldHVybnMgYSBxdWVyeSBzdHJpbmcgaWRlbnRpdHkgc291cmNlLlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBxdWVyeVN0cmluZyhxdWVyeVN0cmluZzogc3RyaW5nKTogc3RyaW5nIHtcbiAgICByZXR1cm4gSWRlbnRpdHlTb3VyY2UudG9TdHJpbmcocXVlcnlTdHJpbmcsICdtZXRob2QucmVxdWVzdC5xdWVyeXN0cmluZycpO1xuICB9XG5cbiAgLyoqXG4gICAqIFByb3ZpZGVzIGEgcHJvcGVybHkgZm9ybWF0dGVkIEFQSSBHYXRld2F5IHN0YWdlIHZhcmlhYmxlIGlkZW50aXR5IHNvdXJjZS5cbiAgICogQHBhcmFtIHN0YWdlVmFyaWFibGUgdGhlIG5hbWUgb2YgdGhlIHN0YWdlIHZhcmlhYmxlIHRoZSBgSWRlbnRpdHlTb3VyY2VgIHdpbGwgcmVwcmVzZW50LlxuICAgKlxuICAgKiBAcmV0dXJucyBhbiBBUEkgR2F0ZXdheSBzdGFnZSB2YXJpYWJsZSBpZGVudGl0eSBzb3VyY2UuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHN0YWdlVmFyaWFibGUoc3RhZ2VWYXJpYWJsZTogc3RyaW5nKTogc3RyaW5nIHtcbiAgICByZXR1cm4gSWRlbnRpdHlTb3VyY2UudG9TdHJpbmcoc3RhZ2VWYXJpYWJsZSwgJ3N0YWdlVmFyaWFibGVzJyk7XG4gIH1cblxuICAvKipcbiAgICogUHJvdmlkZXMgYSBwcm9wZXJseSBmb3JtYXR0ZWQgcmVxdWVzdCBjb250ZXh0IGlkZW50aXR5IHNvdXJjZS5cbiAgICogQHBhcmFtIGNvbnRleHQgdGhlIG5hbWUgb2YgdGhlIGNvbnRleHQgdmFyaWFibGUgdGhlIGBJZGVudGl0eVNvdXJjZWAgd2lsbCByZXByZXNlbnQuXG4gICAqXG4gICAqIEByZXR1cm5zIGEgcmVxdWVzdCBjb250ZXh0IGlkZW50aXR5IHNvdXJjZS5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgY29udGV4dChjb250ZXh0OiBzdHJpbmcpOiBzdHJpbmcge1xuICAgIHJldHVybiBJZGVudGl0eVNvdXJjZS50b1N0cmluZyhjb250ZXh0LCAnY29udGV4dCcpO1xuICB9XG5cbiAgcHJpdmF0ZSBzdGF0aWMgdG9TdHJpbmcoc291cmNlOiBzdHJpbmcsIHR5cGU6IHN0cmluZykge1xuICAgIGlmICghc291cmNlLnRyaW0oKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBJZGVudGl0eVNvdXJjZXMgbXVzdCBiZSBhIG5vbi1lbXB0eSBzdHJpbmcuYCk7XG4gICAgfVxuXG4gICAgcmV0dXJuIGAke3R5cGV9LiR7c291cmNlfWA7XG4gIH1cbn1cbiJdfQ==