"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("../apigateway.generated");
const authorizer_1 = require("../authorizer");
class LambdaAuthorizer extends authorizer_1.Authorizer {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.handler = props.handler;
        this.role = props.assumeRole;
        if (props.resultsCacheTtl && ((_a = props.resultsCacheTtl) === null || _a === void 0 ? void 0 : _a.toSeconds()) > 3600) {
            throw new Error(`Lambda authorizer property 'resultsCacheTtl' must not be greater than 3600 seconds (1 hour)`);
        }
    }
    /**
     * Attaches this authorizer to a specific REST API.
     * @internal
     */
    _attachToApi(restApi) {
        if (this.restApiId && this.restApiId !== restApi.restApiId) {
            throw new Error(`Cannot attach authorizer to two different rest APIs`);
        }
        this.restApiId = restApi.restApiId;
    }
    /**
     * Sets up the permissions necessary for the API Gateway service to invoke the Lambda function.
     */
    setupPermissions() {
        if (!this.role) {
            this.handler.addPermission(`${this.node.uniqueId}:Permissions`, {
                principal: new iam.ServicePrincipal('apigateway.amazonaws.com'),
                sourceArn: this.authorizerArn
            });
        }
        else if (this.role instanceof iam.Role) { // i.e. not imported
            this.role.attachInlinePolicy(new iam.Policy(this, 'authorizerInvokePolicy', {
                statements: [
                    new iam.PolicyStatement({
                        resources: [this.handler.functionArn],
                        actions: ['lambda:InvokeFunction'],
                    })
                ]
            }));
        }
    }
}
/**
 * Token based lambda authorizer that recognizes the caller's identity as a bearer token,
 * such as a JSON Web Token (JWT) or an OAuth token.
 * Based on the token, authorization is performed by a lambda function.
 *
 * @resource AWS::ApiGateway::Authorizer
 */
class TokenAuthorizer extends LambdaAuthorizer {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id, props);
        const restApiId = core_1.Lazy.stringValue({ produce: () => this.restApiId });
        const resource = new apigateway_generated_1.CfnAuthorizer(this, 'Resource', {
            name: (_a = props.authorizerName, (_a !== null && _a !== void 0 ? _a : this.node.uniqueId)),
            restApiId,
            type: 'TOKEN',
            authorizerUri: `arn:aws:apigateway:${core_1.Stack.of(this).region}:lambda:path/2015-03-31/functions/${props.handler.functionArn}/invocations`,
            authorizerCredentials: (_b = props.assumeRole) === null || _b === void 0 ? void 0 : _b.roleArn,
            authorizerResultTtlInSeconds: (_c = props.resultsCacheTtl) === null || _c === void 0 ? void 0 : _c.toSeconds(),
            identitySource: props.identitySource || 'method.request.header.Authorization',
            identityValidationExpression: props.validationRegex,
        });
        this.authorizerId = resource.ref;
        this.authorizerArn = core_1.Stack.of(this).formatArn({
            service: 'execute-api',
            resource: restApiId,
            resourceName: `authorizers/${this.authorizerId}`
        });
        this.setupPermissions();
    }
}
exports.TokenAuthorizer = TokenAuthorizer;
/**
 * Request-based lambda authorizer that recognizes the caller's identity via request parameters,
 * such as headers, paths, query strings, stage variables, or context variables.
 * Based on the request, authorization is performed by a lambda function.
 *
 * @resource AWS::ApiGateway::Authorizer
 */
class RequestAuthorizer extends LambdaAuthorizer {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id, props);
        if ((props.resultsCacheTtl === undefined || props.resultsCacheTtl.toSeconds() !== 0) && props.identitySources.length === 0) {
            throw new Error(`At least one Identity Source is required for a REQUEST-based Lambda authorizer if caching is enabled.`);
        }
        const restApiId = core_1.Lazy.stringValue({ produce: () => this.restApiId });
        const resource = new apigateway_generated_1.CfnAuthorizer(this, 'Resource', {
            name: (_a = props.authorizerName, (_a !== null && _a !== void 0 ? _a : this.node.uniqueId)),
            restApiId,
            type: 'REQUEST',
            authorizerUri: `arn:aws:apigateway:${core_1.Stack.of(this).region}:lambda:path/2015-03-31/functions/${props.handler.functionArn}/invocations`,
            authorizerCredentials: (_b = props.assumeRole) === null || _b === void 0 ? void 0 : _b.roleArn,
            authorizerResultTtlInSeconds: (_c = props.resultsCacheTtl) === null || _c === void 0 ? void 0 : _c.toSeconds(),
            identitySource: props.identitySources.map(is => is.toString()).join(','),
        });
        this.authorizerId = resource.ref;
        this.authorizerArn = core_1.Stack.of(this).formatArn({
            service: 'execute-api',
            resource: restApiId,
            resourceName: `authorizers/${this.authorizerId}`
        });
        this.setupPermissions();
    }
}
exports.RequestAuthorizer = RequestAuthorizer;
//# sourceMappingURL=data:application/json;base64,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